/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to update the user pool client.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateUserPoolClientRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<UpdateUserPoolClientRequest.Builder, UpdateUserPoolClientRequest> {
    private final String userPoolId;

    private final String clientId;

    private final String clientName;

    private final Integer refreshTokenValidity;

    private final List<String> readAttributes;

    private final List<String> writeAttributes;

    private final List<String> explicitAuthFlows;

    private final List<String> supportedIdentityProviders;

    private final List<String> callbackURLs;

    private final List<String> logoutURLs;

    private final String defaultRedirectURI;

    private final List<String> allowedOAuthFlows;

    private final List<String> allowedOAuthScopes;

    private final Boolean allowedOAuthFlowsUserPoolClient;

    private UpdateUserPoolClientRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.clientId = builder.clientId;
        this.clientName = builder.clientName;
        this.refreshTokenValidity = builder.refreshTokenValidity;
        this.readAttributes = builder.readAttributes;
        this.writeAttributes = builder.writeAttributes;
        this.explicitAuthFlows = builder.explicitAuthFlows;
        this.supportedIdentityProviders = builder.supportedIdentityProviders;
        this.callbackURLs = builder.callbackURLs;
        this.logoutURLs = builder.logoutURLs;
        this.defaultRedirectURI = builder.defaultRedirectURI;
        this.allowedOAuthFlows = builder.allowedOAuthFlows;
        this.allowedOAuthScopes = builder.allowedOAuthScopes;
        this.allowedOAuthFlowsUserPoolClient = builder.allowedOAuthFlowsUserPoolClient;
    }

    /**
     * <p>
     * The user pool ID for the user pool where you want to update the user pool client.
     * </p>
     * 
     * @return The user pool ID for the user pool where you want to update the user pool client.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The ID of the client associated with the user pool.
     * </p>
     * 
     * @return The ID of the client associated with the user pool.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The client name from the update user pool client request.
     * </p>
     * 
     * @return The client name from the update user pool client request.
     */
    public String clientName() {
        return clientName;
    }

    /**
     * <p>
     * The time limit, in days, after which the refresh token is no longer valid and cannot be used.
     * </p>
     * 
     * @return The time limit, in days, after which the refresh token is no longer valid and cannot be used.
     */
    public Integer refreshTokenValidity() {
        return refreshTokenValidity;
    }

    /**
     * <p>
     * The read-only attributes of the user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The read-only attributes of the user pool.
     */
    public List<String> readAttributes() {
        return readAttributes;
    }

    /**
     * <p>
     * The writeable attributes of the user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The writeable attributes of the user pool.
     */
    public List<String> writeAttributes() {
        return writeAttributes;
    }

    /**
     * <p>
     * Explicit authentication flows.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Explicit authentication flows.
     */
    public List<ExplicitAuthFlowsType> explicitAuthFlows() {
        return TypeConverter.convert(explicitAuthFlows, ExplicitAuthFlowsType::fromValue);
    }

    /**
     * <p>
     * Explicit authentication flows.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Explicit authentication flows.
     */
    public List<String> explicitAuthFlowsAsStrings() {
        return explicitAuthFlows;
    }

    /**
     * <p>
     * A list of provider names for the identity providers that are supported on this client.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of provider names for the identity providers that are supported on this client.
     */
    public List<String> supportedIdentityProviders() {
        return supportedIdentityProviders;
    }

    /**
     * <p>
     * A list of allowed callback URLs for the identity providers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of allowed callback URLs for the identity providers.
     */
    public List<String> callbackURLs() {
        return callbackURLs;
    }

    /**
     * <p>
     * A list of allowed logout URLs for the identity providers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of allowed logout URLs for the identity providers.
     */
    public List<String> logoutURLs() {
        return logoutURLs;
    }

    /**
     * <p>
     * The default redirect URI. Must be in the <code>CallbackURLs</code> list.
     * </p>
     * 
     * @return The default redirect URI. Must be in the <code>CallbackURLs</code> list.
     */
    public String defaultRedirectURI() {
        return defaultRedirectURI;
    }

    /**
     * <p>
     * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
     * This code can be exchanged for access tokens with the token endpoint.
     * </p>
     * <p>
     * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
     * based on scopes) directly.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
     *         response. This code can be exchanged for access tokens with the token endpoint.</p>
     *         <p>
     *         Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
     *         token, based on scopes) directly.
     */
    public List<OAuthFlowType> allowedOAuthFlows() {
        return TypeConverter.convert(allowedOAuthFlows, OAuthFlowType::fromValue);
    }

    /**
     * <p>
     * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
     * This code can be exchanged for access tokens with the token endpoint.
     * </p>
     * <p>
     * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
     * based on scopes) directly.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
     *         response. This code can be exchanged for access tokens with the token endpoint.</p>
     *         <p>
     *         Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
     *         token, based on scopes) directly.
     */
    public List<String> allowedOAuthFlowsAsStrings() {
        return allowedOAuthFlows;
    }

    /**
     * <p>
     * A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
     * <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
     *         <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
     */
    public List<String> allowedOAuthScopes() {
        return allowedOAuthScopes;
    }

    /**
     * <p>
     * Set to TRUE if the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
     * </p>
     * 
     * @return Set to TRUE if the client is allowed to follow the OAuth protocol when interacting with Cognito user
     *         pools.
     */
    public Boolean allowedOAuthFlowsUserPoolClient() {
        return allowedOAuthFlowsUserPoolClient;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(clientName());
        hashCode = 31 * hashCode + Objects.hashCode(refreshTokenValidity());
        hashCode = 31 * hashCode + Objects.hashCode(readAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(writeAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(explicitAuthFlowsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(supportedIdentityProviders());
        hashCode = 31 * hashCode + Objects.hashCode(callbackURLs());
        hashCode = 31 * hashCode + Objects.hashCode(logoutURLs());
        hashCode = 31 * hashCode + Objects.hashCode(defaultRedirectURI());
        hashCode = 31 * hashCode + Objects.hashCode(allowedOAuthFlowsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(allowedOAuthScopes());
        hashCode = 31 * hashCode + Objects.hashCode(allowedOAuthFlowsUserPoolClient());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateUserPoolClientRequest)) {
            return false;
        }
        UpdateUserPoolClientRequest other = (UpdateUserPoolClientRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(clientName(), other.clientName())
                && Objects.equals(refreshTokenValidity(), other.refreshTokenValidity())
                && Objects.equals(readAttributes(), other.readAttributes())
                && Objects.equals(writeAttributes(), other.writeAttributes())
                && Objects.equals(explicitAuthFlowsAsStrings(), other.explicitAuthFlowsAsStrings())
                && Objects.equals(supportedIdentityProviders(), other.supportedIdentityProviders())
                && Objects.equals(callbackURLs(), other.callbackURLs()) && Objects.equals(logoutURLs(), other.logoutURLs())
                && Objects.equals(defaultRedirectURI(), other.defaultRedirectURI())
                && Objects.equals(allowedOAuthFlowsAsStrings(), other.allowedOAuthFlowsAsStrings())
                && Objects.equals(allowedOAuthScopes(), other.allowedOAuthScopes())
                && Objects.equals(allowedOAuthFlowsUserPoolClient(), other.allowedOAuthFlowsUserPoolClient());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateUserPoolClientRequest").add("UserPoolId", userPoolId()).add("ClientId", clientId())
                .add("ClientName", clientName()).add("RefreshTokenValidity", refreshTokenValidity())
                .add("ReadAttributes", readAttributes()).add("WriteAttributes", writeAttributes())
                .add("ExplicitAuthFlows", explicitAuthFlowsAsStrings())
                .add("SupportedIdentityProviders", supportedIdentityProviders()).add("CallbackURLs", callbackURLs())
                .add("LogoutURLs", logoutURLs()).add("DefaultRedirectURI", defaultRedirectURI())
                .add("AllowedOAuthFlows", allowedOAuthFlowsAsStrings()).add("AllowedOAuthScopes", allowedOAuthScopes())
                .add("AllowedOAuthFlowsUserPoolClient", allowedOAuthFlowsUserPoolClient()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "ClientName":
            return Optional.ofNullable(clazz.cast(clientName()));
        case "RefreshTokenValidity":
            return Optional.ofNullable(clazz.cast(refreshTokenValidity()));
        case "ReadAttributes":
            return Optional.ofNullable(clazz.cast(readAttributes()));
        case "WriteAttributes":
            return Optional.ofNullable(clazz.cast(writeAttributes()));
        case "ExplicitAuthFlows":
            return Optional.ofNullable(clazz.cast(explicitAuthFlowsAsStrings()));
        case "SupportedIdentityProviders":
            return Optional.ofNullable(clazz.cast(supportedIdentityProviders()));
        case "CallbackURLs":
            return Optional.ofNullable(clazz.cast(callbackURLs()));
        case "LogoutURLs":
            return Optional.ofNullable(clazz.cast(logoutURLs()));
        case "DefaultRedirectURI":
            return Optional.ofNullable(clazz.cast(defaultRedirectURI()));
        case "AllowedOAuthFlows":
            return Optional.ofNullable(clazz.cast(allowedOAuthFlowsAsStrings()));
        case "AllowedOAuthScopes":
            return Optional.ofNullable(clazz.cast(allowedOAuthScopes()));
        case "AllowedOAuthFlowsUserPoolClient":
            return Optional.ofNullable(clazz.cast(allowedOAuthFlowsUserPoolClient()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder,
            CopyableBuilder<Builder, UpdateUserPoolClientRequest> {
        /**
         * <p>
         * The user pool ID for the user pool where you want to update the user pool client.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for the user pool where you want to update the user pool client.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The ID of the client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The client name from the update user pool client request.
         * </p>
         * 
         * @param clientName
         *        The client name from the update user pool client request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientName(String clientName);

        /**
         * <p>
         * The time limit, in days, after which the refresh token is no longer valid and cannot be used.
         * </p>
         * 
         * @param refreshTokenValidity
         *        The time limit, in days, after which the refresh token is no longer valid and cannot be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder refreshTokenValidity(Integer refreshTokenValidity);

        /**
         * <p>
         * The read-only attributes of the user pool.
         * </p>
         * 
         * @param readAttributes
         *        The read-only attributes of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readAttributes(Collection<String> readAttributes);

        /**
         * <p>
         * The read-only attributes of the user pool.
         * </p>
         * 
         * @param readAttributes
         *        The read-only attributes of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readAttributes(String... readAttributes);

        /**
         * <p>
         * The writeable attributes of the user pool.
         * </p>
         * 
         * @param writeAttributes
         *        The writeable attributes of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder writeAttributes(Collection<String> writeAttributes);

        /**
         * <p>
         * The writeable attributes of the user pool.
         * </p>
         * 
         * @param writeAttributes
         *        The writeable attributes of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder writeAttributes(String... writeAttributes);

        /**
         * <p>
         * Explicit authentication flows.
         * </p>
         * 
         * @param explicitAuthFlows
         *        Explicit authentication flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitAuthFlowsWithStrings(Collection<String> explicitAuthFlows);

        /**
         * <p>
         * Explicit authentication flows.
         * </p>
         * 
         * @param explicitAuthFlows
         *        Explicit authentication flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitAuthFlowsWithStrings(String... explicitAuthFlows);

        /**
         * <p>
         * Explicit authentication flows.
         * </p>
         * 
         * @param explicitAuthFlows
         *        Explicit authentication flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitAuthFlows(Collection<ExplicitAuthFlowsType> explicitAuthFlows);

        /**
         * <p>
         * Explicit authentication flows.
         * </p>
         * 
         * @param explicitAuthFlows
         *        Explicit authentication flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitAuthFlows(ExplicitAuthFlowsType... explicitAuthFlows);

        /**
         * <p>
         * A list of provider names for the identity providers that are supported on this client.
         * </p>
         * 
         * @param supportedIdentityProviders
         *        A list of provider names for the identity providers that are supported on this client.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedIdentityProviders(Collection<String> supportedIdentityProviders);

        /**
         * <p>
         * A list of provider names for the identity providers that are supported on this client.
         * </p>
         * 
         * @param supportedIdentityProviders
         *        A list of provider names for the identity providers that are supported on this client.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedIdentityProviders(String... supportedIdentityProviders);

        /**
         * <p>
         * A list of allowed callback URLs for the identity providers.
         * </p>
         * 
         * @param callbackURLs
         *        A list of allowed callback URLs for the identity providers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder callbackURLs(Collection<String> callbackURLs);

        /**
         * <p>
         * A list of allowed callback URLs for the identity providers.
         * </p>
         * 
         * @param callbackURLs
         *        A list of allowed callback URLs for the identity providers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder callbackURLs(String... callbackURLs);

        /**
         * <p>
         * A list of allowed logout URLs for the identity providers.
         * </p>
         * 
         * @param logoutURLs
         *        A list of allowed logout URLs for the identity providers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logoutURLs(Collection<String> logoutURLs);

        /**
         * <p>
         * A list of allowed logout URLs for the identity providers.
         * </p>
         * 
         * @param logoutURLs
         *        A list of allowed logout URLs for the identity providers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logoutURLs(String... logoutURLs);

        /**
         * <p>
         * The default redirect URI. Must be in the <code>CallbackURLs</code> list.
         * </p>
         * 
         * @param defaultRedirectURI
         *        The default redirect URI. Must be in the <code>CallbackURLs</code> list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultRedirectURI(String defaultRedirectURI);

        /**
         * <p>
         * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
         * This code can be exchanged for access tokens with the token endpoint.
         * </p>
         * <p>
         * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
         * based on scopes) directly.
         * </p>
         * 
         * @param allowedOAuthFlows
         *        Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
         *        response. This code can be exchanged for access tokens with the token endpoint.</p>
         *        <p>
         *        Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
         *        token, based on scopes) directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthFlowsWithStrings(Collection<String> allowedOAuthFlows);

        /**
         * <p>
         * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
         * This code can be exchanged for access tokens with the token endpoint.
         * </p>
         * <p>
         * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
         * based on scopes) directly.
         * </p>
         * 
         * @param allowedOAuthFlows
         *        Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
         *        response. This code can be exchanged for access tokens with the token endpoint.</p>
         *        <p>
         *        Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
         *        token, based on scopes) directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthFlowsWithStrings(String... allowedOAuthFlows);

        /**
         * <p>
         * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
         * This code can be exchanged for access tokens with the token endpoint.
         * </p>
         * <p>
         * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
         * based on scopes) directly.
         * </p>
         * 
         * @param allowedOAuthFlows
         *        Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
         *        response. This code can be exchanged for access tokens with the token endpoint.</p>
         *        <p>
         *        Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
         *        token, based on scopes) directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthFlows(Collection<OAuthFlowType> allowedOAuthFlows);

        /**
         * <p>
         * Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the response.
         * This code can be exchanged for access tokens with the token endpoint.
         * </p>
         * <p>
         * Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID token,
         * based on scopes) directly.
         * </p>
         * 
         * @param allowedOAuthFlows
         *        Set to <code>code</code> to initiate a code grant flow, which provides an authorization code as the
         *        response. This code can be exchanged for access tokens with the token endpoint.</p>
         *        <p>
         *        Set to <code>token</code> to specify that the client should get the access token (and, optionally, ID
         *        token, based on scopes) directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthFlows(OAuthFlowType... allowedOAuthFlows);

        /**
         * <p>
         * A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
         * <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
         * </p>
         * 
         * @param allowedOAuthScopes
         *        A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
         *        <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthScopes(Collection<String> allowedOAuthScopes);

        /**
         * <p>
         * A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
         * <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
         * </p>
         * 
         * @param allowedOAuthScopes
         *        A list of allowed <code>OAuth</code> scopes. Currently supported values are <code>"phone"</code>,
         *        <code>"email"</code>, <code>"openid"</code>, and <code>"Cognito"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthScopes(String... allowedOAuthScopes);

        /**
         * <p>
         * Set to TRUE if the client is allowed to follow the OAuth protocol when interacting with Cognito user pools.
         * </p>
         * 
         * @param allowedOAuthFlowsUserPoolClient
         *        Set to TRUE if the client is allowed to follow the OAuth protocol when interacting with Cognito user
         *        pools.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedOAuthFlowsUserPoolClient(Boolean allowedOAuthFlowsUserPoolClient);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String clientId;

        private String clientName;

        private Integer refreshTokenValidity;

        private List<String> readAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> writeAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> explicitAuthFlows = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedIdentityProviders = DefaultSdkAutoConstructList.getInstance();

        private List<String> callbackURLs = DefaultSdkAutoConstructList.getInstance();

        private List<String> logoutURLs = DefaultSdkAutoConstructList.getInstance();

        private String defaultRedirectURI;

        private List<String> allowedOAuthFlows = DefaultSdkAutoConstructList.getInstance();

        private List<String> allowedOAuthScopes = DefaultSdkAutoConstructList.getInstance();

        private Boolean allowedOAuthFlowsUserPoolClient;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateUserPoolClientRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            clientId(model.clientId);
            clientName(model.clientName);
            refreshTokenValidity(model.refreshTokenValidity);
            readAttributes(model.readAttributes);
            writeAttributes(model.writeAttributes);
            explicitAuthFlowsWithStrings(model.explicitAuthFlows);
            supportedIdentityProviders(model.supportedIdentityProviders);
            callbackURLs(model.callbackURLs);
            logoutURLs(model.logoutURLs);
            defaultRedirectURI(model.defaultRedirectURI);
            allowedOAuthFlowsWithStrings(model.allowedOAuthFlows);
            allowedOAuthScopes(model.allowedOAuthScopes);
            allowedOAuthFlowsUserPoolClient(model.allowedOAuthFlowsUserPoolClient);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getClientName() {
            return clientName;
        }

        @Override
        public final Builder clientName(String clientName) {
            this.clientName = clientName;
            return this;
        }

        public final void setClientName(String clientName) {
            this.clientName = clientName;
        }

        public final Integer getRefreshTokenValidity() {
            return refreshTokenValidity;
        }

        @Override
        public final Builder refreshTokenValidity(Integer refreshTokenValidity) {
            this.refreshTokenValidity = refreshTokenValidity;
            return this;
        }

        public final void setRefreshTokenValidity(Integer refreshTokenValidity) {
            this.refreshTokenValidity = refreshTokenValidity;
        }

        public final Collection<String> getReadAttributes() {
            return readAttributes;
        }

        @Override
        public final Builder readAttributes(Collection<String> readAttributes) {
            this.readAttributes = ClientPermissionListTypeCopier.copy(readAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder readAttributes(String... readAttributes) {
            readAttributes(Arrays.asList(readAttributes));
            return this;
        }

        public final void setReadAttributes(Collection<String> readAttributes) {
            this.readAttributes = ClientPermissionListTypeCopier.copy(readAttributes);
        }

        public final Collection<String> getWriteAttributes() {
            return writeAttributes;
        }

        @Override
        public final Builder writeAttributes(Collection<String> writeAttributes) {
            this.writeAttributes = ClientPermissionListTypeCopier.copy(writeAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder writeAttributes(String... writeAttributes) {
            writeAttributes(Arrays.asList(writeAttributes));
            return this;
        }

        public final void setWriteAttributes(Collection<String> writeAttributes) {
            this.writeAttributes = ClientPermissionListTypeCopier.copy(writeAttributes);
        }

        public final Collection<String> getExplicitAuthFlows() {
            return explicitAuthFlows;
        }

        @Override
        public final Builder explicitAuthFlowsWithStrings(Collection<String> explicitAuthFlows) {
            this.explicitAuthFlows = ExplicitAuthFlowsListTypeCopier.copy(explicitAuthFlows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder explicitAuthFlowsWithStrings(String... explicitAuthFlows) {
            explicitAuthFlowsWithStrings(Arrays.asList(explicitAuthFlows));
            return this;
        }

        @Override
        public final Builder explicitAuthFlows(Collection<ExplicitAuthFlowsType> explicitAuthFlows) {
            this.explicitAuthFlows = ExplicitAuthFlowsListTypeCopier.copyEnumToString(explicitAuthFlows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder explicitAuthFlows(ExplicitAuthFlowsType... explicitAuthFlows) {
            explicitAuthFlows(Arrays.asList(explicitAuthFlows));
            return this;
        }

        public final void setExplicitAuthFlows(Collection<String> explicitAuthFlows) {
            this.explicitAuthFlows = ExplicitAuthFlowsListTypeCopier.copy(explicitAuthFlows);
        }

        public final Collection<String> getSupportedIdentityProviders() {
            return supportedIdentityProviders;
        }

        @Override
        public final Builder supportedIdentityProviders(Collection<String> supportedIdentityProviders) {
            this.supportedIdentityProviders = SupportedIdentityProvidersListTypeCopier.copy(supportedIdentityProviders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedIdentityProviders(String... supportedIdentityProviders) {
            supportedIdentityProviders(Arrays.asList(supportedIdentityProviders));
            return this;
        }

        public final void setSupportedIdentityProviders(Collection<String> supportedIdentityProviders) {
            this.supportedIdentityProviders = SupportedIdentityProvidersListTypeCopier.copy(supportedIdentityProviders);
        }

        public final Collection<String> getCallbackURLs() {
            return callbackURLs;
        }

        @Override
        public final Builder callbackURLs(Collection<String> callbackURLs) {
            this.callbackURLs = CallbackURLsListTypeCopier.copy(callbackURLs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder callbackURLs(String... callbackURLs) {
            callbackURLs(Arrays.asList(callbackURLs));
            return this;
        }

        public final void setCallbackURLs(Collection<String> callbackURLs) {
            this.callbackURLs = CallbackURLsListTypeCopier.copy(callbackURLs);
        }

        public final Collection<String> getLogoutURLs() {
            return logoutURLs;
        }

        @Override
        public final Builder logoutURLs(Collection<String> logoutURLs) {
            this.logoutURLs = LogoutURLsListTypeCopier.copy(logoutURLs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logoutURLs(String... logoutURLs) {
            logoutURLs(Arrays.asList(logoutURLs));
            return this;
        }

        public final void setLogoutURLs(Collection<String> logoutURLs) {
            this.logoutURLs = LogoutURLsListTypeCopier.copy(logoutURLs);
        }

        public final String getDefaultRedirectURI() {
            return defaultRedirectURI;
        }

        @Override
        public final Builder defaultRedirectURI(String defaultRedirectURI) {
            this.defaultRedirectURI = defaultRedirectURI;
            return this;
        }

        public final void setDefaultRedirectURI(String defaultRedirectURI) {
            this.defaultRedirectURI = defaultRedirectURI;
        }

        public final Collection<String> getAllowedOAuthFlows() {
            return allowedOAuthFlows;
        }

        @Override
        public final Builder allowedOAuthFlowsWithStrings(Collection<String> allowedOAuthFlows) {
            this.allowedOAuthFlows = OAuthFlowsTypeCopier.copy(allowedOAuthFlows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedOAuthFlowsWithStrings(String... allowedOAuthFlows) {
            allowedOAuthFlowsWithStrings(Arrays.asList(allowedOAuthFlows));
            return this;
        }

        @Override
        public final Builder allowedOAuthFlows(Collection<OAuthFlowType> allowedOAuthFlows) {
            this.allowedOAuthFlows = OAuthFlowsTypeCopier.copyEnumToString(allowedOAuthFlows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedOAuthFlows(OAuthFlowType... allowedOAuthFlows) {
            allowedOAuthFlows(Arrays.asList(allowedOAuthFlows));
            return this;
        }

        public final void setAllowedOAuthFlows(Collection<String> allowedOAuthFlows) {
            this.allowedOAuthFlows = OAuthFlowsTypeCopier.copy(allowedOAuthFlows);
        }

        public final Collection<String> getAllowedOAuthScopes() {
            return allowedOAuthScopes;
        }

        @Override
        public final Builder allowedOAuthScopes(Collection<String> allowedOAuthScopes) {
            this.allowedOAuthScopes = ScopeListTypeCopier.copy(allowedOAuthScopes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedOAuthScopes(String... allowedOAuthScopes) {
            allowedOAuthScopes(Arrays.asList(allowedOAuthScopes));
            return this;
        }

        public final void setAllowedOAuthScopes(Collection<String> allowedOAuthScopes) {
            this.allowedOAuthScopes = ScopeListTypeCopier.copy(allowedOAuthScopes);
        }

        public final Boolean getAllowedOAuthFlowsUserPoolClient() {
            return allowedOAuthFlowsUserPoolClient;
        }

        @Override
        public final Builder allowedOAuthFlowsUserPoolClient(Boolean allowedOAuthFlowsUserPoolClient) {
            this.allowedOAuthFlowsUserPoolClient = allowedOAuthFlowsUserPoolClient;
            return this;
        }

        public final void setAllowedOAuthFlowsUserPoolClient(Boolean allowedOAuthFlowsUserPoolClient) {
            this.allowedOAuthFlowsUserPoolClient = allowedOAuthFlowsUserPoolClient;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateUserPoolClientRequest build() {
            return new UpdateUserPoolClientRequest(this);
        }
    }
}
