/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetTokensFromRefreshTokenRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<GetTokensFromRefreshTokenRequest.Builder, GetTokensFromRefreshTokenRequest> {
    private static final SdkField<String> REFRESH_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RefreshToken").getter(getter(GetTokensFromRefreshTokenRequest::refreshToken))
            .setter(setter(Builder::refreshToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RefreshToken").build()).build();

    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(GetTokensFromRefreshTokenRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> CLIENT_SECRET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientSecret").getter(getter(GetTokensFromRefreshTokenRequest::clientSecret))
            .setter(setter(Builder::clientSecret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientSecret").build()).build();

    private static final SdkField<String> DEVICE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceKey").getter(getter(GetTokensFromRefreshTokenRequest::deviceKey))
            .setter(setter(Builder::deviceKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceKey").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(GetTokensFromRefreshTokenRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REFRESH_TOKEN_FIELD,
            CLIENT_ID_FIELD, CLIENT_SECRET_FIELD, DEVICE_KEY_FIELD, CLIENT_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String refreshToken;

    private final String clientId;

    private final String clientSecret;

    private final String deviceKey;

    private final Map<String, String> clientMetadata;

    private GetTokensFromRefreshTokenRequest(BuilderImpl builder) {
        super(builder);
        this.refreshToken = builder.refreshToken;
        this.clientId = builder.clientId;
        this.clientSecret = builder.clientSecret;
        this.deviceKey = builder.deviceKey;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * A valid refresh token that can authorize the request for new tokens. When refresh token rotation is active in the
     * requested app client, this token is invalidated after the request is complete and after an optional grace period.
     * </p>
     * 
     * @return A valid refresh token that can authorize the request for new tokens. When refresh token rotation is
     *         active in the requested app client, this token is invalidated after the request is complete and after an
     *         optional grace period.
     */
    public final String refreshToken() {
        return refreshToken;
    }

    /**
     * <p>
     * The app client that issued the refresh token to the user who wants to request new tokens.
     * </p>
     * 
     * @return The app client that issued the refresh token to the user who wants to request new tokens.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The client secret of the requested app client, if the client has a secret.
     * </p>
     * 
     * @return The client secret of the requested app client, if the client has a secret.
     */
    public final String clientSecret() {
        return clientSecret;
    }

    /**
     * <p>
     * When you enable device remembering, Amazon Cognito issues a device key that you can use for device authentication
     * that bypasses multi-factor authentication (MFA). To implement <code>GetTokensFromRefreshToken</code> in a user
     * pool with device remembering, you must capture the device key from the initial authentication request. If your
     * application doesn't provide the key of a registered device, Amazon Cognito issues a new one. You must provide the
     * confirmed device key in this request if device remembering is enabled in your user pool.
     * </p>
     * <p>
     * For more information about device remembering, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
     * >Working with devices</a>.
     * </p>
     * 
     * @return When you enable device remembering, Amazon Cognito issues a device key that you can use for device
     *         authentication that bypasses multi-factor authentication (MFA). To implement
     *         <code>GetTokensFromRefreshToken</code> in a user pool with device remembering, you must capture the
     *         device key from the initial authentication request. If your application doesn't provide the key of a
     *         registered device, Amazon Cognito issues a new one. You must provide the confirmed device key in this
     *         request if device remembering is enabled in your user pool.</p>
     *         <p>
     *         For more information about device remembering, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
     *         >Working with devices</a>.
     */
    public final String deviceKey() {
        return deviceKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for certain custom workflows that this action
     * triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     * <code>GetTokensFromRefreshToken</code> API action, Amazon Cognito invokes the Lambda function the pre token
     * generation trigger.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned to
     * a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     * <code>ClientMetadata</code> parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the <code>ClientMetadata</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for certain custom workflows that this
     *         action triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         <code>GetTokensFromRefreshToken</code> API action, Amazon Cognito invokes the Lambda function the pre
     *         token generation trigger.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
     *         assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
     *         triggers, the <code>ClientMetadata</code> parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the <code>ClientMetadata</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(refreshToken());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(clientSecret());
        hashCode = 31 * hashCode + Objects.hashCode(deviceKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetTokensFromRefreshTokenRequest)) {
            return false;
        }
        GetTokensFromRefreshTokenRequest other = (GetTokensFromRefreshTokenRequest) obj;
        return Objects.equals(refreshToken(), other.refreshToken()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(clientSecret(), other.clientSecret()) && Objects.equals(deviceKey(), other.deviceKey())
                && hasClientMetadata() == other.hasClientMetadata() && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetTokensFromRefreshTokenRequest")
                .add("RefreshToken", refreshToken() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientSecret", clientSecret() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DeviceKey", deviceKey()).add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RefreshToken":
            return Optional.ofNullable(clazz.cast(refreshToken()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "ClientSecret":
            return Optional.ofNullable(clazz.cast(clientSecret()));
        case "DeviceKey":
            return Optional.ofNullable(clazz.cast(deviceKey()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RefreshToken", REFRESH_TOKEN_FIELD);
        map.put("ClientId", CLIENT_ID_FIELD);
        map.put("ClientSecret", CLIENT_SECRET_FIELD);
        map.put("DeviceKey", DEVICE_KEY_FIELD);
        map.put("ClientMetadata", CLIENT_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GetTokensFromRefreshTokenRequest, T> g) {
        return obj -> g.apply((GetTokensFromRefreshTokenRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetTokensFromRefreshTokenRequest> {
        /**
         * <p>
         * A valid refresh token that can authorize the request for new tokens. When refresh token rotation is active in
         * the requested app client, this token is invalidated after the request is complete and after an optional grace
         * period.
         * </p>
         * 
         * @param refreshToken
         *        A valid refresh token that can authorize the request for new tokens. When refresh token rotation is
         *        active in the requested app client, this token is invalidated after the request is complete and after
         *        an optional grace period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder refreshToken(String refreshToken);

        /**
         * <p>
         * The app client that issued the refresh token to the user who wants to request new tokens.
         * </p>
         * 
         * @param clientId
         *        The app client that issued the refresh token to the user who wants to request new tokens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The client secret of the requested app client, if the client has a secret.
         * </p>
         * 
         * @param clientSecret
         *        The client secret of the requested app client, if the client has a secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientSecret(String clientSecret);

        /**
         * <p>
         * When you enable device remembering, Amazon Cognito issues a device key that you can use for device
         * authentication that bypasses multi-factor authentication (MFA). To implement
         * <code>GetTokensFromRefreshToken</code> in a user pool with device remembering, you must capture the device
         * key from the initial authentication request. If your application doesn't provide the key of a registered
         * device, Amazon Cognito issues a new one. You must provide the confirmed device key in this request if device
         * remembering is enabled in your user pool.
         * </p>
         * <p>
         * For more information about device remembering, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
         * >Working with devices</a>.
         * </p>
         * 
         * @param deviceKey
         *        When you enable device remembering, Amazon Cognito issues a device key that you can use for device
         *        authentication that bypasses multi-factor authentication (MFA). To implement
         *        <code>GetTokensFromRefreshToken</code> in a user pool with device remembering, you must capture the
         *        device key from the initial authentication request. If your application doesn't provide the key of a
         *        registered device, Amazon Cognito issues a new one. You must provide the confirmed device key in this
         *        request if device remembering is enabled in your user pool.</p>
         *        <p>
         *        For more information about device remembering, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
         *        >Working with devices</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceKey(String deviceKey);

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for certain custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * <code>GetTokensFromRefreshToken</code> API action, Amazon Cognito invokes the Lambda function the pre token
         * generation trigger.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned
         * to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * <code>ClientMetadata</code> parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the <code>ClientMetadata</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for certain custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        <code>GetTokensFromRefreshToken</code> API action, Amazon Cognito invokes the Lambda function the pre
         *        token generation trigger.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the
         *        following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
         *        assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
         *        triggers, the <code>ClientMetadata</code> parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the <code>ClientMetadata</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String refreshToken;

        private String clientId;

        private String clientSecret;

        private String deviceKey;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetTokensFromRefreshTokenRequest model) {
            super(model);
            refreshToken(model.refreshToken);
            clientId(model.clientId);
            clientSecret(model.clientSecret);
            deviceKey(model.deviceKey);
            clientMetadata(model.clientMetadata);
        }

        public final String getRefreshToken() {
            return refreshToken;
        }

        public final void setRefreshToken(String refreshToken) {
            this.refreshToken = refreshToken;
        }

        @Override
        public final Builder refreshToken(String refreshToken) {
            this.refreshToken = refreshToken;
            return this;
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getClientSecret() {
            return clientSecret;
        }

        public final void setClientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
        }

        @Override
        public final Builder clientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        public final String getDeviceKey() {
            return deviceKey;
        }

        public final void setDeviceKey(String deviceKey) {
            this.deviceKey = deviceKey;
        }

        @Override
        public final Builder deviceKey(String deviceKey) {
            this.deviceKey = deviceKey;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetTokensFromRefreshTokenRequest build() {
            return new GetTokensFromRefreshTokenRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
