/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The response of <a href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
 * <code>ConfigureAgent</code> </a> that specifies if an agent profiles or not and for how long to return profiling
 * data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AgentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<AgentConfiguration.Builder, AgentConfiguration> {
    private static final SdkField<Map<String, String>> AGENT_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("agentParameters")
            .getter(getter(AgentConfiguration::agentParametersAsStrings))
            .setter(setter(Builder::agentParametersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> PERIOD_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("periodInSeconds").getter(getter(AgentConfiguration::periodInSeconds))
            .setter(setter(Builder::periodInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("periodInSeconds").build()).build();

    private static final SdkField<Boolean> SHOULD_PROFILE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("shouldProfile").getter(getter(AgentConfiguration::shouldProfile)).setter(setter(Builder::shouldProfile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shouldProfile").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AGENT_PARAMETERS_FIELD,
            PERIOD_IN_SECONDS_FIELD, SHOULD_PROFILE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Map<String, String> agentParameters;

    private final Integer periodInSeconds;

    private final Boolean shouldProfile;

    private AgentConfiguration(BuilderImpl builder) {
        this.agentParameters = builder.agentParameters;
        this.periodInSeconds = builder.periodInSeconds;
        this.shouldProfile = builder.shouldProfile;
    }

    /**
     * <p>
     * Parameters used by the profiler. The valid parameters are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the profile. For
     * example, if CodeGuru Profiler finds a method <code>A</code>, which calls method <code>B</code>, which calls
     * method <code>C</code>, which calls method <code>D</code>, then the depth is 4. If the <code>maxDepth</code> is
     * set to 2, then the profiler evaluates <code>A</code> and <code>B</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending reports.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report profiles.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to profile
     * samples.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentParameters} method.
     * </p>
     * 
     * @return Parameters used by the profiler. The valid parameters are: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the
     *         profile. For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method
     *         <code>B</code>, which calls method <code>C</code>, which calls method <code>D</code>, then the depth is
     *         4. If the <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and
     *         <code>B</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
     *         reports.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
     *         profiles.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to
     *         profile samples.
     *         </p>
     *         </li>
     */
    public final Map<AgentParameterField, String> agentParameters() {
        return AgentParametersCopier.copyStringToEnum(agentParameters);
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAgentParameters() {
        return agentParameters != null && !(agentParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Parameters used by the profiler. The valid parameters are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the profile. For
     * example, if CodeGuru Profiler finds a method <code>A</code>, which calls method <code>B</code>, which calls
     * method <code>C</code>, which calls method <code>D</code>, then the depth is 4. If the <code>maxDepth</code> is
     * set to 2, then the profiler evaluates <code>A</code> and <code>B</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending reports.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report profiles.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to profile
     * samples.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentParameters} method.
     * </p>
     * 
     * @return Parameters used by the profiler. The valid parameters are: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the
     *         profile. For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method
     *         <code>B</code>, which calls method <code>C</code>, which calls method <code>D</code>, then the depth is
     *         4. If the <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and
     *         <code>B</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
     *         reports.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
     *         profiles.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to
     *         profile samples.
     *         </p>
     *         </li>
     */
    public final Map<String, String> agentParametersAsStrings() {
        return agentParameters;
    }

    /**
     * <p>
     * How long a profiling agent should send profiling data using <a
     * href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
     * <code>ConfigureAgent</code> </a>. For example, if this is set to 300, the profiling agent calls <a
     * href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
     * <code>ConfigureAgent</code> </a> every 5 minutes to submit the profiled data collected during that period.
     * </p>
     * 
     * @return How long a profiling agent should send profiling data using <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
     *         <code>ConfigureAgent</code> </a>. For example, if this is set to 300, the profiling agent calls <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
     *         <code>ConfigureAgent</code> </a> every 5 minutes to submit the profiled data collected during that
     *         period.
     */
    public final Integer periodInSeconds() {
        return periodInSeconds;
    }

    /**
     * <p>
     * A <code>Boolean</code> that specifies whether the profiling agent collects profiling data or not. Set to
     * <code>true</code> to enable profiling.
     * </p>
     * 
     * @return A <code>Boolean</code> that specifies whether the profiling agent collects profiling data or not. Set to
     *         <code>true</code> to enable profiling.
     */
    public final Boolean shouldProfile() {
        return shouldProfile;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentParameters() ? agentParametersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(periodInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(shouldProfile());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AgentConfiguration)) {
            return false;
        }
        AgentConfiguration other = (AgentConfiguration) obj;
        return hasAgentParameters() == other.hasAgentParameters()
                && Objects.equals(agentParametersAsStrings(), other.agentParametersAsStrings())
                && Objects.equals(periodInSeconds(), other.periodInSeconds())
                && Objects.equals(shouldProfile(), other.shouldProfile());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AgentConfiguration")
                .add("AgentParameters", hasAgentParameters() ? agentParametersAsStrings() : null)
                .add("PeriodInSeconds", periodInSeconds()).add("ShouldProfile", shouldProfile()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "agentParameters":
            return Optional.ofNullable(clazz.cast(agentParametersAsStrings()));
        case "periodInSeconds":
            return Optional.ofNullable(clazz.cast(periodInSeconds()));
        case "shouldProfile":
            return Optional.ofNullable(clazz.cast(shouldProfile()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("agentParameters", AGENT_PARAMETERS_FIELD);
        map.put("periodInSeconds", PERIOD_IN_SECONDS_FIELD);
        map.put("shouldProfile", SHOULD_PROFILE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AgentConfiguration, T> g) {
        return obj -> g.apply((AgentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AgentConfiguration> {
        /**
         * <p>
         * Parameters used by the profiler. The valid parameters are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the profile.
         * For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method <code>B</code>, which
         * calls method <code>C</code>, which calls method <code>D</code>, then the depth is 4. If the
         * <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and <code>B</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
         * reports.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
         * profiles.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to profile
         * samples.
         * </p>
         * </li>
         * </ul>
         * 
         * @param agentParameters
         *        Parameters used by the profiler. The valid parameters are: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the
         *        profile. For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method
         *        <code>B</code>, which calls method <code>C</code>, which calls method <code>D</code>, then the depth
         *        is 4. If the <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and
         *        <code>B</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
         *        reports.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
         *        profiles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to
         *        profile samples.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentParametersWithStrings(Map<String, String> agentParameters);

        /**
         * <p>
         * Parameters used by the profiler. The valid parameters are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the profile.
         * For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method <code>B</code>, which
         * calls method <code>C</code>, which calls method <code>D</code>, then the depth is 4. If the
         * <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and <code>B</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
         * reports.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
         * profiles.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to profile
         * samples.
         * </p>
         * </li>
         * </ul>
         * 
         * @param agentParameters
         *        Parameters used by the profiler. The valid parameters are: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MaxStackDepth</code> - The maximum depth of the stacks in the code that is represented in the
         *        profile. For example, if CodeGuru Profiler finds a method <code>A</code>, which calls method
         *        <code>B</code>, which calls method <code>C</code>, which calls method <code>D</code>, then the depth
         *        is 4. If the <code>maxDepth</code> is set to 2, then the profiler evaluates <code>A</code> and
         *        <code>B</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MemoryUsageLimitPercent</code> - The percentage of memory that is used by the profiler.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MinimumTimeForReportingInMilliseconds</code> - The minimum time in milliseconds between sending
         *        reports.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReportingIntervalInMilliseconds</code> - The reporting interval in milliseconds used to report
         *        profiles.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SamplingIntervalInMilliseconds</code> - The sampling interval in milliseconds that is used to
         *        profile samples.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentParameters(Map<AgentParameterField, String> agentParameters);

        /**
         * <p>
         * How long a profiling agent should send profiling data using <a
         * href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
         * <code>ConfigureAgent</code> </a>. For example, if this is set to 300, the profiling agent calls <a
         * href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
         * <code>ConfigureAgent</code> </a> every 5 minutes to submit the profiled data collected during that period.
         * </p>
         * 
         * @param periodInSeconds
         *        How long a profiling agent should send profiling data using <a
         *        href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
         *        <code>ConfigureAgent</code> </a>. For example, if this is set to 300, the profiling agent calls <a
         *        href="https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html">
         *        <code>ConfigureAgent</code> </a> every 5 minutes to submit the profiled data collected during that
         *        period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder periodInSeconds(Integer periodInSeconds);

        /**
         * <p>
         * A <code>Boolean</code> that specifies whether the profiling agent collects profiling data or not. Set to
         * <code>true</code> to enable profiling.
         * </p>
         * 
         * @param shouldProfile
         *        A <code>Boolean</code> that specifies whether the profiling agent collects profiling data or not. Set
         *        to <code>true</code> to enable profiling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shouldProfile(Boolean shouldProfile);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> agentParameters = DefaultSdkAutoConstructMap.getInstance();

        private Integer periodInSeconds;

        private Boolean shouldProfile;

        private BuilderImpl() {
        }

        private BuilderImpl(AgentConfiguration model) {
            agentParametersWithStrings(model.agentParameters);
            periodInSeconds(model.periodInSeconds);
            shouldProfile(model.shouldProfile);
        }

        public final Map<String, String> getAgentParameters() {
            if (agentParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return agentParameters;
        }

        public final void setAgentParameters(Map<String, String> agentParameters) {
            this.agentParameters = AgentParametersCopier.copy(agentParameters);
        }

        @Override
        public final Builder agentParametersWithStrings(Map<String, String> agentParameters) {
            this.agentParameters = AgentParametersCopier.copy(agentParameters);
            return this;
        }

        @Override
        public final Builder agentParameters(Map<AgentParameterField, String> agentParameters) {
            this.agentParameters = AgentParametersCopier.copyEnumToString(agentParameters);
            return this;
        }

        public final Integer getPeriodInSeconds() {
            return periodInSeconds;
        }

        public final void setPeriodInSeconds(Integer periodInSeconds) {
            this.periodInSeconds = periodInSeconds;
        }

        @Override
        public final Builder periodInSeconds(Integer periodInSeconds) {
            this.periodInSeconds = periodInSeconds;
            return this;
        }

        public final Boolean getShouldProfile() {
            return shouldProfile;
        }

        public final void setShouldProfile(Boolean shouldProfile) {
            this.shouldProfile = shouldProfile;
        }

        @Override
        public final Builder shouldProfile(Boolean shouldProfile) {
            this.shouldProfile = shouldProfile;
            return this;
        }

        @Override
        public AgentConfiguration build() {
            return new AgentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
