/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A potential improvement that was found from analyzing the profiling data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Recommendation implements SdkPojo, Serializable, ToCopyableBuilder<Recommendation.Builder, Recommendation> {
    private static final SdkField<Integer> ALL_MATCHES_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("allMatchesCount").getter(getter(Recommendation::allMatchesCount))
            .setter(setter(Builder::allMatchesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allMatchesCount").build()).build();

    private static final SdkField<Double> ALL_MATCHES_SUM_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("allMatchesSum").getter(getter(Recommendation::allMatchesSum)).setter(setter(Builder::allMatchesSum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allMatchesSum").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime")
            .getter(getter(Recommendation::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Pattern> PATTERN_FIELD = SdkField.<Pattern> builder(MarshallingType.SDK_POJO)
            .memberName("pattern").getter(getter(Recommendation::pattern)).setter(setter(Builder::pattern))
            .constructor(Pattern::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pattern").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime")
            .getter(getter(Recommendation::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<Match>> TOP_MATCHES_FIELD = SdkField
            .<List<Match>> builder(MarshallingType.LIST)
            .memberName("topMatches")
            .getter(getter(Recommendation::topMatches))
            .setter(setter(Builder::topMatches))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("topMatches").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Match> builder(MarshallingType.SDK_POJO)
                                            .constructor(Match::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALL_MATCHES_COUNT_FIELD,
            ALL_MATCHES_SUM_FIELD, END_TIME_FIELD, PATTERN_FIELD, START_TIME_FIELD, TOP_MATCHES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer allMatchesCount;

    private final Double allMatchesSum;

    private final Instant endTime;

    private final Pattern pattern;

    private final Instant startTime;

    private final List<Match> topMatches;

    private Recommendation(BuilderImpl builder) {
        this.allMatchesCount = builder.allMatchesCount;
        this.allMatchesSum = builder.allMatchesSum;
        this.endTime = builder.endTime;
        this.pattern = builder.pattern;
        this.startTime = builder.startTime;
        this.topMatches = builder.topMatches;
    }

    /**
     * <p>
     * How many different places in the profile graph triggered a match.
     * </p>
     * 
     * @return How many different places in the profile graph triggered a match.
     */
    public final Integer allMatchesCount() {
        return allMatchesCount;
    }

    /**
     * <p>
     * How much of the total sample count is potentially affected.
     * </p>
     * 
     * @return How much of the total sample count is potentially affected.
     */
    public final Double allMatchesSum() {
        return allMatchesSum;
    }

    /**
     * <p>
     * End time of the profile that was used by this analysis. This is specified using the ISO 8601 format. For example,
     * 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
     * </p>
     * 
     * @return End time of the profile that was used by this analysis. This is specified using the ISO 8601 format. For
     *         example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The pattern that analysis recognized in the profile to make this recommendation.
     * </p>
     * 
     * @return The pattern that analysis recognized in the profile to make this recommendation.
     */
    public final Pattern pattern() {
        return pattern;
    }

    /**
     * <p>
     * The start time of the profile that was used by this analysis. This is specified using the ISO 8601 format. For
     * example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
     * </p>
     * 
     * @return The start time of the profile that was used by this analysis. This is specified using the ISO 8601
     *         format. For example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the TopMatches property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTopMatches() {
        return topMatches != null && !(topMatches instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of the matches with most impact.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTopMatches} method.
     * </p>
     * 
     * @return List of the matches with most impact.
     */
    public final List<Match> topMatches() {
        return topMatches;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allMatchesCount());
        hashCode = 31 * hashCode + Objects.hashCode(allMatchesSum());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(pattern());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTopMatches() ? topMatches() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Recommendation)) {
            return false;
        }
        Recommendation other = (Recommendation) obj;
        return Objects.equals(allMatchesCount(), other.allMatchesCount())
                && Objects.equals(allMatchesSum(), other.allMatchesSum()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(pattern(), other.pattern()) && Objects.equals(startTime(), other.startTime())
                && hasTopMatches() == other.hasTopMatches() && Objects.equals(topMatches(), other.topMatches());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Recommendation").add("AllMatchesCount", allMatchesCount()).add("AllMatchesSum", allMatchesSum())
                .add("EndTime", endTime()).add("Pattern", pattern()).add("StartTime", startTime())
                .add("TopMatches", hasTopMatches() ? topMatches() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allMatchesCount":
            return Optional.ofNullable(clazz.cast(allMatchesCount()));
        case "allMatchesSum":
            return Optional.ofNullable(clazz.cast(allMatchesSum()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "pattern":
            return Optional.ofNullable(clazz.cast(pattern()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "topMatches":
            return Optional.ofNullable(clazz.cast(topMatches()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Recommendation, T> g) {
        return obj -> g.apply((Recommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Recommendation> {
        /**
         * <p>
         * How many different places in the profile graph triggered a match.
         * </p>
         * 
         * @param allMatchesCount
         *        How many different places in the profile graph triggered a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allMatchesCount(Integer allMatchesCount);

        /**
         * <p>
         * How much of the total sample count is potentially affected.
         * </p>
         * 
         * @param allMatchesSum
         *        How much of the total sample count is potentially affected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allMatchesSum(Double allMatchesSum);

        /**
         * <p>
         * End time of the profile that was used by this analysis. This is specified using the ISO 8601 format. For
         * example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
         * </p>
         * 
         * @param endTime
         *        End time of the profile that was used by this analysis. This is specified using the ISO 8601 format.
         *        For example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The pattern that analysis recognized in the profile to make this recommendation.
         * </p>
         * 
         * @param pattern
         *        The pattern that analysis recognized in the profile to make this recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pattern(Pattern pattern);

        /**
         * <p>
         * The pattern that analysis recognized in the profile to make this recommendation.
         * </p>
         * This is a convenience method that creates an instance of the {@link Pattern.Builder} avoiding the need to
         * create one manually via {@link Pattern#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Pattern.Builder#build()} is called immediately and its result is
         * passed to {@link #pattern(Pattern)}.
         * 
         * @param pattern
         *        a consumer that will call methods on {@link Pattern.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pattern(Pattern)
         */
        default Builder pattern(Consumer<Pattern.Builder> pattern) {
            return pattern(Pattern.builder().applyMutation(pattern).build());
        }

        /**
         * <p>
         * The start time of the profile that was used by this analysis. This is specified using the ISO 8601 format.
         * For example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM UTC.
         * </p>
         * 
         * @param startTime
         *        The start time of the profile that was used by this analysis. This is specified using the ISO 8601
         *        format. For example, 2020-06-01T13:15:02.001Z represents 1 millisecond past June 1, 2020 1:15:02 PM
         *        UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * List of the matches with most impact.
         * </p>
         * 
         * @param topMatches
         *        List of the matches with most impact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topMatches(Collection<Match> topMatches);

        /**
         * <p>
         * List of the matches with most impact.
         * </p>
         * 
         * @param topMatches
         *        List of the matches with most impact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topMatches(Match... topMatches);

        /**
         * <p>
         * List of the matches with most impact.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codeguruprofiler.model.Match.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codeguruprofiler.model.Match#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codeguruprofiler.model.Match.Builder#build()} is called immediately
         * and its result is passed to {@link #topMatches(List<Match>)}.
         * 
         * @param topMatches
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codeguruprofiler.model.Match.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #topMatches(java.util.Collection<Match>)
         */
        Builder topMatches(Consumer<Match.Builder>... topMatches);
    }

    static final class BuilderImpl implements Builder {
        private Integer allMatchesCount;

        private Double allMatchesSum;

        private Instant endTime;

        private Pattern pattern;

        private Instant startTime;

        private List<Match> topMatches = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Recommendation model) {
            allMatchesCount(model.allMatchesCount);
            allMatchesSum(model.allMatchesSum);
            endTime(model.endTime);
            pattern(model.pattern);
            startTime(model.startTime);
            topMatches(model.topMatches);
        }

        public final Integer getAllMatchesCount() {
            return allMatchesCount;
        }

        public final void setAllMatchesCount(Integer allMatchesCount) {
            this.allMatchesCount = allMatchesCount;
        }

        @Override
        public final Builder allMatchesCount(Integer allMatchesCount) {
            this.allMatchesCount = allMatchesCount;
            return this;
        }

        public final Double getAllMatchesSum() {
            return allMatchesSum;
        }

        public final void setAllMatchesSum(Double allMatchesSum) {
            this.allMatchesSum = allMatchesSum;
        }

        @Override
        public final Builder allMatchesSum(Double allMatchesSum) {
            this.allMatchesSum = allMatchesSum;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Pattern.Builder getPattern() {
            return pattern != null ? pattern.toBuilder() : null;
        }

        public final void setPattern(Pattern.BuilderImpl pattern) {
            this.pattern = pattern != null ? pattern.build() : null;
        }

        @Override
        public final Builder pattern(Pattern pattern) {
            this.pattern = pattern;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final List<Match.Builder> getTopMatches() {
            List<Match.Builder> result = MatchesCopier.copyToBuilder(this.topMatches);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTopMatches(Collection<Match.BuilderImpl> topMatches) {
            this.topMatches = MatchesCopier.copyFromBuilder(topMatches);
        }

        @Override
        public final Builder topMatches(Collection<Match> topMatches) {
            this.topMatches = MatchesCopier.copy(topMatches);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topMatches(Match... topMatches) {
            topMatches(Arrays.asList(topMatches));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topMatches(Consumer<Match.Builder>... topMatches) {
            topMatches(Stream.of(topMatches).map(c -> Match.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Recommendation build() {
            return new Recommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
