/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for notifications stored for each profiling group. This includes up to to two channels and a list
 * of event publishers associated with each channel.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotificationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<NotificationConfiguration.Builder, NotificationConfiguration> {
    private static final SdkField<List<Channel>> CHANNELS_FIELD = SdkField
            .<List<Channel>> builder(MarshallingType.LIST)
            .memberName("channels")
            .getter(getter(NotificationConfiguration::channels))
            .setter(setter(Builder::channels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("channels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Channel> builder(MarshallingType.SDK_POJO)
                                            .constructor(Channel::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANNELS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Channel> channels;

    private NotificationConfiguration(BuilderImpl builder) {
        this.channels = builder.channels;
    }

    /**
     * Returns true if the Channels property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasChannels() {
        return channels != null && !(channels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of up to two channels to be used for sending notifications for events detected from the application profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasChannels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List of up to two channels to be used for sending notifications for events detected from the application
     *         profile.
     */
    public final List<Channel> channels() {
        return channels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasChannels() ? channels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotificationConfiguration)) {
            return false;
        }
        NotificationConfiguration other = (NotificationConfiguration) obj;
        return hasChannels() == other.hasChannels() && Objects.equals(channels(), other.channels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NotificationConfiguration").add("Channels", hasChannels() ? channels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "channels":
            return Optional.ofNullable(clazz.cast(channels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NotificationConfiguration, T> g) {
        return obj -> g.apply((NotificationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NotificationConfiguration> {
        /**
         * <p>
         * List of up to two channels to be used for sending notifications for events detected from the application
         * profile.
         * </p>
         * 
         * @param channels
         *        List of up to two channels to be used for sending notifications for events detected from the
         *        application profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channels(Collection<Channel> channels);

        /**
         * <p>
         * List of up to two channels to be used for sending notifications for events detected from the application
         * profile.
         * </p>
         * 
         * @param channels
         *        List of up to two channels to be used for sending notifications for events detected from the
         *        application profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channels(Channel... channels);

        /**
         * <p>
         * List of up to two channels to be used for sending notifications for events detected from the application
         * profile.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Channel>.Builder} avoiding the need to
         * create one manually via {@link List<Channel>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Channel>.Builder#build()} is called immediately and its
         * result is passed to {@link #channels(List<Channel>)}.
         * 
         * @param channels
         *        a consumer that will call methods on {@link List<Channel>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #channels(List<Channel>)
         */
        Builder channels(Consumer<Channel.Builder>... channels);
    }

    static final class BuilderImpl implements Builder {
        private List<Channel> channels = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NotificationConfiguration model) {
            channels(model.channels);
        }

        public final Collection<Channel.Builder> getChannels() {
            if (channels instanceof SdkAutoConstructList) {
                return null;
            }
            return channels != null ? channels.stream().map(Channel::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder channels(Collection<Channel> channels) {
            this.channels = ChannelsCopier.copy(channels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder channels(Channel... channels) {
            channels(Arrays.asList(channels));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder channels(Consumer<Channel.Builder>... channels) {
            channels(Stream.of(channels).map(c -> Channel.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setChannels(Collection<Channel.BuilderImpl> channels) {
            this.channels = ChannelsCopier.copyFromBuilder(channels);
        }

        @Override
        public NotificationConfiguration build() {
            return new NotificationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
