/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about the metric that the analysis used when it detected the anomaly. The metric what is analyzed to create
 * recommendations. It includes the name of the frame that was analyzed and the type and thread states used to derive
 * the metric value for that frame.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Metric implements SdkPojo, Serializable, ToCopyableBuilder<Metric.Builder, Metric> {
    private static final SdkField<String> FRAME_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("frameName").getter(getter(Metric::frameName)).setter(setter(Builder::frameName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("frameName").build()).build();

    private static final SdkField<List<String>> THREAD_STATES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("threadStates")
            .getter(getter(Metric::threadStates))
            .setter(setter(Builder::threadStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threadStates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Metric::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FRAME_NAME_FIELD,
            THREAD_STATES_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String frameName;

    private final List<String> threadStates;

    private final String type;

    private Metric(BuilderImpl builder) {
        this.frameName = builder.frameName;
        this.threadStates = builder.threadStates;
        this.type = builder.type;
    }

    /**
     * <p>
     * The name of the method that appears as a frame in any stack in a profile.
     * </p>
     * 
     * @return The name of the method that appears as a frame in any stack in a profile.
     */
    public final String frameName() {
        return frameName;
    }

    /**
     * Returns true if the ThreadStates property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasThreadStates() {
        return threadStates != null && !(threadStates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of application runtime thread states that is used to calculate the metric value for the frame.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasThreadStates()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of application runtime thread states that is used to calculate the metric value for the frame.
     */
    public final List<String> threadStates() {
        return threadStates;
    }

    /**
     * <p>
     * A type that specifies how a metric for a frame is analyzed. The supported value
     * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is calculated
     * across the occurences of all frames in a profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MetricType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return A type that specifies how a metric for a frame is analyzed. The supported value
     *         <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
     *         calculated across the occurences of all frames in a profile.
     * @see MetricType
     */
    public final MetricType type() {
        return MetricType.fromValue(type);
    }

    /**
     * <p>
     * A type that specifies how a metric for a frame is analyzed. The supported value
     * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is calculated
     * across the occurences of all frames in a profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MetricType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return A type that specifies how a metric for a frame is analyzed. The supported value
     *         <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
     *         calculated across the occurences of all frames in a profile.
     * @see MetricType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(frameName());
        hashCode = 31 * hashCode + Objects.hashCode(hasThreadStates() ? threadStates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Metric)) {
            return false;
        }
        Metric other = (Metric) obj;
        return Objects.equals(frameName(), other.frameName()) && hasThreadStates() == other.hasThreadStates()
                && Objects.equals(threadStates(), other.threadStates()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Metric").add("FrameName", frameName())
                .add("ThreadStates", hasThreadStates() ? threadStates() : null).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "frameName":
            return Optional.ofNullable(clazz.cast(frameName()));
        case "threadStates":
            return Optional.ofNullable(clazz.cast(threadStates()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Metric, T> g) {
        return obj -> g.apply((Metric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Metric> {
        /**
         * <p>
         * The name of the method that appears as a frame in any stack in a profile.
         * </p>
         * 
         * @param frameName
         *        The name of the method that appears as a frame in any stack in a profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameName(String frameName);

        /**
         * <p>
         * The list of application runtime thread states that is used to calculate the metric value for the frame.
         * </p>
         * 
         * @param threadStates
         *        The list of application runtime thread states that is used to calculate the metric value for the
         *        frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadStates(Collection<String> threadStates);

        /**
         * <p>
         * The list of application runtime thread states that is used to calculate the metric value for the frame.
         * </p>
         * 
         * @param threadStates
         *        The list of application runtime thread states that is used to calculate the metric value for the
         *        frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadStates(String... threadStates);

        /**
         * <p>
         * A type that specifies how a metric for a frame is analyzed. The supported value
         * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         * calculated across the occurences of all frames in a profile.
         * </p>
         * 
         * @param type
         *        A type that specifies how a metric for a frame is analyzed. The supported value
         *        <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         *        calculated across the occurences of all frames in a profile.
         * @see MetricType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricType
         */
        Builder type(String type);

        /**
         * <p>
         * A type that specifies how a metric for a frame is analyzed. The supported value
         * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         * calculated across the occurences of all frames in a profile.
         * </p>
         * 
         * @param type
         *        A type that specifies how a metric for a frame is analyzed. The supported value
         *        <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         *        calculated across the occurences of all frames in a profile.
         * @see MetricType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricType
         */
        Builder type(MetricType type);
    }

    static final class BuilderImpl implements Builder {
        private String frameName;

        private List<String> threadStates = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(Metric model) {
            frameName(model.frameName);
            threadStates(model.threadStates);
            type(model.type);
        }

        public final String getFrameName() {
            return frameName;
        }

        @Override
        public final Builder frameName(String frameName) {
            this.frameName = frameName;
            return this;
        }

        public final void setFrameName(String frameName) {
            this.frameName = frameName;
        }

        public final Collection<String> getThreadStates() {
            if (threadStates instanceof SdkAutoConstructList) {
                return null;
            }
            return threadStates;
        }

        @Override
        public final Builder threadStates(Collection<String> threadStates) {
            this.threadStates = StringsCopier.copy(threadStates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threadStates(String... threadStates) {
            threadStates(Arrays.asList(threadStates));
            return this;
        }

        public final void setThreadStates(Collection<String> threadStates) {
            this.threadStates = StringsCopier.copy(threadStates);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MetricType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public Metric build() {
            return new Metric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
