/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the build input source code for the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectSource implements SdkPojo, Serializable, ToCopyableBuilder<ProjectSource.Builder, ProjectSource> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectSource::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectSource::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<Integer> GIT_CLONE_DEPTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ProjectSource::gitCloneDepth)).setter(setter(Builder::gitCloneDepth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gitCloneDepth").build()).build();

    private static final SdkField<GitSubmodulesConfig> GIT_SUBMODULES_CONFIG_FIELD = SdkField
            .<GitSubmodulesConfig> builder(MarshallingType.SDK_POJO).getter(getter(ProjectSource::gitSubmodulesConfig))
            .setter(setter(Builder::gitSubmodulesConfig)).constructor(GitSubmodulesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gitSubmodulesConfig").build())
            .build();

    private static final SdkField<String> BUILDSPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectSource::buildspec)).setter(setter(Builder::buildspec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildspec").build()).build();

    private static final SdkField<SourceAuth> AUTH_FIELD = SdkField.<SourceAuth> builder(MarshallingType.SDK_POJO)
            .getter(getter(ProjectSource::auth)).setter(setter(Builder::auth)).constructor(SourceAuth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("auth").build()).build();

    private static final SdkField<Boolean> REPORT_BUILD_STATUS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ProjectSource::reportBuildStatus)).setter(setter(Builder::reportBuildStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reportBuildStatus").build()).build();

    private static final SdkField<Boolean> INSECURE_SSL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ProjectSource::insecureSsl)).setter(setter(Builder::insecureSsl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("insecureSsl").build()).build();

    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectSource::sourceIdentifier)).setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceIdentifier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            GIT_CLONE_DEPTH_FIELD, GIT_SUBMODULES_CONFIG_FIELD, BUILDSPEC_FIELD, AUTH_FIELD, REPORT_BUILD_STATUS_FIELD,
            INSECURE_SSL_FIELD, SOURCE_IDENTIFIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final Integer gitCloneDepth;

    private final GitSubmodulesConfig gitSubmodulesConfig;

    private final String buildspec;

    private final SourceAuth auth;

    private final Boolean reportBuildStatus;

    private final Boolean insecureSsl;

    private final String sourceIdentifier;

    private ProjectSource(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.gitCloneDepth = builder.gitCloneDepth;
        this.gitSubmodulesConfig = builder.gitSubmodulesConfig;
        this.buildspec = builder.buildspec;
        this.auth = builder.auth;
        this.reportBuildStatus = builder.reportBuildStatus;
        this.insecureSsl = builder.insecureSsl;
        this.sourceIdentifier = builder.sourceIdentifier;
    }

    /**
     * <p>
     * The type of repository that contains the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in AWS
     * CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB</code>: The source code is in a GitHub repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_SOURCE</code>: The project does not have input source code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of repository that contains the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     *         AWS CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB</code>: The source code is in a GitHub repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_SOURCE</code>: The project does not have input source code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
     *         </p>
     *         </li>
     * @see SourceType
     */
    public SourceType type() {
        return SourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of repository that contains the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in AWS
     * CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB</code>: The source code is in a GitHub repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_SOURCE</code>: The project does not have input source code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of repository that contains the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     *         AWS CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB</code>: The source code is in a GitHub repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_SOURCE</code>: The project does not have input source code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
     *         </p>
     *         </li>
     * @see SourceType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about the location of the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For source code settings that are specified in the source action of a pipeline in AWS CodePipeline,
     * <code>location</code> should not be specified. If it is specified, AWS CodePipeline ignores it. This is because
     * AWS CodePipeline uses the settings in a pipeline's source action instead of this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in an AWS CodeCommit repository, the HTTPS clone URL to the repository that contains the source
     * code and the build spec (for example,
     * <code>https://git-codecommit.<i>region-ID</i>.amazonaws.com/v1/repos/<i>repo-name</i> </code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in an Amazon Simple Storage Service (Amazon S3) input bucket, one of the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The path to the ZIP file that contains the source code (for example,
     * <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>object-name</i>.zip</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The path to the folder that contains the source code (for example,
     * <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>source-code</i>/<i>folder</i>/</code>).
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source and the
     * build spec. You must connect your AWS account to your GitHub account. Use the AWS CodeBuild console to start
     * creating a build project. When you use the console to connect (or reconnect) with GitHub, on the GitHub
     * <b>Authorize application</b> page, for <b>Organization access</b>, choose <b>Request access</b> next to each
     * repository you want to allow AWS CodeBuild to have access to, and then choose <b>Authorize application</b>.
     * (After you have connected to your GitHub account, you do not need to finish creating the build project. You can
     * leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use this connection, in the <code>source</code>
     * object, set the <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source and the
     * build spec. You must connect your AWS account to your Bitbucket account. Use the AWS CodeBuild console to start
     * creating a build project. When you use the console to connect (or reconnect) with Bitbucket, on the Bitbucket
     * <b>Confirm access to your account</b> page, choose <b>Grant access</b>. (After you have connected to your
     * Bitbucket account, you do not need to finish creating the build project. You can leave the AWS CodeBuild
     * console.) To instruct AWS CodeBuild to use this connection, in the <code>source</code> object, set the
     * <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Information about the location of the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For source code settings that are specified in the source action of a pipeline in AWS CodePipeline,
     *         <code>location</code> should not be specified. If it is specified, AWS CodePipeline ignores it. This is
     *         because AWS CodePipeline uses the settings in a pipeline's source action instead of this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in an AWS CodeCommit repository, the HTTPS clone URL to the repository that contains the
     *         source code and the build spec (for example,
     *         <code>https://git-codecommit.<i>region-ID</i>.amazonaws.com/v1/repos/<i>repo-name</i> </code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in an Amazon Simple Storage Service (Amazon S3) input bucket, one of the following.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The path to the ZIP file that contains the source code (for example,
     *         <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>object-name</i>.zip</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The path to the folder that contains the source code (for example,
     *         <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>source-code</i>/<i>folder</i>/</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source
     *         and the build spec. You must connect your AWS account to your GitHub account. Use the AWS CodeBuild
     *         console to start creating a build project. When you use the console to connect (or reconnect) with
     *         GitHub, on the GitHub <b>Authorize application</b> page, for <b>Organization access</b>, choose
     *         <b>Request access</b> next to each repository you want to allow AWS CodeBuild to have access to, and then
     *         choose <b>Authorize application</b>. (After you have connected to your GitHub account, you do not need to
     *         finish creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to
     *         use this connection, in the <code>source</code> object, set the <code>auth</code> object's
     *         <code>type</code> value to <code>OAUTH</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source
     *         and the build spec. You must connect your AWS account to your Bitbucket account. Use the AWS CodeBuild
     *         console to start creating a build project. When you use the console to connect (or reconnect) with
     *         Bitbucket, on the Bitbucket <b>Confirm access to your account</b> page, choose <b>Grant access</b>.
     *         (After you have connected to your Bitbucket account, you do not need to finish creating the build
     *         project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use this connection, in
     *         the <code>source</code> object, set the <code>auth</code> object's <code>type</code> value to
     *         <code>OAUTH</code>.
     *         </p>
     *         </li>
     */
    public String location() {
        return location;
    }

    /**
     * <p>
     * Information about the Git clone depth for the build project.
     * </p>
     * 
     * @return Information about the Git clone depth for the build project.
     */
    public Integer gitCloneDepth() {
        return gitCloneDepth;
    }

    /**
     * <p>
     * Information about the Git submodules configuration for the build project.
     * </p>
     * 
     * @return Information about the Git submodules configuration for the build project.
     */
    public GitSubmodulesConfig gitSubmodulesConfig() {
        return gitSubmodulesConfig;
    }

    /**
     * <p>
     * The build spec declaration to use for the builds in this build project.
     * </p>
     * <p>
     * If this value is not specified, a build spec must be included along with the source code to be built.
     * </p>
     * 
     * @return The build spec declaration to use for the builds in this build project.</p>
     *         <p>
     *         If this value is not specified, a build spec must be included along with the source code to be built.
     */
    public String buildspec() {
        return buildspec;
    }

    /**
     * <p>
     * Information about the authorization settings for AWS CodeBuild to access the source code to be built.
     * </p>
     * <p>
     * This information is for the AWS CodeBuild console's use only. Your code should not get or set this information
     * directly.
     * </p>
     * 
     * @return Information about the authorization settings for AWS CodeBuild to access the source code to be built.</p>
     *         <p>
     *         This information is for the AWS CodeBuild console's use only. Your code should not get or set this
     *         information directly.
     */
    public SourceAuth auth() {
        return auth;
    }

    /**
     * <p>
     * Set to true to report the status of a build's start and finish to your source provider. This option is valid only
     * when your source provider is GitHub, GitHub Enterprise, or Bitbucket. If this is set and you use a different
     * source provider, an invalidInputException is thrown.
     * </p>
     * <note>
     * <p>
     * The status of a build triggered by a webhook is always reported to your source provider.
     * </p>
     * </note>
     * 
     * @return Set to true to report the status of a build's start and finish to your source provider. This option is
     *         valid only when your source provider is GitHub, GitHub Enterprise, or Bitbucket. If this is set and you
     *         use a different source provider, an invalidInputException is thrown. </p> <note>
     *         <p>
     *         The status of a build triggered by a webhook is always reported to your source provider.
     *         </p>
     */
    public Boolean reportBuildStatus() {
        return reportBuildStatus;
    }

    /**
     * <p>
     * Enable this flag to ignore SSL warnings while connecting to the project source code.
     * </p>
     * 
     * @return Enable this flag to ignore SSL warnings while connecting to the project source code.
     */
    public Boolean insecureSsl() {
        return insecureSsl;
    }

    /**
     * <p>
     * An identifier for this project source.
     * </p>
     * 
     * @return An identifier for this project source.
     */
    public String sourceIdentifier() {
        return sourceIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(gitCloneDepth());
        hashCode = 31 * hashCode + Objects.hashCode(gitSubmodulesConfig());
        hashCode = 31 * hashCode + Objects.hashCode(buildspec());
        hashCode = 31 * hashCode + Objects.hashCode(auth());
        hashCode = 31 * hashCode + Objects.hashCode(reportBuildStatus());
        hashCode = 31 * hashCode + Objects.hashCode(insecureSsl());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectSource)) {
            return false;
        }
        ProjectSource other = (ProjectSource) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(gitCloneDepth(), other.gitCloneDepth())
                && Objects.equals(gitSubmodulesConfig(), other.gitSubmodulesConfig())
                && Objects.equals(buildspec(), other.buildspec()) && Objects.equals(auth(), other.auth())
                && Objects.equals(reportBuildStatus(), other.reportBuildStatus())
                && Objects.equals(insecureSsl(), other.insecureSsl())
                && Objects.equals(sourceIdentifier(), other.sourceIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProjectSource").add("Type", typeAsString()).add("Location", location())
                .add("GitCloneDepth", gitCloneDepth()).add("GitSubmodulesConfig", gitSubmodulesConfig())
                .add("Buildspec", buildspec()).add("Auth", auth()).add("ReportBuildStatus", reportBuildStatus())
                .add("InsecureSsl", insecureSsl()).add("SourceIdentifier", sourceIdentifier()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "gitCloneDepth":
            return Optional.ofNullable(clazz.cast(gitCloneDepth()));
        case "gitSubmodulesConfig":
            return Optional.ofNullable(clazz.cast(gitSubmodulesConfig()));
        case "buildspec":
            return Optional.ofNullable(clazz.cast(buildspec()));
        case "auth":
            return Optional.ofNullable(clazz.cast(auth()));
        case "reportBuildStatus":
            return Optional.ofNullable(clazz.cast(reportBuildStatus()));
        case "insecureSsl":
            return Optional.ofNullable(clazz.cast(insecureSsl()));
        case "sourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectSource, T> g) {
        return obj -> g.apply((ProjectSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectSource> {
        /**
         * <p>
         * The type of repository that contains the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in AWS
         * CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB</code>: The source code is in a GitHub repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_SOURCE</code>: The project does not have input source code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of repository that contains the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline
         *        in AWS CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB</code>: The source code is in a GitHub repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_SOURCE</code>: The project does not have input source code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
         *        </p>
         *        </li>
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of repository that contains the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in AWS
         * CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB</code>: The source code is in a GitHub repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_SOURCE</code>: The project does not have input source code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of repository that contains the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODECOMMIT</code>: The source code is in an AWS CodeCommit repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline
         *        in AWS CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB</code>: The source code is in a GitHub repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_SOURCE</code>: The project does not have input source code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The source code is in an Amazon Simple Storage Service (Amazon S3) input bucket.
         *        </p>
         *        </li>
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder type(SourceType type);

        /**
         * <p>
         * Information about the location of the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For source code settings that are specified in the source action of a pipeline in AWS CodePipeline,
         * <code>location</code> should not be specified. If it is specified, AWS CodePipeline ignores it. This is
         * because AWS CodePipeline uses the settings in a pipeline's source action instead of this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in an AWS CodeCommit repository, the HTTPS clone URL to the repository that contains the
         * source code and the build spec (for example,
         * <code>https://git-codecommit.<i>region-ID</i>.amazonaws.com/v1/repos/<i>repo-name</i> </code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in an Amazon Simple Storage Service (Amazon S3) input bucket, one of the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The path to the ZIP file that contains the source code (for example,
         * <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>object-name</i>.zip</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The path to the folder that contains the source code (for example,
         * <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>source-code</i>/<i>folder</i>/</code>).
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source and
         * the build spec. You must connect your AWS account to your GitHub account. Use the AWS CodeBuild console to
         * start creating a build project. When you use the console to connect (or reconnect) with GitHub, on the GitHub
         * <b>Authorize application</b> page, for <b>Organization access</b>, choose <b>Request access</b> next to each
         * repository you want to allow AWS CodeBuild to have access to, and then choose <b>Authorize application</b>.
         * (After you have connected to your GitHub account, you do not need to finish creating the build project. You
         * can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use this connection, in the
         * <code>source</code> object, set the <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source and
         * the build spec. You must connect your AWS account to your Bitbucket account. Use the AWS CodeBuild console to
         * start creating a build project. When you use the console to connect (or reconnect) with Bitbucket, on the
         * Bitbucket <b>Confirm access to your account</b> page, choose <b>Grant access</b>. (After you have connected
         * to your Bitbucket account, you do not need to finish creating the build project. You can leave the AWS
         * CodeBuild console.) To instruct AWS CodeBuild to use this connection, in the <code>source</code> object, set
         * the <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param location
         *        Information about the location of the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For source code settings that are specified in the source action of a pipeline in AWS CodePipeline,
         *        <code>location</code> should not be specified. If it is specified, AWS CodePipeline ignores it. This
         *        is because AWS CodePipeline uses the settings in a pipeline's source action instead of this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in an AWS CodeCommit repository, the HTTPS clone URL to the repository that contains
         *        the source code and the build spec (for example,
         *        <code>https://git-codecommit.<i>region-ID</i>.amazonaws.com/v1/repos/<i>repo-name</i> </code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in an Amazon Simple Storage Service (Amazon S3) input bucket, one of the following.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The path to the ZIP file that contains the source code (for example,
         *        <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>object-name</i>.zip</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The path to the folder that contains the source code (for example,
         *        <code> <i>bucket-name</i>/<i>path</i>/<i>to</i>/<i>source-code</i>/<i>folder</i>/</code>).
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source
         *        and the build spec. You must connect your AWS account to your GitHub account. Use the AWS CodeBuild
         *        console to start creating a build project. When you use the console to connect (or reconnect) with
         *        GitHub, on the GitHub <b>Authorize application</b> page, for <b>Organization access</b>, choose
         *        <b>Request access</b> next to each repository you want to allow AWS CodeBuild to have access to, and
         *        then choose <b>Authorize application</b>. (After you have connected to your GitHub account, you do not
         *        need to finish creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS
         *        CodeBuild to use this connection, in the <code>source</code> object, set the <code>auth</code>
         *        object's <code>type</code> value to <code>OAUTH</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the
         *        source and the build spec. You must connect your AWS account to your Bitbucket account. Use the AWS
         *        CodeBuild console to start creating a build project. When you use the console to connect (or
         *        reconnect) with Bitbucket, on the Bitbucket <b>Confirm access to your account</b> page, choose
         *        <b>Grant access</b>. (After you have connected to your Bitbucket account, you do not need to finish
         *        creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use
         *        this connection, in the <code>source</code> object, set the <code>auth</code> object's
         *        <code>type</code> value to <code>OAUTH</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * Information about the Git clone depth for the build project.
         * </p>
         * 
         * @param gitCloneDepth
         *        Information about the Git clone depth for the build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gitCloneDepth(Integer gitCloneDepth);

        /**
         * <p>
         * Information about the Git submodules configuration for the build project.
         * </p>
         * 
         * @param gitSubmodulesConfig
         *        Information about the Git submodules configuration for the build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gitSubmodulesConfig(GitSubmodulesConfig gitSubmodulesConfig);

        /**
         * <p>
         * Information about the Git submodules configuration for the build project.
         * </p>
         * This is a convenience that creates an instance of the {@link GitSubmodulesConfig.Builder} avoiding the need
         * to create one manually via {@link GitSubmodulesConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link GitSubmodulesConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #gitSubmodulesConfig(GitSubmodulesConfig)}.
         * 
         * @param gitSubmodulesConfig
         *        a consumer that will call methods on {@link GitSubmodulesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gitSubmodulesConfig(GitSubmodulesConfig)
         */
        default Builder gitSubmodulesConfig(Consumer<GitSubmodulesConfig.Builder> gitSubmodulesConfig) {
            return gitSubmodulesConfig(GitSubmodulesConfig.builder().applyMutation(gitSubmodulesConfig).build());
        }

        /**
         * <p>
         * The build spec declaration to use for the builds in this build project.
         * </p>
         * <p>
         * If this value is not specified, a build spec must be included along with the source code to be built.
         * </p>
         * 
         * @param buildspec
         *        The build spec declaration to use for the builds in this build project.</p>
         *        <p>
         *        If this value is not specified, a build spec must be included along with the source code to be built.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildspec(String buildspec);

        /**
         * <p>
         * Information about the authorization settings for AWS CodeBuild to access the source code to be built.
         * </p>
         * <p>
         * This information is for the AWS CodeBuild console's use only. Your code should not get or set this
         * information directly.
         * </p>
         * 
         * @param auth
         *        Information about the authorization settings for AWS CodeBuild to access the source code to be
         *        built.</p>
         *        <p>
         *        This information is for the AWS CodeBuild console's use only. Your code should not get or set this
         *        information directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auth(SourceAuth auth);

        /**
         * <p>
         * Information about the authorization settings for AWS CodeBuild to access the source code to be built.
         * </p>
         * <p>
         * This information is for the AWS CodeBuild console's use only. Your code should not get or set this
         * information directly.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceAuth.Builder} avoiding the need to create
         * one manually via {@link SourceAuth#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceAuth.Builder#build()} is called immediately and its result
         * is passed to {@link #auth(SourceAuth)}.
         * 
         * @param auth
         *        a consumer that will call methods on {@link SourceAuth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auth(SourceAuth)
         */
        default Builder auth(Consumer<SourceAuth.Builder> auth) {
            return auth(SourceAuth.builder().applyMutation(auth).build());
        }

        /**
         * <p>
         * Set to true to report the status of a build's start and finish to your source provider. This option is valid
         * only when your source provider is GitHub, GitHub Enterprise, or Bitbucket. If this is set and you use a
         * different source provider, an invalidInputException is thrown.
         * </p>
         * <note>
         * <p>
         * The status of a build triggered by a webhook is always reported to your source provider.
         * </p>
         * </note>
         * 
         * @param reportBuildStatus
         *        Set to true to report the status of a build's start and finish to your source provider. This option is
         *        valid only when your source provider is GitHub, GitHub Enterprise, or Bitbucket. If this is set and
         *        you use a different source provider, an invalidInputException is thrown. </p> <note>
         *        <p>
         *        The status of a build triggered by a webhook is always reported to your source provider.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportBuildStatus(Boolean reportBuildStatus);

        /**
         * <p>
         * Enable this flag to ignore SSL warnings while connecting to the project source code.
         * </p>
         * 
         * @param insecureSsl
         *        Enable this flag to ignore SSL warnings while connecting to the project source code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insecureSsl(Boolean insecureSsl);

        /**
         * <p>
         * An identifier for this project source.
         * </p>
         * 
         * @param sourceIdentifier
         *        An identifier for this project source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private Integer gitCloneDepth;

        private GitSubmodulesConfig gitSubmodulesConfig;

        private String buildspec;

        private SourceAuth auth;

        private Boolean reportBuildStatus;

        private Boolean insecureSsl;

        private String sourceIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectSource model) {
            type(model.type);
            location(model.location);
            gitCloneDepth(model.gitCloneDepth);
            gitSubmodulesConfig(model.gitSubmodulesConfig);
            buildspec(model.buildspec);
            auth(model.auth);
            reportBuildStatus(model.reportBuildStatus);
            insecureSsl(model.insecureSsl);
            sourceIdentifier(model.sourceIdentifier);
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getLocation() {
            return location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        public final Integer getGitCloneDepth() {
            return gitCloneDepth;
        }

        @Override
        public final Builder gitCloneDepth(Integer gitCloneDepth) {
            this.gitCloneDepth = gitCloneDepth;
            return this;
        }

        public final void setGitCloneDepth(Integer gitCloneDepth) {
            this.gitCloneDepth = gitCloneDepth;
        }

        public final GitSubmodulesConfig.Builder getGitSubmodulesConfig() {
            return gitSubmodulesConfig != null ? gitSubmodulesConfig.toBuilder() : null;
        }

        @Override
        public final Builder gitSubmodulesConfig(GitSubmodulesConfig gitSubmodulesConfig) {
            this.gitSubmodulesConfig = gitSubmodulesConfig;
            return this;
        }

        public final void setGitSubmodulesConfig(GitSubmodulesConfig.BuilderImpl gitSubmodulesConfig) {
            this.gitSubmodulesConfig = gitSubmodulesConfig != null ? gitSubmodulesConfig.build() : null;
        }

        public final String getBuildspec() {
            return buildspec;
        }

        @Override
        public final Builder buildspec(String buildspec) {
            this.buildspec = buildspec;
            return this;
        }

        public final void setBuildspec(String buildspec) {
            this.buildspec = buildspec;
        }

        public final SourceAuth.Builder getAuth() {
            return auth != null ? auth.toBuilder() : null;
        }

        @Override
        public final Builder auth(SourceAuth auth) {
            this.auth = auth;
            return this;
        }

        public final void setAuth(SourceAuth.BuilderImpl auth) {
            this.auth = auth != null ? auth.build() : null;
        }

        public final Boolean getReportBuildStatus() {
            return reportBuildStatus;
        }

        @Override
        public final Builder reportBuildStatus(Boolean reportBuildStatus) {
            this.reportBuildStatus = reportBuildStatus;
            return this;
        }

        public final void setReportBuildStatus(Boolean reportBuildStatus) {
            this.reportBuildStatus = reportBuildStatus;
        }

        public final Boolean getInsecureSsl() {
            return insecureSsl;
        }

        @Override
        public final Builder insecureSsl(Boolean insecureSsl) {
            this.insecureSsl = insecureSsl;
            return this;
        }

        public final void setInsecureSsl(Boolean insecureSsl) {
            this.insecureSsl = insecureSsl;
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        public ProjectSource build() {
            return new ProjectSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
