/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codebuild.model.AccountLimitExceededException;
import software.amazon.awssdk.services.codebuild.model.BatchDeleteBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchDeleteBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportsResponse;
import software.amazon.awssdk.services.codebuild.model.CodeBuildException;
import software.amazon.awssdk.services.codebuild.model.CreateProjectRequest;
import software.amazon.awssdk.services.codebuild.model.CreateProjectResponse;
import software.amazon.awssdk.services.codebuild.model.CreateReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.CreateReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.CreateWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.CreateWebhookResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteProjectRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteProjectResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteReportRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteReportResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.codebuild.model.DescribeTestCasesRequest;
import software.amazon.awssdk.services.codebuild.model.DescribeTestCasesResponse;
import software.amazon.awssdk.services.codebuild.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.ImportSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.ImportSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.InvalidInputException;
import software.amazon.awssdk.services.codebuild.model.InvalidateProjectCacheRequest;
import software.amazon.awssdk.services.codebuild.model.InvalidateProjectCacheResponse;
import software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectRequest;
import software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectResponse;
import software.amazon.awssdk.services.codebuild.model.ListBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.ListBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.ListCuratedEnvironmentImagesRequest;
import software.amazon.awssdk.services.codebuild.model.ListCuratedEnvironmentImagesResponse;
import software.amazon.awssdk.services.codebuild.model.ListProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.ListProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportsRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSharedProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSharedProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.OAuthProviderException;
import software.amazon.awssdk.services.codebuild.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codebuild.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codebuild.model.StartBuildRequest;
import software.amazon.awssdk.services.codebuild.model.StartBuildResponse;
import software.amazon.awssdk.services.codebuild.model.StopBuildRequest;
import software.amazon.awssdk.services.codebuild.model.StopBuildResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateProjectRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateProjectResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateWebhookResponse;
import software.amazon.awssdk.services.codebuild.transform.BatchDeleteBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetReportsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateWebhookRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteReportRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DescribeTestCasesRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ImportSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.InvalidateProjectCacheRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListBuildsForProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListCuratedEnvironmentImagesRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportsForReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSharedProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSharedReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.StartBuildRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.StopBuildRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateWebhookRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeBuildAsyncClient}.
 *
 * @see CodeBuildAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeBuildAsyncClient implements CodeBuildAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeBuildAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeBuildAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes one or more builds.
     * </p>
     *
     * @param batchDeleteBuildsRequest
     * @return A Java Future containing the result of the BatchDeleteBuilds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.BatchDeleteBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchDeleteBuilds" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteBuildsResponse> batchDeleteBuilds(BatchDeleteBuildsRequest batchDeleteBuildsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteBuildsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteBuildsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteBuildsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteBuildsRequest, BatchDeleteBuildsResponse>()
                            .withOperationName("BatchDeleteBuilds")
                            .withMarshaller(new BatchDeleteBuildsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchDeleteBuildsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about one or more builds.
     * </p>
     *
     * @param batchGetBuildsRequest
     * @return A Java Future containing the result of the BatchGetBuilds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.BatchGetBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetBuildsResponse> batchGetBuilds(BatchGetBuildsRequest batchGetBuildsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetBuildsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetBuildsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetBuildsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetBuildsRequest, BatchGetBuildsResponse>()
                            .withOperationName("BatchGetBuilds")
                            .withMarshaller(new BatchGetBuildsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetBuildsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about one or more build projects.
     * </p>
     *
     * @param batchGetProjectsRequest
     * @return A Java Future containing the result of the BatchGetProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.BatchGetProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetProjectsResponse> batchGetProjects(BatchGetProjectsRequest batchGetProjectsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetProjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetProjectsRequest, BatchGetProjectsResponse>()
                            .withOperationName("BatchGetProjects")
                            .withMarshaller(new BatchGetProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetProjectsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of report groups.
     * </p>
     *
     * @param batchGetReportGroupsRequest
     * @return A Java Future containing the result of the BatchGetReportGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.BatchGetReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetReportGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetReportGroupsResponse> batchGetReportGroups(
            BatchGetReportGroupsRequest batchGetReportGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetReportGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetReportGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetReportGroupsRequest, BatchGetReportGroupsResponse>()
                            .withOperationName("BatchGetReportGroups")
                            .withMarshaller(new BatchGetReportGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetReportGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of reports.
     * </p>
     *
     * @param batchGetReportsRequest
     * @return A Java Future containing the result of the BatchGetReports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.BatchGetReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetReportsResponse> batchGetReports(BatchGetReportsRequest batchGetReportsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetReportsRequest, BatchGetReportsResponse>()
                            .withOperationName("BatchGetReports")
                            .withMarshaller(new BatchGetReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetReportsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a build project.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceAlreadyExistsException The specified AWS resource cannot be created, because an AWS resource
     *         with the same settings already exists.</li>
     *         <li>AccountLimitExceededException An AWS service limit was exceeded for the calling AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.CreateProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createProjectRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a report group. A report group contains a collection of reports.
     * </p>
     *
     * @param createReportGroupRequest
     * @return A Java Future containing the result of the CreateReportGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceAlreadyExistsException The specified AWS resource cannot be created, because an AWS resource
     *         with the same settings already exists.</li>
     *         <li>AccountLimitExceededException An AWS service limit was exceeded for the calling AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.CreateReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateReportGroupResponse> createReportGroup(CreateReportGroupRequest createReportGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateReportGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateReportGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateReportGroupRequest, CreateReportGroupResponse>()
                            .withOperationName("CreateReportGroup")
                            .withMarshaller(new CreateReportGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createReportGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository,
     * enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.
     * </p>
     * <important>
     * <p>
     * If you enable webhooks for an AWS CodeBuild project, and the project is used as a build step in AWS CodePipeline,
     * then two identical builds are created for each commit. One build is triggered through webhooks, and one through
     * AWS CodePipeline. Because billing is on a per-build basis, you are billed for both builds. Therefore, if you are
     * using AWS CodePipeline, we recommend that you disable webhooks in AWS CodeBuild. In the AWS CodeBuild console,
     * clear the Webhook box. For more information, see step 5 in <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/change-project.html#change-project-console">Change a
     * Build Project's Settings</a>.
     * </p>
     * </important>
     *
     * @param createWebhookRequest
     * @return A Java Future containing the result of the CreateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>OAuthProviderException There was a problem with the underlying OAuth provider.</li>
     *         <li>ResourceAlreadyExistsException The specified AWS resource cannot be created, because an AWS resource
     *         with the same settings already exists.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.CreateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWebhookResponse> createWebhook(CreateWebhookRequest createWebhookRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWebhookRequest, CreateWebhookResponse>()
                            .withOperationName("CreateWebhook")
                            .withMarshaller(new CreateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createWebhookRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a build project. When you delete a project, its builds are not deleted.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteProjectRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a report.
     * </p>
     *
     * @param deleteReportRequest
     * @return A Java Future containing the result of the DeleteReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteReport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReportResponse> deleteReport(DeleteReportRequest deleteReportRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReportRequest, DeleteReportResponse>()
                            .withOperationName("DeleteReport").withMarshaller(new DeleteReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteReportRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>DeleteReportGroup</code>: Deletes a report group. Before you delete a report group, you must delete its
     * reports. Use <a
     * href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html"
     * >ListReportsForReportGroup</a> to get the reports in a report group. Use <a
     * href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html">DeleteReport</a> to delete
     * the reports. If you call <code>DeleteReportGroup</code> for a report group that contains one or more reports, an
     * exception is thrown.
     * </p>
     *
     * @param deleteReportGroupRequest
     * @return A Java Future containing the result of the DeleteReportGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReportGroupResponse> deleteReportGroup(DeleteReportGroupRequest deleteReportGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReportGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReportGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReportGroupRequest, DeleteReportGroupResponse>()
                            .withOperationName("DeleteReportGroup")
                            .withMarshaller(new DeleteReportGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteReportGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resource policy that is identified by its resource ARN.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy")
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteResourcePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.
     * </p>
     *
     * @param deleteSourceCredentialsRequest
     * @return A Java Future containing the result of the DeleteSourceCredentials operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSourceCredentialsResponse> deleteSourceCredentials(
            DeleteSourceCredentialsRequest deleteSourceCredentialsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSourceCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSourceCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSourceCredentialsRequest, DeleteSourceCredentialsResponse>()
                            .withOperationName("DeleteSourceCredentials")
                            .withMarshaller(new DeleteSourceCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSourceCredentialsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository,
     * stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.
     * </p>
     *
     * @param deleteWebhookRequest
     * @return A Java Future containing the result of the DeleteWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>OAuthProviderException There was a problem with the underlying OAuth provider.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DeleteWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWebhookResponse> deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                            .withOperationName("DeleteWebhook")
                            .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteWebhookRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of details about test cases for a report.
     * </p>
     *
     * @param describeTestCasesRequest
     * @return A Java Future containing the result of the DescribeTestCases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.DescribeTestCases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DescribeTestCases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTestCasesResponse> describeTestCases(DescribeTestCasesRequest describeTestCasesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTestCasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTestCasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTestCasesRequest, DescribeTestCasesResponse>()
                            .withOperationName("DescribeTestCases")
                            .withMarshaller(new DescribeTestCasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeTestCasesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a resource policy that is identified by its resource ARN.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.GetResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/GetResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy")
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResourcePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a
     * GitHub, GitHub Enterprise, or Bitbucket repository.
     * </p>
     *
     * @param importSourceCredentialsRequest
     * @return A Java Future containing the result of the ImportSourceCredentials operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>AccountLimitExceededException An AWS service limit was exceeded for the calling AWS account.</li>
     *         <li>ResourceAlreadyExistsException The specified AWS resource cannot be created, because an AWS resource
     *         with the same settings already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ImportSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ImportSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportSourceCredentialsResponse> importSourceCredentials(
            ImportSourceCredentialsRequest importSourceCredentialsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportSourceCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportSourceCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportSourceCredentialsRequest, ImportSourceCredentialsResponse>()
                            .withOperationName("ImportSourceCredentials")
                            .withMarshaller(new ImportSourceCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(importSourceCredentialsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resets the cache for a project.
     * </p>
     *
     * @param invalidateProjectCacheRequest
     * @return A Java Future containing the result of the InvalidateProjectCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.InvalidateProjectCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/InvalidateProjectCache"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InvalidateProjectCacheResponse> invalidateProjectCache(
            InvalidateProjectCacheRequest invalidateProjectCacheRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvalidateProjectCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InvalidateProjectCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InvalidateProjectCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InvalidateProjectCacheRequest, InvalidateProjectCacheResponse>()
                            .withOperationName("InvalidateProjectCache")
                            .withMarshaller(new InvalidateProjectCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(invalidateProjectCacheRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of build IDs, with each build ID representing a single build.
     * </p>
     *
     * @param listBuildsRequest
     * @return A Java Future containing the result of the ListBuilds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBuildsResponse> listBuilds(ListBuildsRequest listBuildsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBuildsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBuildsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBuildsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBuildsRequest, ListBuildsResponse>().withOperationName("ListBuilds")
                            .withMarshaller(new ListBuildsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listBuildsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of build IDs for the specified build project, with each build ID representing a single build.
     * </p>
     *
     * @param listBuildsForProjectRequest
     * @return A Java Future containing the result of the ListBuildsForProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListBuildsForProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuildsForProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBuildsForProjectResponse> listBuildsForProject(
            ListBuildsForProjectRequest listBuildsForProjectRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBuildsForProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBuildsForProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBuildsForProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBuildsForProjectRequest, ListBuildsForProjectResponse>()
                            .withOperationName("ListBuildsForProject")
                            .withMarshaller(new ListBuildsForProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listBuildsForProjectRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about Docker images that are managed by AWS CodeBuild.
     * </p>
     *
     * @param listCuratedEnvironmentImagesRequest
     * @return A Java Future containing the result of the ListCuratedEnvironmentImages operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListCuratedEnvironmentImages
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListCuratedEnvironmentImages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCuratedEnvironmentImagesResponse> listCuratedEnvironmentImages(
            ListCuratedEnvironmentImagesRequest listCuratedEnvironmentImagesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCuratedEnvironmentImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCuratedEnvironmentImagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCuratedEnvironmentImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCuratedEnvironmentImagesRequest, ListCuratedEnvironmentImagesResponse>()
                            .withOperationName("ListCuratedEnvironmentImages")
                            .withMarshaller(new ListCuratedEnvironmentImagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listCuratedEnvironmentImagesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of build project names, with each build project name representing a single build project.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listProjectsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list ARNs for the report groups in the current AWS account.
     * </p>
     *
     * @param listReportGroupsRequest
     * @return A Java Future containing the result of the ListReportGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListReportGroupsResponse> listReportGroups(ListReportGroupsRequest listReportGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReportGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReportGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReportGroupsRequest, ListReportGroupsResponse>()
                            .withOperationName("ListReportGroups")
                            .withMarshaller(new ListReportGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listReportGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports in the current AWS account.
     * </p>
     *
     * @param listReportsRequest
     * @return A Java Future containing the result of the ListReports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListReportsResponse> listReports(ListReportsRequest listReportsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReportsRequest, ListReportsResponse>()
                            .withOperationName("ListReports").withMarshaller(new ListReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listReportsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports that belong to a <code>ReportGroup</code>.
     * </p>
     *
     * @param listReportsForReportGroupRequest
     * @return A Java Future containing the result of the ListReportsForReportGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListReportsForReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportsForReportGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReportsForReportGroupResponse> listReportsForReportGroup(
            ListReportsForReportGroupRequest listReportsForReportGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReportsForReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReportsForReportGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReportsForReportGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReportsForReportGroupRequest, ListReportsForReportGroupResponse>()
                            .withOperationName("ListReportsForReportGroup")
                            .withMarshaller(new ListReportsForReportGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listReportsForReportGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of projects that are shared with other AWS accounts or users.
     * </p>
     *
     * @param listSharedProjectsRequest
     * @return A Java Future containing the result of the ListSharedProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListSharedProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedProjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSharedProjectsResponse> listSharedProjects(ListSharedProjectsRequest listSharedProjectsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSharedProjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSharedProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSharedProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSharedProjectsRequest, ListSharedProjectsResponse>()
                            .withOperationName("ListSharedProjects")
                            .withMarshaller(new ListSharedProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSharedProjectsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of report groups that are shared with other AWS accounts or users.
     * </p>
     *
     * @param listSharedReportGroupsRequest
     * @return A Java Future containing the result of the ListSharedReportGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListSharedReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedReportGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSharedReportGroupsResponse> listSharedReportGroups(
            ListSharedReportGroupsRequest listSharedReportGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSharedReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSharedReportGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSharedReportGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSharedReportGroupsRequest, ListSharedReportGroupsResponse>()
                            .withOperationName("ListSharedReportGroups")
                            .withMarshaller(new ListSharedReportGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSharedReportGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>SourceCredentialsInfo</code> objects.
     * </p>
     *
     * @param listSourceCredentialsRequest
     * @return A Java Future containing the result of the ListSourceCredentials operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.ListSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSourceCredentialsResponse> listSourceCredentials(
            ListSourceCredentialsRequest listSourceCredentialsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSourceCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSourceCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSourceCredentialsRequest, ListSourceCredentialsResponse>()
                            .withOperationName("ListSourceCredentials")
                            .withMarshaller(new ListSourceCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSourceCredentialsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stores a resource policy for the ARN of a <code>Project</code> or <code>ReportGroup</code> object.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.PutResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/PutResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy")
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putResourcePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts running a build.
     * </p>
     *
     * @param startBuildRequest
     * @return A Java Future containing the result of the StartBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>AccountLimitExceededException An AWS service limit was exceeded for the calling AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.StartBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/StartBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartBuildResponse> startBuild(StartBuildRequest startBuildRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartBuildRequest, StartBuildResponse>().withOperationName("StartBuild")
                            .withMarshaller(new StartBuildRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startBuildRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to stop running a build.
     * </p>
     *
     * @param stopBuildRequest
     * @return A Java Future containing the result of the StopBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.StopBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/StopBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopBuildResponse> stopBuild(StopBuildRequest stopBuildRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopBuildRequest, StopBuildResponse>().withOperationName("StopBuild")
                            .withMarshaller(new StopBuildRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(stopBuildRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the settings of a build project.
     * </p>
     *
     * @param updateProjectRequest
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.UpdateProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject")
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateProjectRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a report group.
     * </p>
     *
     * @param updateReportGroupRequest
     * @return A Java Future containing the result of the UpdateReportGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.UpdateReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReportGroupResponse> updateReportGroup(UpdateReportGroupRequest updateReportGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReportGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReportGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReportGroupRequest, UpdateReportGroupResponse>()
                            .withOperationName("UpdateReportGroup")
                            .withMarshaller(new UpdateReportGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateReportGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the webhook associated with an AWS CodeBuild build project.
     * </p>
     * <note>
     * <p>
     * If you use Bitbucket for your repository, <code>rotateSecret</code> is ignored.
     * </p>
     * </note>
     *
     * @param updateWebhookRequest
     * @return A Java Future containing the result of the UpdateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input value that was provided is not valid.</li>
     *         <li>ResourceNotFoundException The specified AWS resource cannot be found.</li>
     *         <li>OAuthProviderException There was a problem with the underlying OAuth provider.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeBuildException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeBuildAsyncClient.UpdateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWebhookResponse> updateWebhook(UpdateWebhookRequest updateWebhookRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWebhookRequest, UpdateWebhookResponse>()
                            .withOperationName("UpdateWebhook")
                            .withMarshaller(new UpdateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateWebhookRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeBuildException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OAuthProviderException")
                                .exceptionBuilderSupplier(OAuthProviderException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountLimitExceededException")
                                .exceptionBuilderSupplier(AccountLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
