/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an individual contributor to a multi-timeseries alarm, containing information about a specific time series
 * and its contribution to the alarm's state.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AlarmContributor implements SdkPojo, Serializable,
        ToCopyableBuilder<AlarmContributor.Builder, AlarmContributor> {
    private static final SdkField<String> CONTRIBUTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContributorId").getter(getter(AlarmContributor::contributorId)).setter(setter(Builder::contributorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContributorId").build()).build();

    private static final SdkField<Map<String, String>> CONTRIBUTOR_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ContributorAttributes")
            .getter(getter(AlarmContributor::contributorAttributes))
            .setter(setter(Builder::contributorAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContributorAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReason").getter(getter(AlarmContributor::stateReason)).setter(setter(Builder::stateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReason").build()).build();

    private static final SdkField<Instant> STATE_TRANSITIONED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StateTransitionedTimestamp")
            .getter(getter(AlarmContributor::stateTransitionedTimestamp))
            .setter(setter(Builder::stateTransitionedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateTransitionedTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTRIBUTOR_ID_FIELD,
            CONTRIBUTOR_ATTRIBUTES_FIELD, STATE_REASON_FIELD, STATE_TRANSITIONED_TIMESTAMP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String contributorId;

    private final Map<String, String> contributorAttributes;

    private final String stateReason;

    private final Instant stateTransitionedTimestamp;

    private AlarmContributor(BuilderImpl builder) {
        this.contributorId = builder.contributorId;
        this.contributorAttributes = builder.contributorAttributes;
        this.stateReason = builder.stateReason;
        this.stateTransitionedTimestamp = builder.stateTransitionedTimestamp;
    }

    /**
     * <p>
     * The unique identifier for this alarm contributor.
     * </p>
     * 
     * @return The unique identifier for this alarm contributor.
     */
    public final String contributorId() {
        return contributorId;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContributorAttributes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasContributorAttributes() {
        return contributorAttributes != null && !(contributorAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attributes that describe the contributor, such as metric dimensions and other identifying
     * characteristics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContributorAttributes} method.
     * </p>
     * 
     * @return A map of attributes that describe the contributor, such as metric dimensions and other identifying
     *         characteristics.
     */
    public final Map<String, String> contributorAttributes() {
        return contributorAttributes;
    }

    /**
     * <p>
     * An explanation for the contributor's current state, providing context about why it is in its current condition.
     * </p>
     * 
     * @return An explanation for the contributor's current state, providing context about why it is in its current
     *         condition.
     */
    public final String stateReason() {
        return stateReason;
    }

    /**
     * <p>
     * The timestamp when the contributor last transitioned to its current state.
     * </p>
     * 
     * @return The timestamp when the contributor last transitioned to its current state.
     */
    public final Instant stateTransitionedTimestamp() {
        return stateTransitionedTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(contributorId());
        hashCode = 31 * hashCode + Objects.hashCode(hasContributorAttributes() ? contributorAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        hashCode = 31 * hashCode + Objects.hashCode(stateTransitionedTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AlarmContributor)) {
            return false;
        }
        AlarmContributor other = (AlarmContributor) obj;
        return Objects.equals(contributorId(), other.contributorId())
                && hasContributorAttributes() == other.hasContributorAttributes()
                && Objects.equals(contributorAttributes(), other.contributorAttributes())
                && Objects.equals(stateReason(), other.stateReason())
                && Objects.equals(stateTransitionedTimestamp(), other.stateTransitionedTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AlarmContributor").add("ContributorId", contributorId())
                .add("ContributorAttributes", hasContributorAttributes() ? contributorAttributes() : null)
                .add("StateReason", stateReason()).add("StateTransitionedTimestamp", stateTransitionedTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContributorId":
            return Optional.ofNullable(clazz.cast(contributorId()));
        case "ContributorAttributes":
            return Optional.ofNullable(clazz.cast(contributorAttributes()));
        case "StateReason":
            return Optional.ofNullable(clazz.cast(stateReason()));
        case "StateTransitionedTimestamp":
            return Optional.ofNullable(clazz.cast(stateTransitionedTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ContributorId", CONTRIBUTOR_ID_FIELD);
        map.put("ContributorAttributes", CONTRIBUTOR_ATTRIBUTES_FIELD);
        map.put("StateReason", STATE_REASON_FIELD);
        map.put("StateTransitionedTimestamp", STATE_TRANSITIONED_TIMESTAMP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AlarmContributor, T> g) {
        return obj -> g.apply((AlarmContributor) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AlarmContributor> {
        /**
         * <p>
         * The unique identifier for this alarm contributor.
         * </p>
         * 
         * @param contributorId
         *        The unique identifier for this alarm contributor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contributorId(String contributorId);

        /**
         * <p>
         * A map of attributes that describe the contributor, such as metric dimensions and other identifying
         * characteristics.
         * </p>
         * 
         * @param contributorAttributes
         *        A map of attributes that describe the contributor, such as metric dimensions and other identifying
         *        characteristics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contributorAttributes(Map<String, String> contributorAttributes);

        /**
         * <p>
         * An explanation for the contributor's current state, providing context about why it is in its current
         * condition.
         * </p>
         * 
         * @param stateReason
         *        An explanation for the contributor's current state, providing context about why it is in its current
         *        condition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);

        /**
         * <p>
         * The timestamp when the contributor last transitioned to its current state.
         * </p>
         * 
         * @param stateTransitionedTimestamp
         *        The timestamp when the contributor last transitioned to its current state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateTransitionedTimestamp(Instant stateTransitionedTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String contributorId;

        private Map<String, String> contributorAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String stateReason;

        private Instant stateTransitionedTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(AlarmContributor model) {
            contributorId(model.contributorId);
            contributorAttributes(model.contributorAttributes);
            stateReason(model.stateReason);
            stateTransitionedTimestamp(model.stateTransitionedTimestamp);
        }

        public final String getContributorId() {
            return contributorId;
        }

        public final void setContributorId(String contributorId) {
            this.contributorId = contributorId;
        }

        @Override
        public final Builder contributorId(String contributorId) {
            this.contributorId = contributorId;
            return this;
        }

        public final Map<String, String> getContributorAttributes() {
            if (contributorAttributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return contributorAttributes;
        }

        public final void setContributorAttributes(Map<String, String> contributorAttributes) {
            this.contributorAttributes = ContributorAttributesCopier.copy(contributorAttributes);
        }

        @Override
        public final Builder contributorAttributes(Map<String, String> contributorAttributes) {
            this.contributorAttributes = ContributorAttributesCopier.copy(contributorAttributes);
            return this;
        }

        public final String getStateReason() {
            return stateReason;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        @Override
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        public final Instant getStateTransitionedTimestamp() {
            return stateTransitionedTimestamp;
        }

        public final void setStateTransitionedTimestamp(Instant stateTransitionedTimestamp) {
            this.stateTransitionedTimestamp = stateTransitionedTimestamp;
        }

        @Override
        public final Builder stateTransitionedTimestamp(Instant stateTransitionedTimestamp) {
            this.stateTransitionedTimestamp = stateTransitionedTimestamp;
            return this;
        }

        @Override
        public AlarmContributor build() {
            return new AlarmContributor(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
