/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtraildata;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudtraildata.model.ChannelInsufficientPermissionException;
import software.amazon.awssdk.services.cloudtraildata.model.ChannelNotFoundException;
import software.amazon.awssdk.services.cloudtraildata.model.ChannelUnsupportedSchemaException;
import software.amazon.awssdk.services.cloudtraildata.model.CloudTrailDataException;
import software.amazon.awssdk.services.cloudtraildata.model.DuplicatedAuditEventIdException;
import software.amazon.awssdk.services.cloudtraildata.model.InvalidChannelArnException;
import software.amazon.awssdk.services.cloudtraildata.model.PutAuditEventsRequest;
import software.amazon.awssdk.services.cloudtraildata.model.PutAuditEventsResponse;
import software.amazon.awssdk.services.cloudtraildata.model.UnsupportedOperationException;
import software.amazon.awssdk.services.cloudtraildata.transform.PutAuditEventsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudTrailDataAsyncClient}.
 *
 * @see CloudTrailDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudTrailDataAsyncClient implements CloudTrailDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudTrailDataAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CloudTrailDataServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCloudTrailDataAsyncClient(CloudTrailDataServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Ingests your application events into CloudTrail Lake. A required parameter, <code>auditEvents</code>, accepts the
     * JSON records (also called <i>payload</i>) of events that you want CloudTrail to ingest. You can add up to 100 of
     * these events (or up to 1 MB) per <code>PutAuditEvents</code> request.
     * </p>
     *
     * @param putAuditEventsRequest
     * @return A Java Future containing the result of the PutAuditEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ChannelInsufficientPermissionException The caller's account ID must be the same as the channel
     *         owner's account ID.</li>
     *         <li>ChannelNotFoundException The channel could not be found.</li>
     *         <li>InvalidChannelArnException The specified channel ARN is not a valid channel ARN.</li>
     *         <li>ChannelUnsupportedSchemaException The schema type of the event is not supported.</li>
     *         <li>DuplicatedAuditEventIdException Two or more entries in the request have the same event ID.</li>
     *         <li>UnsupportedOperationException The operation requested is not supported in this region or account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailDataException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudTrailDataAsyncClient.PutAuditEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-data-2021-08-11/PutAuditEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAuditEventsResponse> putAuditEvents(PutAuditEventsRequest putAuditEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAuditEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAuditEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAuditEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAuditEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAuditEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAuditEventsRequest, PutAuditEventsResponse>()
                            .withOperationName("PutAuditEvents")
                            .withMarshaller(new PutAuditEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putAuditEventsRequest));
            CompletableFuture<PutAuditEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CloudTrailDataServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudTrailDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelUnsupportedSchema")
                                .exceptionBuilderSupplier(ChannelUnsupportedSchemaException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelNotFound")
                                .exceptionBuilderSupplier(ChannelNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicatedAuditEventId")
                                .exceptionBuilderSupplier(DuplicatedAuditEventIdException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelInsufficientPermission")
                                .exceptionBuilderSupplier(ChannelInsufficientPermissionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidChannelARN")
                                .exceptionBuilderSupplier(InvalidChannelArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
