/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that specifies the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Whether you want viewers to use HTTP or HTTPS to request your objects.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you want viewers to use HTTPS, whether you're using an alternate domain name such as <code>example.com</code> or
 * the CloudFront domain name for your distribution, such as <code>d111111abcdef8.cloudfront.net</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you're using an alternate domain name, whether AWS Certificate Manager (ACM) provided the certificate, or you
 * purchased a certificate from a third-party certificate authority and imported it into ACM or uploaded it to the IAM
 * certificate store.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Specify only one of the following values:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
 * >ACMCertificateArn</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
 * >IAMCertificateId</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
 * >CloudFrontDefaultCertificate</a>
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/SecureConnections.html#CNAMEsAndHTTPS">
 * Using Alternate Domain Names and HTTPS</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ViewerCertificate implements SdkPojo, Serializable,
        ToCopyableBuilder<ViewerCertificate.Builder, ViewerCertificate> {
    private static final SdkField<Boolean> CLOUD_FRONT_DEFAULT_CERTIFICATE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ViewerCertificate::cloudFrontDefaultCertificate))
            .setter(setter(Builder::cloudFrontDefaultCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudFrontDefaultCertificate")
                    .unmarshallLocationName("CloudFrontDefaultCertificate").build()).build();

    private static final SdkField<String> IAM_CERTIFICATE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::iamCertificateId))
            .setter(setter(Builder::iamCertificateId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMCertificateId")
                    .unmarshallLocationName("IAMCertificateId").build()).build();

    private static final SdkField<String> ACM_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::acmCertificateArn))
            .setter(setter(Builder::acmCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ACMCertificateArn")
                    .unmarshallLocationName("ACMCertificateArn").build()).build();

    private static final SdkField<String> SSL_SUPPORT_METHOD_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::sslSupportMethodAsString))
            .setter(setter(Builder::sslSupportMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSLSupportMethod")
                    .unmarshallLocationName("SSLSupportMethod").build()).build();

    private static final SdkField<String> MINIMUM_PROTOCOL_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::minimumProtocolVersionAsString))
            .setter(setter(Builder::minimumProtocolVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumProtocolVersion")
                    .unmarshallLocationName("MinimumProtocolVersion").build()).build();

    private static final SdkField<String> CERTIFICATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::certificate))
            .setter(setter(Builder::certificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Certificate")
                    .unmarshallLocationName("Certificate").build()).build();

    private static final SdkField<String> CERTIFICATE_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ViewerCertificate::certificateSourceAsString))
            .setter(setter(Builder::certificateSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateSource")
                    .unmarshallLocationName("CertificateSource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CLOUD_FRONT_DEFAULT_CERTIFICATE_FIELD, IAM_CERTIFICATE_ID_FIELD, ACM_CERTIFICATE_ARN_FIELD, SSL_SUPPORT_METHOD_FIELD,
            MINIMUM_PROTOCOL_VERSION_FIELD, CERTIFICATE_FIELD, CERTIFICATE_SOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean cloudFrontDefaultCertificate;

    private final String iamCertificateId;

    private final String acmCertificateArn;

    private final String sslSupportMethod;

    private final String minimumProtocolVersion;

    private final String certificate;

    private final String certificateSource;

    private ViewerCertificate(BuilderImpl builder) {
        this.cloudFrontDefaultCertificate = builder.cloudFrontDefaultCertificate;
        this.iamCertificateId = builder.iamCertificateId;
        this.acmCertificateArn = builder.acmCertificateArn;
        this.sslSupportMethod = builder.sslSupportMethod;
        this.minimumProtocolVersion = builder.minimumProtocolVersion;
        this.certificate = builder.certificate;
        this.certificateSource = builder.certificateSource;
    }

    /**
     * <p>
     * If you're using the CloudFront domain name for your distribution, such as
     * <code>d111111abcdef8.cloudfront.net</code>, specify the following value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt; </code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return If you're using the CloudFront domain name for your distribution, such as
     *         <code>d111111abcdef8.cloudfront.net</code>, specify the following value:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt; </code>
     *         </p>
     *         </li>
     */
    public Boolean cloudFrontDefaultCertificate() {
        return cloudFrontDefaultCertificate;
    }

    /**
     * <p>
     * If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you must
     * choose the type of certificate that you want to use. Specify the following value if you purchased your
     * certificate from a third-party certificate authority:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>&lt;IAMCertificateId&gt;<i>IAM certificate ID</i>&lt;IAMCertificateId&gt;</code> where
     * <code> <i>IAM certificate ID</i> </code> is the ID that IAM returned when you added the certificate to the IAM
     * certificate store.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify <code>IAMCertificateId</code>, you must also specify a value for <code>SSLSupportMethod</code>.
     * </p>
     * 
     * @return If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you
     *         must choose the type of certificate that you want to use. Specify the following value if you purchased
     *         your certificate from a third-party certificate authority:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>&lt;IAMCertificateId&gt;<i>IAM certificate ID</i>&lt;IAMCertificateId&gt;</code> where
     *         <code> <i>IAM certificate ID</i> </code> is the ID that IAM returned when you added the certificate to
     *         the IAM certificate store.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify <code>IAMCertificateId</code>, you must also specify a value for
     *         <code>SSLSupportMethod</code>.
     */
    public String iamCertificateId() {
        return iamCertificateId;
    }

    /**
     * <p>
     * If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you must
     * choose the type of certificate that you want to use. Specify the following value if ACM provided your
     * certificate:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>&lt;ACMCertificateArn&gt;<i>ARN for ACM SSL/TLS certificate</i>&lt;ACMCertificateArn&gt;</code> where
     * <code> <i>ARN for ACM SSL/TLS certificate</i> </code> is the ARN for the ACM SSL/TLS certificate that you want to
     * use for this distribution.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify <code>ACMCertificateArn</code>, you must also specify a value for <code>SSLSupportMethod</code>.
     * </p>
     * 
     * @return If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you
     *         must choose the type of certificate that you want to use. Specify the following value if ACM provided
     *         your certificate:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>&lt;ACMCertificateArn&gt;<i>ARN for ACM SSL/TLS certificate</i>&lt;ACMCertificateArn&gt;</code>
     *         where <code> <i>ARN for ACM SSL/TLS certificate</i> </code> is the ARN for the ACM SSL/TLS certificate
     *         that you want to use for this distribution.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify <code>ACMCertificateArn</code>, you must also specify a value for
     *         <code>SSLSupportMethod</code>.
     */
    public String acmCertificateArn() {
        return acmCertificateArn;
    }

    /**
     * <p>
     * If you specify a value for <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     * >ACMCertificateArn</a> or for <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     * >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a method that
     * works for browsers and clients released after 2010 or one that works for all clients.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name Indication
     * (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of the browsers that
     * support SNI, see the <a href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name
     * Indication</a>. To learn about options to explore if you have users with browsers that don't include SNI support,
     * see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS requests from
     * any viewer. However, there are additional monthly charges. For details, including specific pricing information,
     * see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon CloudFront</a>
     * on the AWS marketing site.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Don't specify a value for <code>SSLSupportMethod</code> if you specified
     * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslSupportMethod}
     * will return {@link SSLSupportMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sslSupportMethodAsString}.
     * </p>
     * 
     * @return If you specify a value for <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     *         >ACMCertificateArn</a> or for <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     *         >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a
     *         method that works for browsers and clients released after 2010 or one that works for all clients.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
     *         Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of
     *         the browsers that support SNI, see the <a
     *         href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name Indication</a>. To
     *         learn about options to explore if you have users with browsers that don't include SNI support, see <a
     *         href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     *         >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS
     *         requests from any viewer. However, there are additional monthly charges. For details, including specific
     *         pricing information, see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL
     *         options for Amazon CloudFront</a> on the AWS marketing site.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Don't specify a value for <code>SSLSupportMethod</code> if you specified
     *         <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     *         >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * @see SSLSupportMethod
     */
    public SSLSupportMethod sslSupportMethod() {
        return SSLSupportMethod.fromValue(sslSupportMethod);
    }

    /**
     * <p>
     * If you specify a value for <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     * >ACMCertificateArn</a> or for <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     * >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a method that
     * works for browsers and clients released after 2010 or one that works for all clients.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name Indication
     * (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of the browsers that
     * support SNI, see the <a href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name
     * Indication</a>. To learn about options to explore if you have users with browsers that don't include SNI support,
     * see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS requests from
     * any viewer. However, there are additional monthly charges. For details, including specific pricing information,
     * see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon CloudFront</a>
     * on the AWS marketing site.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Don't specify a value for <code>SSLSupportMethod</code> if you specified
     * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslSupportMethod}
     * will return {@link SSLSupportMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sslSupportMethodAsString}.
     * </p>
     * 
     * @return If you specify a value for <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     *         >ACMCertificateArn</a> or for <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     *         >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a
     *         method that works for browsers and clients released after 2010 or one that works for all clients.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
     *         Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of
     *         the browsers that support SNI, see the <a
     *         href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name Indication</a>. To
     *         learn about options to explore if you have users with browsers that don't include SNI support, see <a
     *         href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     *         >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS
     *         requests from any viewer. However, there are additional monthly charges. For details, including specific
     *         pricing information, see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL
     *         options for Amazon CloudFront</a> on the AWS marketing site.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Don't specify a value for <code>SSLSupportMethod</code> if you specified
     *         <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
     *         >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * @see SSLSupportMethod
     */
    public String sslSupportMethodAsString() {
        return sslSupportMethod;
    }

    /**
     * <p>
     * Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy determines
     * two settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
     * </p>
     * </li>
     * <li>
     * <p>
     * The cipher that CloudFront uses to encrypt the content that it returns to viewers
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * On the CloudFront console, this setting is called <b>Security policy</b>.
     * </p>
     * </note>
     * <p>
     * We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices that do
     * not support TLSv1.1 or later.
     * </p>
     * <p>
     * When both of the following are true, you must specify <code>TLSv1</code> or later for the security policy:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
     * <code>IAMCertificateId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically sets the
     * security policy to <code>TLSv1</code> regardless of the value that you specify for
     * <code>MinimumProtocolVersion</code>.
     * </p>
     * <p>
     * For information about the relationship between the security policy that you choose and the protocols and ciphers
     * that CloudFront uses to communicate with viewers, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     * > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the <i>Amazon
     * CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #minimumProtocolVersion} will return {@link MinimumProtocolVersion#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #minimumProtocolVersionAsString}.
     * </p>
     * 
     * @return Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
     *         determines two settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cipher that CloudFront uses to encrypt the content that it returns to viewers
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         On the CloudFront console, this setting is called <b>Security policy</b>.
     *         </p>
     *         </note>
     *         <p>
     *         We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices
     *         that do not support TLSv1.1 or later.
     *         </p>
     *         <p>
     *         When both of the following are true, you must specify <code>TLSv1</code> or later for the security
     *         policy:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
     *         <code>IAMCertificateId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically
     *         sets the security policy to <code>TLSv1</code> regardless of the value that you specify for
     *         <code>MinimumProtocolVersion</code>.
     *         </p>
     *         <p>
     *         For information about the relationship between the security policy that you choose and the protocols and
     *         ciphers that CloudFront uses to communicate with viewers, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     *         > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
     *         <i>Amazon CloudFront Developer Guide</i>.
     * @see MinimumProtocolVersion
     */
    public MinimumProtocolVersion minimumProtocolVersion() {
        return MinimumProtocolVersion.fromValue(minimumProtocolVersion);
    }

    /**
     * <p>
     * Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy determines
     * two settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
     * </p>
     * </li>
     * <li>
     * <p>
     * The cipher that CloudFront uses to encrypt the content that it returns to viewers
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * On the CloudFront console, this setting is called <b>Security policy</b>.
     * </p>
     * </note>
     * <p>
     * We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices that do
     * not support TLSv1.1 or later.
     * </p>
     * <p>
     * When both of the following are true, you must specify <code>TLSv1</code> or later for the security policy:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
     * <code>IAMCertificateId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically sets the
     * security policy to <code>TLSv1</code> regardless of the value that you specify for
     * <code>MinimumProtocolVersion</code>.
     * </p>
     * <p>
     * For information about the relationship between the security policy that you choose and the protocols and ciphers
     * that CloudFront uses to communicate with viewers, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     * > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the <i>Amazon
     * CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #minimumProtocolVersion} will return {@link MinimumProtocolVersion#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #minimumProtocolVersionAsString}.
     * </p>
     * 
     * @return Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
     *         determines two settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cipher that CloudFront uses to encrypt the content that it returns to viewers
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         On the CloudFront console, this setting is called <b>Security policy</b>.
     *         </p>
     *         </note>
     *         <p>
     *         We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices
     *         that do not support TLSv1.1 or later.
     *         </p>
     *         <p>
     *         When both of the following are true, you must specify <code>TLSv1</code> or later for the security
     *         policy:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
     *         <code>IAMCertificateId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically
     *         sets the security policy to <code>TLSv1</code> regardless of the value that you specify for
     *         <code>MinimumProtocolVersion</code>.
     *         </p>
     *         <p>
     *         For information about the relationship between the security policy that you choose and the protocols and
     *         ciphers that CloudFront uses to communicate with viewers, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     *         > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
     *         <i>Amazon CloudFront Developer Guide</i>.
     * @see MinimumProtocolVersion
     */
    public String minimumProtocolVersionAsString() {
        return minimumProtocolVersion;
    }

    /**
     * <p>
     * This field is no longer used. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     * >ACMCertificateArn</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     * >IAMCertificateId</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     * >CloudFrontDefaultCertificate</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return This field is no longer used. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     *         >ACMCertificateArn</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     *         >IAMCertificateId</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     *         >CloudFrontDefaultCertificate</a>
     *         </p>
     *         </li>
     */
    public String certificate() {
        return certificate;
    }

    /**
     * <p>
     * This field is no longer used. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     * >ACMCertificateArn</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     * >IAMCertificateId</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     * >CloudFrontDefaultCertificate</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateSource}
     * will return {@link CertificateSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #certificateSourceAsString}.
     * </p>
     * 
     * @return This field is no longer used. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     *         >ACMCertificateArn</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     *         >IAMCertificateId</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     *         >CloudFrontDefaultCertificate</a>
     *         </p>
     *         </li>
     * @see CertificateSource
     */
    public CertificateSource certificateSource() {
        return CertificateSource.fromValue(certificateSource);
    }

    /**
     * <p>
     * This field is no longer used. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     * >ACMCertificateArn</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     * >IAMCertificateId</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     * >CloudFrontDefaultCertificate</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateSource}
     * will return {@link CertificateSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #certificateSourceAsString}.
     * </p>
     * 
     * @return This field is no longer used. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
     *         >ACMCertificateArn</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
     *         >IAMCertificateId</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
     *         >CloudFrontDefaultCertificate</a>
     *         </p>
     *         </li>
     * @see CertificateSource
     */
    public String certificateSourceAsString() {
        return certificateSource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cloudFrontDefaultCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(iamCertificateId());
        hashCode = 31 * hashCode + Objects.hashCode(acmCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(sslSupportMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(minimumProtocolVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(certificateSourceAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ViewerCertificate)) {
            return false;
        }
        ViewerCertificate other = (ViewerCertificate) obj;
        return Objects.equals(cloudFrontDefaultCertificate(), other.cloudFrontDefaultCertificate())
                && Objects.equals(iamCertificateId(), other.iamCertificateId())
                && Objects.equals(acmCertificateArn(), other.acmCertificateArn())
                && Objects.equals(sslSupportMethodAsString(), other.sslSupportMethodAsString())
                && Objects.equals(minimumProtocolVersionAsString(), other.minimumProtocolVersionAsString())
                && Objects.equals(certificate(), other.certificate())
                && Objects.equals(certificateSourceAsString(), other.certificateSourceAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ViewerCertificate").add("CloudFrontDefaultCertificate", cloudFrontDefaultCertificate())
                .add("IAMCertificateId", iamCertificateId()).add("ACMCertificateArn", acmCertificateArn())
                .add("SSLSupportMethod", sslSupportMethodAsString())
                .add("MinimumProtocolVersion", minimumProtocolVersionAsString()).add("Certificate", certificate())
                .add("CertificateSource", certificateSourceAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CloudFrontDefaultCertificate":
            return Optional.ofNullable(clazz.cast(cloudFrontDefaultCertificate()));
        case "IAMCertificateId":
            return Optional.ofNullable(clazz.cast(iamCertificateId()));
        case "ACMCertificateArn":
            return Optional.ofNullable(clazz.cast(acmCertificateArn()));
        case "SSLSupportMethod":
            return Optional.ofNullable(clazz.cast(sslSupportMethodAsString()));
        case "MinimumProtocolVersion":
            return Optional.ofNullable(clazz.cast(minimumProtocolVersionAsString()));
        case "Certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "CertificateSource":
            return Optional.ofNullable(clazz.cast(certificateSourceAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ViewerCertificate, T> g) {
        return obj -> g.apply((ViewerCertificate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ViewerCertificate> {
        /**
         * <p>
         * If you're using the CloudFront domain name for your distribution, such as
         * <code>d111111abcdef8.cloudfront.net</code>, specify the following value:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt; </code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param cloudFrontDefaultCertificate
         *        If you're using the CloudFront domain name for your distribution, such as
         *        <code>d111111abcdef8.cloudfront.net</code>, specify the following value:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt; </code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate);

        /**
         * <p>
         * If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you must
         * choose the type of certificate that you want to use. Specify the following value if you purchased your
         * certificate from a third-party certificate authority:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>&lt;IAMCertificateId&gt;<i>IAM certificate ID</i>&lt;IAMCertificateId&gt;</code> where
         * <code> <i>IAM certificate ID</i> </code> is the ID that IAM returned when you added the certificate to the
         * IAM certificate store.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify <code>IAMCertificateId</code>, you must also specify a value for <code>SSLSupportMethod</code>
         * .
         * </p>
         * 
         * @param iamCertificateId
         *        If you want viewers to use HTTPS to request your objects and you're using an alternate domain name,
         *        you must choose the type of certificate that you want to use. Specify the following value if you
         *        purchased your certificate from a third-party certificate authority:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>&lt;IAMCertificateId&gt;<i>IAM certificate ID</i>&lt;IAMCertificateId&gt;</code> where
         *        <code> <i>IAM certificate ID</i> </code> is the ID that IAM returned when you added the certificate to
         *        the IAM certificate store.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify <code>IAMCertificateId</code>, you must also specify a value for
         *        <code>SSLSupportMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamCertificateId(String iamCertificateId);

        /**
         * <p>
         * If you want viewers to use HTTPS to request your objects and you're using an alternate domain name, you must
         * choose the type of certificate that you want to use. Specify the following value if ACM provided your
         * certificate:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>&lt;ACMCertificateArn&gt;<i>ARN for ACM SSL/TLS certificate</i>&lt;ACMCertificateArn&gt;</code> where
         * <code> <i>ARN for ACM SSL/TLS certificate</i> </code> is the ARN for the ACM SSL/TLS certificate that you
         * want to use for this distribution.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify <code>ACMCertificateArn</code>, you must also specify a value for
         * <code>SSLSupportMethod</code>.
         * </p>
         * 
         * @param acmCertificateArn
         *        If you want viewers to use HTTPS to request your objects and you're using an alternate domain name,
         *        you must choose the type of certificate that you want to use. Specify the following value if ACM
         *        provided your certificate:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>&lt;ACMCertificateArn&gt;<i>ARN for ACM SSL/TLS certificate</i>&lt;ACMCertificateArn&gt;</code>
         *        where <code> <i>ARN for ACM SSL/TLS certificate</i> </code> is the ARN for the ACM SSL/TLS certificate
         *        that you want to use for this distribution.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify <code>ACMCertificateArn</code>, you must also specify a value for
         *        <code>SSLSupportMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acmCertificateArn(String acmCertificateArn);

        /**
         * <p>
         * If you specify a value for <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         * >ACMCertificateArn</a> or for <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         * >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a method
         * that works for browsers and clients released after 2010 or one that works for all clients.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
         * Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of the
         * browsers that support SNI, see the <a href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia
         * entry Server Name Indication</a>. To learn about options to explore if you have users with browsers that
         * don't include SNI support, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS requests
         * from any viewer. However, there are additional monthly charges. For details, including specific pricing
         * information, see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon
         * CloudFront</a> on the AWS marketing site.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Don't specify a value for <code>SSLSupportMethod</code> if you specified
         * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param sslSupportMethod
         *        If you specify a value for <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         *        >ACMCertificateArn</a> or for <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         *        >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a
         *        method that works for browsers and clients released after 2010 or one that works for all clients.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
         *        Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list
         *        of the browsers that support SNI, see the <a
         *        href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name Indication</a>.
         *        To learn about options to explore if you have users with browsers that don't include SNI support, see
         *        <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         *        >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS
         *        requests from any viewer. However, there are additional monthly charges. For details, including
         *        specific pricing information, see <a
         *        href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon
         *        CloudFront</a> on the AWS marketing site.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Don't specify a value for <code>SSLSupportMethod</code> if you specified
         *        <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         *        >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @see SSLSupportMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLSupportMethod
         */
        Builder sslSupportMethod(String sslSupportMethod);

        /**
         * <p>
         * If you specify a value for <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         * >ACMCertificateArn</a> or for <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         * >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a method
         * that works for browsers and clients released after 2010 or one that works for all clients.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
         * Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list of the
         * browsers that support SNI, see the <a href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia
         * entry Server Name Indication</a>. To learn about options to explore if you have users with browsers that
         * don't include SNI support, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS requests
         * from any viewer. However, there are additional monthly charges. For details, including specific pricing
         * information, see <a href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon
         * CloudFront</a> on the AWS marketing site.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Don't specify a value for <code>SSLSupportMethod</code> if you specified
         * <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         * >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param sslSupportMethod
         *        If you specify a value for <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         *        >ACMCertificateArn</a> or for <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         *        >IAMCertificateId</a>, you must also specify how you want CloudFront to serve HTTPS requests: using a
         *        method that works for browsers and clients released after 2010 or one that works for all clients.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>sni-only</code>: CloudFront can respond to HTTPS requests from viewers that support Server Name
         *        Indication (SNI). All modern browsers support SNI, but there are a few that don't. For a current list
         *        of the browsers that support SNI, see the <a
         *        href="http://en.wikipedia.org/wiki/Server_Name_Indication">Wikipedia entry Server Name Indication</a>.
         *        To learn about options to explore if you have users with browsers that don't include SNI support, see
         *        <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         *        >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vip</code>: CloudFront uses dedicated IP addresses for your content and can respond to HTTPS
         *        requests from any viewer. However, there are additional monthly charges. For details, including
         *        specific pricing information, see <a
         *        href="http://aws.amazon.com/cloudfront/custom-ssl-domains/">Custom SSL options for Amazon
         *        CloudFront</a> on the AWS marketing site.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Don't specify a value for <code>SSLSupportMethod</code> if you specified
         *        <code>&lt;CloudFrontDefaultCertificate&gt;true&lt;CloudFrontDefaultCertificate&gt;</code>.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-https-dedicated-ip-or-sni.html"
         *        >Choosing How CloudFront Serves HTTPS Requests</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @see SSLSupportMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLSupportMethod
         */
        Builder sslSupportMethod(SSLSupportMethod sslSupportMethod);

        /**
         * <p>
         * Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
         * determines two settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
         * </p>
         * </li>
         * <li>
         * <p>
         * The cipher that CloudFront uses to encrypt the content that it returns to viewers
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * On the CloudFront console, this setting is called <b>Security policy</b>.
         * </p>
         * </note>
         * <p>
         * We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices that
         * do not support TLSv1.1 or later.
         * </p>
         * <p>
         * When both of the following are true, you must specify <code>TLSv1</code> or later for the security policy:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
         * <code>IAMCertificateId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically sets
         * the security policy to <code>TLSv1</code> regardless of the value that you specify for
         * <code>MinimumProtocolVersion</code>.
         * </p>
         * <p>
         * For information about the relationship between the security policy that you choose and the protocols and
         * ciphers that CloudFront uses to communicate with viewers, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         * > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
         * <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param minimumProtocolVersion
         *        Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
         *        determines two settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The cipher that CloudFront uses to encrypt the content that it returns to viewers
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        On the CloudFront console, this setting is called <b>Security policy</b>.
         *        </p>
         *        </note>
         *        <p>
         *        We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or
         *        devices that do not support TLSv1.1 or later.
         *        </p>
         *        <p>
         *        When both of the following are true, you must specify <code>TLSv1</code> or later for the security
         *        policy:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
         *        <code>IAMCertificateId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront
         *        automatically sets the security policy to <code>TLSv1</code> regardless of the value that you specify
         *        for <code>MinimumProtocolVersion</code>.
         *        </p>
         *        <p>
         *        For information about the relationship between the security policy that you choose and the protocols
         *        and ciphers that CloudFront uses to communicate with viewers, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         *        > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
         *        <i>Amazon CloudFront Developer Guide</i>.
         * @see MinimumProtocolVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MinimumProtocolVersion
         */
        Builder minimumProtocolVersion(String minimumProtocolVersion);

        /**
         * <p>
         * Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
         * determines two settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
         * </p>
         * </li>
         * <li>
         * <p>
         * The cipher that CloudFront uses to encrypt the content that it returns to viewers
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * On the CloudFront console, this setting is called <b>Security policy</b>.
         * </p>
         * </note>
         * <p>
         * We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or devices that
         * do not support TLSv1.1 or later.
         * </p>
         * <p>
         * When both of the following are true, you must specify <code>TLSv1</code> or later for the security policy:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
         * <code>IAMCertificateId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront automatically sets
         * the security policy to <code>TLSv1</code> regardless of the value that you specify for
         * <code>MinimumProtocolVersion</code>.
         * </p>
         * <p>
         * For information about the relationship between the security policy that you choose and the protocols and
         * ciphers that CloudFront uses to communicate with viewers, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         * > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
         * <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param minimumProtocolVersion
         *        Specify the security policy that you want CloudFront to use for HTTPS connections. A security policy
         *        determines two settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The minimum SSL/TLS protocol that CloudFront uses to communicate with viewers
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The cipher that CloudFront uses to encrypt the content that it returns to viewers
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        On the CloudFront console, this setting is called <b>Security policy</b>.
         *        </p>
         *        </note>
         *        <p>
         *        We recommend that you specify <code>TLSv1.1_2016</code> unless your users are using browsers or
         *        devices that do not support TLSv1.1 or later.
         *        </p>
         *        <p>
         *        When both of the following are true, you must specify <code>TLSv1</code> or later for the security
         *        policy:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You're using a custom certificate: you specified a value for <code>ACMCertificateArn</code> or for
         *        <code>IAMCertificateId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You're using SNI: you specified <code>sni-only</code> for <code>SSLSupportMethod</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify <code>true</code> for <code>CloudFrontDefaultCertificate</code>, CloudFront
         *        automatically sets the security policy to <code>TLSv1</code> regardless of the value that you specify
         *        for <code>MinimumProtocolVersion</code>.
         *        </p>
         *        <p>
         *        For information about the relationship between the security policy that you choose and the protocols
         *        and ciphers that CloudFront uses to communicate with viewers, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         *        > Supported SSL/TLS Protocols and Ciphers for Communication Between Viewers and CloudFront</a> in the
         *        <i>Amazon CloudFront Developer Guide</i>.
         * @see MinimumProtocolVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MinimumProtocolVersion
         */
        Builder minimumProtocolVersion(MinimumProtocolVersion minimumProtocolVersion);

        /**
         * <p>
         * This field is no longer used. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         * >ACMCertificateArn</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         * >IAMCertificateId</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         * >CloudFrontDefaultCertificate</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificate
         *        This field is no longer used. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         *        >ACMCertificateArn</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         *        >IAMCertificateId</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         *        >CloudFrontDefaultCertificate</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(String certificate);

        /**
         * <p>
         * This field is no longer used. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         * >ACMCertificateArn</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         * >IAMCertificateId</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         * >CloudFrontDefaultCertificate</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificateSource
         *        This field is no longer used. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         *        >ACMCertificateArn</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         *        >IAMCertificateId</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         *        >CloudFrontDefaultCertificate</a>
         *        </p>
         *        </li>
         * @see CertificateSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateSource
         */
        Builder certificateSource(String certificateSource);

        /**
         * <p>
         * This field is no longer used. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         * >ACMCertificateArn</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         * >IAMCertificateId</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         * >CloudFrontDefaultCertificate</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificateSource
         *        This field is no longer used. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-ACMCertificateArn"
         *        >ACMCertificateArn</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-IAMCertificateId"
         *        >IAMCertificateId</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html#cloudfront-Type-ViewerCertificate-CloudFrontDefaultCertificate"
         *        >CloudFrontDefaultCertificate</a>
         *        </p>
         *        </li>
         * @see CertificateSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateSource
         */
        Builder certificateSource(CertificateSource certificateSource);
    }

    static final class BuilderImpl implements Builder {
        private Boolean cloudFrontDefaultCertificate;

        private String iamCertificateId;

        private String acmCertificateArn;

        private String sslSupportMethod;

        private String minimumProtocolVersion;

        private String certificate;

        private String certificateSource;

        private BuilderImpl() {
        }

        private BuilderImpl(ViewerCertificate model) {
            cloudFrontDefaultCertificate(model.cloudFrontDefaultCertificate);
            iamCertificateId(model.iamCertificateId);
            acmCertificateArn(model.acmCertificateArn);
            sslSupportMethod(model.sslSupportMethod);
            minimumProtocolVersion(model.minimumProtocolVersion);
            certificate(model.certificate);
            certificateSource(model.certificateSource);
        }

        public final Boolean getCloudFrontDefaultCertificate() {
            return cloudFrontDefaultCertificate;
        }

        @Override
        public final Builder cloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate) {
            this.cloudFrontDefaultCertificate = cloudFrontDefaultCertificate;
            return this;
        }

        public final void setCloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate) {
            this.cloudFrontDefaultCertificate = cloudFrontDefaultCertificate;
        }

        public final String getIamCertificateId() {
            return iamCertificateId;
        }

        @Override
        public final Builder iamCertificateId(String iamCertificateId) {
            this.iamCertificateId = iamCertificateId;
            return this;
        }

        public final void setIamCertificateId(String iamCertificateId) {
            this.iamCertificateId = iamCertificateId;
        }

        public final String getAcmCertificateArn() {
            return acmCertificateArn;
        }

        @Override
        public final Builder acmCertificateArn(String acmCertificateArn) {
            this.acmCertificateArn = acmCertificateArn;
            return this;
        }

        public final void setAcmCertificateArn(String acmCertificateArn) {
            this.acmCertificateArn = acmCertificateArn;
        }

        public final String getSslSupportMethodAsString() {
            return sslSupportMethod;
        }

        @Override
        public final Builder sslSupportMethod(String sslSupportMethod) {
            this.sslSupportMethod = sslSupportMethod;
            return this;
        }

        @Override
        public final Builder sslSupportMethod(SSLSupportMethod sslSupportMethod) {
            this.sslSupportMethod(sslSupportMethod.toString());
            return this;
        }

        public final void setSslSupportMethod(String sslSupportMethod) {
            this.sslSupportMethod = sslSupportMethod;
        }

        public final String getMinimumProtocolVersionAsString() {
            return minimumProtocolVersion;
        }

        @Override
        public final Builder minimumProtocolVersion(String minimumProtocolVersion) {
            this.minimumProtocolVersion = minimumProtocolVersion;
            return this;
        }

        @Override
        public final Builder minimumProtocolVersion(MinimumProtocolVersion minimumProtocolVersion) {
            this.minimumProtocolVersion(minimumProtocolVersion.toString());
            return this;
        }

        public final void setMinimumProtocolVersion(String minimumProtocolVersion) {
            this.minimumProtocolVersion = minimumProtocolVersion;
        }

        public final String getCertificate() {
            return certificate;
        }

        @Override
        public final Builder certificate(String certificate) {
            this.certificate = certificate;
            return this;
        }

        public final void setCertificate(String certificate) {
            this.certificate = certificate;
        }

        public final String getCertificateSourceAsString() {
            return certificateSource;
        }

        @Override
        public final Builder certificateSource(String certificateSource) {
            this.certificateSource = certificateSource;
            return this;
        }

        @Override
        public final Builder certificateSource(CertificateSource certificateSource) {
            this.certificateSource(certificateSource.toString());
            return this;
        }

        public final void setCertificateSource(String certificateSource) {
            this.certificateSource = certificateSource;
        }

        @Override
        public ViewerCertificate build() {
            return new ViewerCertificate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
