/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the domain conflict. Use this information to determine the affected domain, the related
 * resource, and the affected Amazon Web Services account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DomainConflict implements SdkPojo, Serializable, ToCopyableBuilder<DomainConflict.Builder, DomainConflict> {
    private static final SdkField<String> DOMAIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Domain")
            .getter(getter(DomainConflict::domain))
            .setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain")
                    .unmarshallLocationName("Domain").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(DomainConflict::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("ResourceType").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceId")
            .getter(getter(DomainConflict::resourceId))
            .setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("ResourceId").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(DomainConflict::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId")
                    .unmarshallLocationName("AccountId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_FIELD,
            RESOURCE_TYPE_FIELD, RESOURCE_ID_FIELD, ACCOUNT_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String domain;

    private final String resourceType;

    private final String resourceId;

    private final String accountId;

    private DomainConflict(BuilderImpl builder) {
        this.domain = builder.domain;
        this.resourceType = builder.resourceType;
        this.resourceId = builder.resourceId;
        this.accountId = builder.accountId;
    }

    /**
     * <p>
     * The domain used to find existing conflicts for domain configurations.
     * </p>
     * 
     * @return The domain used to find existing conflicts for domain configurations.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * The CloudFront resource type that has a domain conflict.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link DistributionResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The CloudFront resource type that has a domain conflict.
     * @see DistributionResourceType
     */
    public final DistributionResourceType resourceType() {
        return DistributionResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The CloudFront resource type that has a domain conflict.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link DistributionResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The CloudFront resource type that has a domain conflict.
     * @see DistributionResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The ID of the resource that has a domain conflict.
     * </p>
     * 
     * @return The ID of the resource that has a domain conflict.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account for the domain conflict.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account for the domain conflict.
     */
    public final String accountId() {
        return accountId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DomainConflict)) {
            return false;
        }
        DomainConflict other = (DomainConflict) obj;
        return Objects.equals(domain(), other.domain()) && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(accountId(), other.accountId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DomainConflict").add("Domain", domain()).add("ResourceType", resourceTypeAsString())
                .add("ResourceId", resourceId()).add("AccountId", accountId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Domain", DOMAIN_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("AccountId", ACCOUNT_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DomainConflict, T> g) {
        return obj -> g.apply((DomainConflict) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DomainConflict> {
        /**
         * <p>
         * The domain used to find existing conflicts for domain configurations.
         * </p>
         * 
         * @param domain
         *        The domain used to find existing conflicts for domain configurations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * The CloudFront resource type that has a domain conflict.
         * </p>
         * 
         * @param resourceType
         *        The CloudFront resource type that has a domain conflict.
         * @see DistributionResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistributionResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The CloudFront resource type that has a domain conflict.
         * </p>
         * 
         * @param resourceType
         *        The CloudFront resource type that has a domain conflict.
         * @see DistributionResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistributionResourceType
         */
        Builder resourceType(DistributionResourceType resourceType);

        /**
         * <p>
         * The ID of the resource that has a domain conflict.
         * </p>
         * 
         * @param resourceId
         *        The ID of the resource that has a domain conflict.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The ID of the Amazon Web Services account for the domain conflict.
         * </p>
         * 
         * @param accountId
         *        The ID of the Amazon Web Services account for the domain conflict.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);
    }

    static final class BuilderImpl implements Builder {
        private String domain;

        private String resourceType;

        private String resourceId;

        private String accountId;

        private BuilderImpl() {
        }

        private BuilderImpl(DomainConflict model) {
            domain(model.domain);
            resourceType(model.resourceType);
            resourceId(model.resourceId);
            accountId(model.accountId);
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(DistributionResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        @Override
        public DomainConflict build() {
            return new DomainConflict(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
