/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.chime.ChimeClient;
import software.amazon.awssdk.services.chime.model.ListBotsRequest;
import software.amazon.awssdk.services.chime.model.ListBotsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.chime.ChimeClient#listBotsPaginator(software.amazon.awssdk.services.chime.model.ListBotsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.chime.model.ListBotsResponse} that can be used to iterate through all the
 * response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.chime.paginators.ListBotsIterable responses = client.listBotsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.chime.paginators.ListBotsIterable responses = client.listBotsPaginator(request);
 *     for (software.amazon.awssdk.services.chime.model.ListBotsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.chime.paginators.ListBotsIterable responses = client.listBotsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listBots(software.amazon.awssdk.services.chime.model.ListBotsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListBotsIterable implements SdkIterable<ListBotsResponse> {
    private final ChimeClient client;

    private final ListBotsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListBotsIterable(ChimeClient client, ListBotsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListBotsResponseFetcher();
    }

    @Override
    public Iterator<ListBotsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListBotsResponseFetcher implements SyncPageFetcher<ListBotsResponse> {
        @Override
        public boolean hasNextPage(ListBotsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListBotsResponse nextPage(ListBotsResponse previousPage) {
            if (previousPage == null) {
                return client.listBots(firstRequest);
            }
            return client.listBots(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
