/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a step adjustment for a <a
 * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepScalingPolicyConfiguration.html"
 * >StepScalingPolicyConfiguration</a>. Describes an adjustment based on the difference between the value of the
 * aggregated CloudWatch metric and the breach threshold that you've defined for the alarm.
 * </p>
 * <p>
 * For the following examples, suppose that you have an alarm with a breach threshold of 50:
 * </p>
 * <ul>
 * <li>
 * <p>
 * To initiate the adjustment when the metric is greater than or equal to 50 and less than 60, specify a lower bound of
 * <code>0</code> and an upper bound of <code>10</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * To initiate the adjustment when the metric is greater than 40 and less than or equal to 50, specify a lower bound of
 * <code>-10</code> and an upper bound of <code>0</code>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * There are a few rules for the step adjustments for your step policy:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The ranges of your step adjustments can't overlap or have a gap.
 * </p>
 * </li>
 * <li>
 * <p>
 * At most one step adjustment can have a null lower bound. If one step adjustment has a negative lower bound, then
 * there must be a step adjustment with a null lower bound.
 * </p>
 * </li>
 * <li>
 * <p>
 * At most one step adjustment can have a null upper bound. If one step adjustment has a positive upper bound, then
 * there must be a step adjustment with a null upper bound.
 * </p>
 * </li>
 * <li>
 * <p>
 * The upper and lower bound can't be null in the same step adjustment.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StepAdjustment implements SdkPojo, Serializable, ToCopyableBuilder<StepAdjustment.Builder, StepAdjustment> {
    private static final SdkField<Double> METRIC_INTERVAL_LOWER_BOUND_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MetricIntervalLowerBound").getter(getter(StepAdjustment::metricIntervalLowerBound))
            .setter(setter(Builder::metricIntervalLowerBound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricIntervalLowerBound").build())
            .build();

    private static final SdkField<Double> METRIC_INTERVAL_UPPER_BOUND_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MetricIntervalUpperBound").getter(getter(StepAdjustment::metricIntervalUpperBound))
            .setter(setter(Builder::metricIntervalUpperBound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricIntervalUpperBound").build())
            .build();

    private static final SdkField<Integer> SCALING_ADJUSTMENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScalingAdjustment").getter(getter(StepAdjustment::scalingAdjustment))
            .setter(setter(Builder::scalingAdjustment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingAdjustment").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            METRIC_INTERVAL_LOWER_BOUND_FIELD, METRIC_INTERVAL_UPPER_BOUND_FIELD, SCALING_ADJUSTMENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Double metricIntervalLowerBound;

    private final Double metricIntervalUpperBound;

    private final Integer scalingAdjustment;

    private StepAdjustment(BuilderImpl builder) {
        this.metricIntervalLowerBound = builder.metricIntervalLowerBound;
        this.metricIntervalUpperBound = builder.metricIntervalUpperBound;
        this.scalingAdjustment = builder.scalingAdjustment;
    }

    /**
     * <p>
     * The lower bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value is
     * above the breach threshold, the lower bound is inclusive (the metric must be greater than or equal to the
     * threshold plus the lower bound). Otherwise, it's exclusive (the metric must be greater than the threshold plus
     * the lower bound). A null value indicates negative infinity.
     * </p>
     * 
     * @return The lower bound for the difference between the alarm threshold and the CloudWatch metric. If the metric
     *         value is above the breach threshold, the lower bound is inclusive (the metric must be greater than or
     *         equal to the threshold plus the lower bound). Otherwise, it's exclusive (the metric must be greater than
     *         the threshold plus the lower bound). A null value indicates negative infinity.
     */
    public final Double metricIntervalLowerBound() {
        return metricIntervalLowerBound;
    }

    /**
     * <p>
     * The upper bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value is
     * above the breach threshold, the upper bound is exclusive (the metric must be less than the threshold plus the
     * upper bound). Otherwise, it's inclusive (the metric must be less than or equal to the threshold plus the upper
     * bound). A null value indicates positive infinity.
     * </p>
     * <p>
     * The upper bound must be greater than the lower bound.
     * </p>
     * 
     * @return The upper bound for the difference between the alarm threshold and the CloudWatch metric. If the metric
     *         value is above the breach threshold, the upper bound is exclusive (the metric must be less than the
     *         threshold plus the upper bound). Otherwise, it's inclusive (the metric must be less than or equal to the
     *         threshold plus the upper bound). A null value indicates positive infinity.</p>
     *         <p>
     *         The upper bound must be greater than the lower bound.
     */
    public final Double metricIntervalUpperBound() {
        return metricIntervalUpperBound;
    }

    /**
     * <p>
     * The amount by which to scale, based on the specified adjustment type. A positive value adds to the current
     * capacity while a negative number removes from the current capacity. For exact capacity, you must specify a
     * non-negative value.
     * </p>
     * 
     * @return The amount by which to scale, based on the specified adjustment type. A positive value adds to the
     *         current capacity while a negative number removes from the current capacity. For exact capacity, you must
     *         specify a non-negative value.
     */
    public final Integer scalingAdjustment() {
        return scalingAdjustment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricIntervalLowerBound());
        hashCode = 31 * hashCode + Objects.hashCode(metricIntervalUpperBound());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustment());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StepAdjustment)) {
            return false;
        }
        StepAdjustment other = (StepAdjustment) obj;
        return Objects.equals(metricIntervalLowerBound(), other.metricIntervalLowerBound())
                && Objects.equals(metricIntervalUpperBound(), other.metricIntervalUpperBound())
                && Objects.equals(scalingAdjustment(), other.scalingAdjustment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StepAdjustment").add("MetricIntervalLowerBound", metricIntervalLowerBound())
                .add("MetricIntervalUpperBound", metricIntervalUpperBound()).add("ScalingAdjustment", scalingAdjustment())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricIntervalLowerBound":
            return Optional.ofNullable(clazz.cast(metricIntervalLowerBound()));
        case "MetricIntervalUpperBound":
            return Optional.ofNullable(clazz.cast(metricIntervalUpperBound()));
        case "ScalingAdjustment":
            return Optional.ofNullable(clazz.cast(scalingAdjustment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StepAdjustment, T> g) {
        return obj -> g.apply((StepAdjustment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StepAdjustment> {
        /**
         * <p>
         * The lower bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value
         * is above the breach threshold, the lower bound is inclusive (the metric must be greater than or equal to the
         * threshold plus the lower bound). Otherwise, it's exclusive (the metric must be greater than the threshold
         * plus the lower bound). A null value indicates negative infinity.
         * </p>
         * 
         * @param metricIntervalLowerBound
         *        The lower bound for the difference between the alarm threshold and the CloudWatch metric. If the
         *        metric value is above the breach threshold, the lower bound is inclusive (the metric must be greater
         *        than or equal to the threshold plus the lower bound). Otherwise, it's exclusive (the metric must be
         *        greater than the threshold plus the lower bound). A null value indicates negative infinity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricIntervalLowerBound(Double metricIntervalLowerBound);

        /**
         * <p>
         * The upper bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value
         * is above the breach threshold, the upper bound is exclusive (the metric must be less than the threshold plus
         * the upper bound). Otherwise, it's inclusive (the metric must be less than or equal to the threshold plus the
         * upper bound). A null value indicates positive infinity.
         * </p>
         * <p>
         * The upper bound must be greater than the lower bound.
         * </p>
         * 
         * @param metricIntervalUpperBound
         *        The upper bound for the difference between the alarm threshold and the CloudWatch metric. If the
         *        metric value is above the breach threshold, the upper bound is exclusive (the metric must be less than
         *        the threshold plus the upper bound). Otherwise, it's inclusive (the metric must be less than or equal
         *        to the threshold plus the upper bound). A null value indicates positive infinity.</p>
         *        <p>
         *        The upper bound must be greater than the lower bound.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricIntervalUpperBound(Double metricIntervalUpperBound);

        /**
         * <p>
         * The amount by which to scale, based on the specified adjustment type. A positive value adds to the current
         * capacity while a negative number removes from the current capacity. For exact capacity, you must specify a
         * non-negative value.
         * </p>
         * 
         * @param scalingAdjustment
         *        The amount by which to scale, based on the specified adjustment type. A positive value adds to the
         *        current capacity while a negative number removes from the current capacity. For exact capacity, you
         *        must specify a non-negative value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingAdjustment(Integer scalingAdjustment);
    }

    static final class BuilderImpl implements Builder {
        private Double metricIntervalLowerBound;

        private Double metricIntervalUpperBound;

        private Integer scalingAdjustment;

        private BuilderImpl() {
        }

        private BuilderImpl(StepAdjustment model) {
            metricIntervalLowerBound(model.metricIntervalLowerBound);
            metricIntervalUpperBound(model.metricIntervalUpperBound);
            scalingAdjustment(model.scalingAdjustment);
        }

        public final Double getMetricIntervalLowerBound() {
            return metricIntervalLowerBound;
        }

        public final void setMetricIntervalLowerBound(Double metricIntervalLowerBound) {
            this.metricIntervalLowerBound = metricIntervalLowerBound;
        }

        @Override
        public final Builder metricIntervalLowerBound(Double metricIntervalLowerBound) {
            this.metricIntervalLowerBound = metricIntervalLowerBound;
            return this;
        }

        public final Double getMetricIntervalUpperBound() {
            return metricIntervalUpperBound;
        }

        public final void setMetricIntervalUpperBound(Double metricIntervalUpperBound) {
            this.metricIntervalUpperBound = metricIntervalUpperBound;
        }

        @Override
        public final Builder metricIntervalUpperBound(Double metricIntervalUpperBound) {
            this.metricIntervalUpperBound = metricIntervalUpperBound;
            return this;
        }

        public final Integer getScalingAdjustment() {
            return scalingAdjustment;
        }

        public final void setScalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
        }

        @Override
        public final Builder scalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
            return this;
        }

        @Override
        public StepAdjustment build() {
            return new StepAdjustment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
