/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains incremental updates to citation information during streaming responses. This allows clients to build up
 * citation data progressively as the response is generated.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CitationsDelta implements SdkPojo, Serializable, ToCopyableBuilder<CitationsDelta.Builder, CitationsDelta> {
    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(CitationsDelta::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(CitationsDelta::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<List<CitationSourceContentDelta>> SOURCE_CONTENT_FIELD = SdkField
            .<List<CitationSourceContentDelta>> builder(MarshallingType.LIST)
            .memberName("sourceContent")
            .getter(getter(CitationsDelta::sourceContent))
            .setter(setter(Builder::sourceContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceContent").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CitationSourceContentDelta> builder(MarshallingType.SDK_POJO)
                                            .constructor(CitationSourceContentDelta::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CitationLocation> LOCATION_FIELD = SdkField
            .<CitationLocation> builder(MarshallingType.SDK_POJO).memberName("location").getter(getter(CitationsDelta::location))
            .setter(setter(Builder::location)).constructor(CitationLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TITLE_FIELD, SOURCE_FIELD,
            SOURCE_CONTENT_FIELD, LOCATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String title;

    private final String source;

    private final List<CitationSourceContentDelta> sourceContent;

    private final CitationLocation location;

    private CitationsDelta(BuilderImpl builder) {
        this.title = builder.title;
        this.source = builder.source;
        this.sourceContent = builder.sourceContent;
        this.location = builder.location;
    }

    /**
     * <p>
     * The title or identifier of the source document being cited.
     * </p>
     * 
     * @return The title or identifier of the source document being cited.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The source from the original search result that provided the cited content.
     * </p>
     * 
     * @return The source from the original search result that provided the cited content.
     */
    public final String source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceContent property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceContent() {
        return sourceContent != null && !(sourceContent instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific content from the source document that was referenced or cited in the generated response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceContent} method.
     * </p>
     * 
     * @return The specific content from the source document that was referenced or cited in the generated response.
     */
    public final List<CitationSourceContentDelta> sourceContent() {
        return sourceContent;
    }

    /**
     * <p>
     * Specifies the precise location within a source document where cited content can be found. This can include
     * character-level positions, page numbers, or document chunks depending on the document type and indexing method.
     * </p>
     * 
     * @return Specifies the precise location within a source document where cited content can be found. This can
     *         include character-level positions, page numbers, or document chunks depending on the document type and
     *         indexing method.
     */
    public final CitationLocation location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceContent() ? sourceContent() : null);
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CitationsDelta)) {
            return false;
        }
        CitationsDelta other = (CitationsDelta) obj;
        return Objects.equals(title(), other.title()) && Objects.equals(source(), other.source())
                && hasSourceContent() == other.hasSourceContent() && Objects.equals(sourceContent(), other.sourceContent())
                && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CitationsDelta").add("Title", title()).add("Source", source())
                .add("SourceContent", hasSourceContent() ? sourceContent() : null).add("Location", location()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "sourceContent":
            return Optional.ofNullable(clazz.cast(sourceContent()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("title", TITLE_FIELD);
        map.put("source", SOURCE_FIELD);
        map.put("sourceContent", SOURCE_CONTENT_FIELD);
        map.put("location", LOCATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CitationsDelta, T> g) {
        return obj -> g.apply((CitationsDelta) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CitationsDelta> {
        /**
         * <p>
         * The title or identifier of the source document being cited.
         * </p>
         * 
         * @param title
         *        The title or identifier of the source document being cited.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The source from the original search result that provided the cited content.
         * </p>
         * 
         * @param source
         *        The source from the original search result that provided the cited content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The specific content from the source document that was referenced or cited in the generated response.
         * </p>
         * 
         * @param sourceContent
         *        The specific content from the source document that was referenced or cited in the generated response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceContent(Collection<CitationSourceContentDelta> sourceContent);

        /**
         * <p>
         * The specific content from the source document that was referenced or cited in the generated response.
         * </p>
         * 
         * @param sourceContent
         *        The specific content from the source document that was referenced or cited in the generated response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceContent(CitationSourceContentDelta... sourceContent);

        /**
         * <p>
         * The specific content from the source document that was referenced or cited in the generated response.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrockruntime.model.CitationSourceContentDelta.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.bedrockruntime.model.CitationSourceContentDelta#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrockruntime.model.CitationSourceContentDelta.Builder#build()} is
         * called immediately and its result is passed to {@link #sourceContent(List<CitationSourceContentDelta>)}.
         * 
         * @param sourceContent
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrockruntime.model.CitationSourceContentDelta.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceContent(java.util.Collection<CitationSourceContentDelta>)
         */
        Builder sourceContent(Consumer<CitationSourceContentDelta.Builder>... sourceContent);

        /**
         * <p>
         * Specifies the precise location within a source document where cited content can be found. This can include
         * character-level positions, page numbers, or document chunks depending on the document type and indexing
         * method.
         * </p>
         * 
         * @param location
         *        Specifies the precise location within a source document where cited content can be found. This can
         *        include character-level positions, page numbers, or document chunks depending on the document type and
         *        indexing method.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(CitationLocation location);

        /**
         * <p>
         * Specifies the precise location within a source document where cited content can be found. This can include
         * character-level positions, page numbers, or document chunks depending on the document type and indexing
         * method.
         * </p>
         * This is a convenience method that creates an instance of the {@link CitationLocation.Builder} avoiding the
         * need to create one manually via {@link CitationLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CitationLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(CitationLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link CitationLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(CitationLocation)
         */
        default Builder location(Consumer<CitationLocation.Builder> location) {
            return location(CitationLocation.builder().applyMutation(location).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String title;

        private String source;

        private List<CitationSourceContentDelta> sourceContent = DefaultSdkAutoConstructList.getInstance();

        private CitationLocation location;

        private BuilderImpl() {
        }

        private BuilderImpl(CitationsDelta model) {
            title(model.title);
            source(model.source);
            sourceContent(model.sourceContent);
            location(model.location);
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final List<CitationSourceContentDelta.Builder> getSourceContent() {
            List<CitationSourceContentDelta.Builder> result = CitationSourceContentListDeltaCopier
                    .copyToBuilder(this.sourceContent);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceContent(Collection<CitationSourceContentDelta.BuilderImpl> sourceContent) {
            this.sourceContent = CitationSourceContentListDeltaCopier.copyFromBuilder(sourceContent);
        }

        @Override
        public final Builder sourceContent(Collection<CitationSourceContentDelta> sourceContent) {
            this.sourceContent = CitationSourceContentListDeltaCopier.copy(sourceContent);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceContent(CitationSourceContentDelta... sourceContent) {
            sourceContent(Arrays.asList(sourceContent));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceContent(Consumer<CitationSourceContentDelta.Builder>... sourceContent) {
            sourceContent(Stream.of(sourceContent).map(c -> CitationSourceContentDelta.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final CitationLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        public final void setLocation(CitationLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        @Override
        public final Builder location(CitationLocation location) {
            this.location = location;
            return this;
        }

        @Override
        public CitationsDelta build() {
            return new CitationsDelta(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
