/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcorecontrol.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for a Model Context Protocol (MCP) gateway. This structure defines how the gateway implements the
 * MCP protocol.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MCPGatewayConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MCPGatewayConfiguration.Builder, MCPGatewayConfiguration> {
    private static final SdkField<List<String>> SUPPORTED_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedVersions")
            .getter(getter(MCPGatewayConfiguration::supportedVersions))
            .setter(setter(Builder::supportedVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTRUCTIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instructions").getter(getter(MCPGatewayConfiguration::instructions))
            .setter(setter(Builder::instructions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instructions").build()).build();

    private static final SdkField<String> SEARCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("searchType").getter(getter(MCPGatewayConfiguration::searchTypeAsString))
            .setter(setter(Builder::searchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("searchType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUPPORTED_VERSIONS_FIELD,
            INSTRUCTIONS_FIELD, SEARCH_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> supportedVersions;

    private final String instructions;

    private final String searchType;

    private MCPGatewayConfiguration(BuilderImpl builder) {
        this.supportedVersions = builder.supportedVersions;
        this.instructions = builder.instructions;
        this.searchType = builder.searchType;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedVersions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedVersions() {
        return supportedVersions != null && !(supportedVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The supported versions of the Model Context Protocol. This field specifies which versions of the protocol the
     * gateway can use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedVersions} method.
     * </p>
     * 
     * @return The supported versions of the Model Context Protocol. This field specifies which versions of the protocol
     *         the gateway can use.
     */
    public final List<String> supportedVersions() {
        return supportedVersions;
    }

    /**
     * <p>
     * The instructions for using the Model Context Protocol gateway. These instructions provide guidance on how to
     * interact with the gateway.
     * </p>
     * 
     * @return The instructions for using the Model Context Protocol gateway. These instructions provide guidance on how
     *         to interact with the gateway.
     */
    public final String instructions() {
        return instructions;
    }

    /**
     * <p>
     * The search type for the Model Context Protocol gateway. This field specifies how the gateway handles search
     * operations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #searchType} will
     * return {@link SearchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #searchTypeAsString}.
     * </p>
     * 
     * @return The search type for the Model Context Protocol gateway. This field specifies how the gateway handles
     *         search operations.
     * @see SearchType
     */
    public final SearchType searchType() {
        return SearchType.fromValue(searchType);
    }

    /**
     * <p>
     * The search type for the Model Context Protocol gateway. This field specifies how the gateway handles search
     * operations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #searchType} will
     * return {@link SearchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #searchTypeAsString}.
     * </p>
     * 
     * @return The search type for the Model Context Protocol gateway. This field specifies how the gateway handles
     *         search operations.
     * @see SearchType
     */
    public final String searchTypeAsString() {
        return searchType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedVersions() ? supportedVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instructions());
        hashCode = 31 * hashCode + Objects.hashCode(searchTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MCPGatewayConfiguration)) {
            return false;
        }
        MCPGatewayConfiguration other = (MCPGatewayConfiguration) obj;
        return hasSupportedVersions() == other.hasSupportedVersions()
                && Objects.equals(supportedVersions(), other.supportedVersions())
                && Objects.equals(instructions(), other.instructions())
                && Objects.equals(searchTypeAsString(), other.searchTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MCPGatewayConfiguration")
                .add("SupportedVersions", hasSupportedVersions() ? supportedVersions() : null)
                .add("Instructions", instructions()).add("SearchType", searchTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "supportedVersions":
            return Optional.ofNullable(clazz.cast(supportedVersions()));
        case "instructions":
            return Optional.ofNullable(clazz.cast(instructions()));
        case "searchType":
            return Optional.ofNullable(clazz.cast(searchTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("supportedVersions", SUPPORTED_VERSIONS_FIELD);
        map.put("instructions", INSTRUCTIONS_FIELD);
        map.put("searchType", SEARCH_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MCPGatewayConfiguration, T> g) {
        return obj -> g.apply((MCPGatewayConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MCPGatewayConfiguration> {
        /**
         * <p>
         * The supported versions of the Model Context Protocol. This field specifies which versions of the protocol the
         * gateway can use.
         * </p>
         * 
         * @param supportedVersions
         *        The supported versions of the Model Context Protocol. This field specifies which versions of the
         *        protocol the gateway can use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedVersions(Collection<String> supportedVersions);

        /**
         * <p>
         * The supported versions of the Model Context Protocol. This field specifies which versions of the protocol the
         * gateway can use.
         * </p>
         * 
         * @param supportedVersions
         *        The supported versions of the Model Context Protocol. This field specifies which versions of the
         *        protocol the gateway can use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedVersions(String... supportedVersions);

        /**
         * <p>
         * The instructions for using the Model Context Protocol gateway. These instructions provide guidance on how to
         * interact with the gateway.
         * </p>
         * 
         * @param instructions
         *        The instructions for using the Model Context Protocol gateway. These instructions provide guidance on
         *        how to interact with the gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instructions(String instructions);

        /**
         * <p>
         * The search type for the Model Context Protocol gateway. This field specifies how the gateway handles search
         * operations.
         * </p>
         * 
         * @param searchType
         *        The search type for the Model Context Protocol gateway. This field specifies how the gateway handles
         *        search operations.
         * @see SearchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SearchType
         */
        Builder searchType(String searchType);

        /**
         * <p>
         * The search type for the Model Context Protocol gateway. This field specifies how the gateway handles search
         * operations.
         * </p>
         * 
         * @param searchType
         *        The search type for the Model Context Protocol gateway. This field specifies how the gateway handles
         *        search operations.
         * @see SearchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SearchType
         */
        Builder searchType(SearchType searchType);
    }

    static final class BuilderImpl implements Builder {
        private List<String> supportedVersions = DefaultSdkAutoConstructList.getInstance();

        private String instructions;

        private String searchType;

        private BuilderImpl() {
        }

        private BuilderImpl(MCPGatewayConfiguration model) {
            supportedVersions(model.supportedVersions);
            instructions(model.instructions);
            searchType(model.searchType);
        }

        public final Collection<String> getSupportedVersions() {
            if (supportedVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedVersions;
        }

        public final void setSupportedVersions(Collection<String> supportedVersions) {
            this.supportedVersions = McpSupportedVersionsCopier.copy(supportedVersions);
        }

        @Override
        public final Builder supportedVersions(Collection<String> supportedVersions) {
            this.supportedVersions = McpSupportedVersionsCopier.copy(supportedVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedVersions(String... supportedVersions) {
            supportedVersions(Arrays.asList(supportedVersions));
            return this;
        }

        public final String getInstructions() {
            return instructions;
        }

        public final void setInstructions(String instructions) {
            this.instructions = instructions;
        }

        @Override
        public final Builder instructions(String instructions) {
            this.instructions = instructions;
            return this;
        }

        public final String getSearchType() {
            return searchType;
        }

        public final void setSearchType(String searchType) {
            this.searchType = searchType;
        }

        @Override
        public final Builder searchType(String searchType) {
            this.searchType = searchType;
            return this;
        }

        @Override
        public final Builder searchType(SearchType searchType) {
            this.searchType(searchType == null ? null : searchType.toString());
            return this;
        }

        @Override
        public MCPGatewayConfiguration build() {
            return new MCPGatewayConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
