/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcorecontrol.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An OAuth credential provider for gateway authentication. This structure contains the configuration for authenticating
 * with the target endpoint using OAuth.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OAuthCredentialProvider implements SdkPojo, Serializable,
        ToCopyableBuilder<OAuthCredentialProvider.Builder, OAuthCredentialProvider> {
    private static final SdkField<String> PROVIDER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("providerArn").getter(getter(OAuthCredentialProvider::providerArn)).setter(setter(Builder::providerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("providerArn").build()).build();

    private static final SdkField<List<String>> SCOPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("scopes")
            .getter(getter(OAuthCredentialProvider::scopes))
            .setter(setter(Builder::scopes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scopes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CUSTOM_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("customParameters")
            .getter(getter(OAuthCredentialProvider::customParameters))
            .setter(setter(Builder::customParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROVIDER_ARN_FIELD,
            SCOPES_FIELD, CUSTOM_PARAMETERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String providerArn;

    private final List<String> scopes;

    private final Map<String, String> customParameters;

    private OAuthCredentialProvider(BuilderImpl builder) {
        this.providerArn = builder.providerArn;
        this.scopes = builder.scopes;
        this.customParameters = builder.customParameters;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the OAuth credential provider. This ARN identifies the provider in Amazon Web
     * Services.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the OAuth credential provider. This ARN identifies the provider in
     *         Amazon Web Services.
     */
    public final String providerArn() {
        return providerArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Scopes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasScopes() {
        return scopes != null && !(scopes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The OAuth scopes for the credential provider. These scopes define the level of access requested from the OAuth
     * provider.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScopes} method.
     * </p>
     * 
     * @return The OAuth scopes for the credential provider. These scopes define the level of access requested from the
     *         OAuth provider.
     */
    public final List<String> scopes() {
        return scopes;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomParameters() {
        return customParameters != null && !(customParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The custom parameters for the OAuth credential provider. These parameters provide additional configuration for
     * the OAuth authentication process.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomParameters} method.
     * </p>
     * 
     * @return The custom parameters for the OAuth credential provider. These parameters provide additional
     *         configuration for the OAuth authentication process.
     */
    public final Map<String, String> customParameters() {
        return customParameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(providerArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasScopes() ? scopes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomParameters() ? customParameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OAuthCredentialProvider)) {
            return false;
        }
        OAuthCredentialProvider other = (OAuthCredentialProvider) obj;
        return Objects.equals(providerArn(), other.providerArn()) && hasScopes() == other.hasScopes()
                && Objects.equals(scopes(), other.scopes()) && hasCustomParameters() == other.hasCustomParameters()
                && Objects.equals(customParameters(), other.customParameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OAuthCredentialProvider").add("ProviderArn", providerArn())
                .add("Scopes", hasScopes() ? scopes() : null)
                .add("CustomParameters", customParameters() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "providerArn":
            return Optional.ofNullable(clazz.cast(providerArn()));
        case "scopes":
            return Optional.ofNullable(clazz.cast(scopes()));
        case "customParameters":
            return Optional.ofNullable(clazz.cast(customParameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("providerArn", PROVIDER_ARN_FIELD);
        map.put("scopes", SCOPES_FIELD);
        map.put("customParameters", CUSTOM_PARAMETERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OAuthCredentialProvider, T> g) {
        return obj -> g.apply((OAuthCredentialProvider) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OAuthCredentialProvider> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the OAuth credential provider. This ARN identifies the provider in Amazon
         * Web Services.
         * </p>
         * 
         * @param providerArn
         *        The Amazon Resource Name (ARN) of the OAuth credential provider. This ARN identifies the provider in
         *        Amazon Web Services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerArn(String providerArn);

        /**
         * <p>
         * The OAuth scopes for the credential provider. These scopes define the level of access requested from the
         * OAuth provider.
         * </p>
         * 
         * @param scopes
         *        The OAuth scopes for the credential provider. These scopes define the level of access requested from
         *        the OAuth provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(Collection<String> scopes);

        /**
         * <p>
         * The OAuth scopes for the credential provider. These scopes define the level of access requested from the
         * OAuth provider.
         * </p>
         * 
         * @param scopes
         *        The OAuth scopes for the credential provider. These scopes define the level of access requested from
         *        the OAuth provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(String... scopes);

        /**
         * <p>
         * The custom parameters for the OAuth credential provider. These parameters provide additional configuration
         * for the OAuth authentication process.
         * </p>
         * 
         * @param customParameters
         *        The custom parameters for the OAuth credential provider. These parameters provide additional
         *        configuration for the OAuth authentication process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customParameters(Map<String, String> customParameters);
    }

    static final class BuilderImpl implements Builder {
        private String providerArn;

        private List<String> scopes = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> customParameters = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OAuthCredentialProvider model) {
            providerArn(model.providerArn);
            scopes(model.scopes);
            customParameters(model.customParameters);
        }

        public final String getProviderArn() {
            return providerArn;
        }

        public final void setProviderArn(String providerArn) {
            this.providerArn = providerArn;
        }

        @Override
        public final Builder providerArn(String providerArn) {
            this.providerArn = providerArn;
            return this;
        }

        public final Collection<String> getScopes() {
            if (scopes instanceof SdkAutoConstructList) {
                return null;
            }
            return scopes;
        }

        public final void setScopes(Collection<String> scopes) {
            this.scopes = OAuthScopesCopier.copy(scopes);
        }

        @Override
        public final Builder scopes(Collection<String> scopes) {
            this.scopes = OAuthScopesCopier.copy(scopes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scopes(String... scopes) {
            scopes(Arrays.asList(scopes));
            return this;
        }

        public final Map<String, String> getCustomParameters() {
            if (customParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return customParameters;
        }

        public final void setCustomParameters(Map<String, String> customParameters) {
            this.customParameters = OAuthCustomParametersCopier.copy(customParameters);
        }

        @Override
        public final Builder customParameters(Map<String, String> customParameters) {
            this.customParameters = OAuthCustomParametersCopier.copy(customParameters);
            return this;
        }

        @Override
        public OAuthCredentialProvider build() {
            return new OAuthCredentialProvider(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
