/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcore;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockagentcore.internal.BedrockAgentCoreServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockagentcore.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.bedrockagentcore.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockagentcore.model.BedrockAgentCoreException;
import software.amazon.awssdk.services.bedrockagentcore.model.CodeInterpreterStreamOutput;
import software.amazon.awssdk.services.bedrockagentcore.model.ConflictException;
import software.amazon.awssdk.services.bedrockagentcore.model.CreateEventRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.CreateEventResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.DeleteEventRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.DeleteEventResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.DeleteMemoryRecordRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.DeleteMemoryRecordResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetBrowserSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetBrowserSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetCodeInterpreterSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetCodeInterpreterSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetEventRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetEventResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetMemoryRecordRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetMemoryRecordResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetResourceApiKeyRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetResourceApiKeyResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetResourceOauth2TokenRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetResourceOauth2TokenResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenForJwtRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenForJwtResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenForUserIdRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenForUserIdResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.GetWorkloadAccessTokenResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.InternalServerException;
import software.amazon.awssdk.services.bedrockagentcore.model.InvalidInputException;
import software.amazon.awssdk.services.bedrockagentcore.model.InvokeAgentRuntimeRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.InvokeAgentRuntimeResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.InvokeCodeInterpreterRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.InvokeCodeInterpreterResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.InvokeCodeInterpreterResponseHandler;
import software.amazon.awssdk.services.bedrockagentcore.model.ListActorsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListActorsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ListBrowserSessionsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListBrowserSessionsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ListCodeInterpreterSessionsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListCodeInterpreterSessionsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ListEventsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListEventsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ListMemoryRecordsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListMemoryRecordsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ListSessionsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.ListSessionsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockagentcore.model.RetrieveMemoryRecordsRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.RetrieveMemoryRecordsResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.RuntimeClientErrorException;
import software.amazon.awssdk.services.bedrockagentcore.model.ServiceException;
import software.amazon.awssdk.services.bedrockagentcore.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockagentcore.model.StartBrowserSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.StartBrowserSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.StartCodeInterpreterSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.StartCodeInterpreterSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.StopBrowserSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.StopBrowserSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.StopCodeInterpreterSessionRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.StopCodeInterpreterSessionResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ThrottledException;
import software.amazon.awssdk.services.bedrockagentcore.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockagentcore.model.UnauthorizedException;
import software.amazon.awssdk.services.bedrockagentcore.model.UpdateBrowserStreamRequest;
import software.amazon.awssdk.services.bedrockagentcore.model.UpdateBrowserStreamResponse;
import software.amazon.awssdk.services.bedrockagentcore.model.ValidationException;
import software.amazon.awssdk.services.bedrockagentcore.transform.CreateEventRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.DeleteEventRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.DeleteMemoryRecordRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetBrowserSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetCodeInterpreterSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetEventRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetMemoryRecordRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetResourceApiKeyRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetResourceOauth2TokenRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetWorkloadAccessTokenForJwtRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetWorkloadAccessTokenForUserIdRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.GetWorkloadAccessTokenRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.InvokeAgentRuntimeRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.InvokeCodeInterpreterRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListActorsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListBrowserSessionsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListCodeInterpreterSessionsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListEventsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListMemoryRecordsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.ListSessionsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.RetrieveMemoryRecordsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.StartBrowserSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.StartCodeInterpreterSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.StopBrowserSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.StopCodeInterpreterSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentcore.transform.UpdateBrowserStreamRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link BedrockAgentCoreAsyncClient}.
 *
 * @see BedrockAgentCoreAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockAgentCoreAsyncClient implements BedrockAgentCoreAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockAgentCoreAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultBedrockAgentCoreAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Bedrock_AgentCore" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Creates an event in an AgentCore Memory resource. Events represent interactions or activities that occur within a
     * session and are associated with specific actors.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:CreateEvent</code> permission.
     * </p>
     * <p>
     * This operation is subject to request rate limiting.
     * </p>
     *
     * @param createEventRequest
     * @return A Java Future containing the result of the CreateEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.CreateEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/CreateEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventResponse> createEvent(CreateEventRequest createEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateEventResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventRequest, CreateEventResponse>()
                            .withOperationName("CreateEvent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEventRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEventRequest));
            CompletableFuture<CreateEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an event from an AgentCore Memory resource. When you delete an event, it is permanently removed.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:DeleteEvent</code> permission.
     * </p>
     *
     * @param deleteEventRequest
     * @return A Java Future containing the result of the DeleteEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.DeleteEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/DeleteEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventResponse> deleteEvent(DeleteEventRequest deleteEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteEventResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventRequest, DeleteEventResponse>()
                            .withOperationName("DeleteEvent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEventRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEventRequest));
            CompletableFuture<DeleteEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a memory record from an AgentCore Memory resource. When you delete a memory record, it is permanently
     * removed.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:DeleteMemoryRecord</code> permission.
     * </p>
     *
     * @param deleteMemoryRecordRequest
     * @return A Java Future containing the result of the DeleteMemoryRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.DeleteMemoryRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/DeleteMemoryRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMemoryRecordResponse> deleteMemoryRecord(DeleteMemoryRecordRequest deleteMemoryRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMemoryRecordRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMemoryRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMemoryRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMemoryRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMemoryRecordResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMemoryRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMemoryRecordRequest, DeleteMemoryRecordResponse>()
                            .withOperationName("DeleteMemoryRecord").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMemoryRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMemoryRecordRequest));
            CompletableFuture<DeleteMemoryRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific browser session in Amazon Bedrock. This operation returns the
     * session's configuration, current status, associated streams, and metadata.
     * </p>
     * <p>
     * To get a browser session, you must specify both the browser identifier and the session ID. The response includes
     * information about the session's viewport configuration, timeout settings, and stream endpoints.
     * </p>
     * <p>
     * The following operations are related to <code>GetBrowserSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartBrowserSession.html">StartBrowserSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_ListBrowserSessions.html">ListBrowserSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StopBrowserSession.html">StopBrowserSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getBrowserSessionRequest
     * @return A Java Future containing the result of the GetBrowserSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetBrowserSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetBrowserSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBrowserSessionResponse> getBrowserSession(GetBrowserSessionRequest getBrowserSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBrowserSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBrowserSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBrowserSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBrowserSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBrowserSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetBrowserSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBrowserSessionRequest, GetBrowserSessionResponse>()
                            .withOperationName("GetBrowserSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBrowserSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBrowserSessionRequest));
            CompletableFuture<GetBrowserSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific code interpreter session in Amazon Bedrock. This operation
     * returns the session's configuration, current status, and metadata.
     * </p>
     * <p>
     * To get a code interpreter session, you must specify both the code interpreter identifier and the session ID. The
     * response includes information about the session's timeout settings and current status.
     * </p>
     * <p>
     * The following operations are related to <code>GetCodeInterpreterSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartCodeInterpreterSession.html">StartCodeInterpreterSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_ListCodeInterpreterSessions.html">ListCodeInterpreterSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StopCodeInterpreterSession.html">StopCodeInterpreterSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getCodeInterpreterSessionRequest
     * @return A Java Future containing the result of the GetCodeInterpreterSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetCodeInterpreterSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetCodeInterpreterSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCodeInterpreterSessionResponse> getCodeInterpreterSession(
            GetCodeInterpreterSessionRequest getCodeInterpreterSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCodeInterpreterSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCodeInterpreterSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCodeInterpreterSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCodeInterpreterSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCodeInterpreterSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetCodeInterpreterSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCodeInterpreterSessionRequest, GetCodeInterpreterSessionResponse>()
                            .withOperationName("GetCodeInterpreterSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCodeInterpreterSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCodeInterpreterSessionRequest));
            CompletableFuture<GetCodeInterpreterSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a specific event in an AgentCore Memory resource.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:GetEvent</code> permission.
     * </p>
     *
     * @param getEventRequest
     * @return A Java Future containing the result of the GetEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventResponse> getEvent(GetEventRequest getEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetEventResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventRequest, GetEventResponse>().withOperationName("GetEvent")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEventRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEventRequest));
            CompletableFuture<GetEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specific memory record from an AgentCore Memory resource.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:GetMemoryRecord</code> permission.
     * </p>
     *
     * @param getMemoryRecordRequest
     * @return A Java Future containing the result of the GetMemoryRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetMemoryRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetMemoryRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMemoryRecordResponse> getMemoryRecord(GetMemoryRecordRequest getMemoryRecordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMemoryRecordRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMemoryRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMemoryRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMemoryRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMemoryRecordResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetMemoryRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMemoryRecordRequest, GetMemoryRecordResponse>()
                            .withOperationName("GetMemoryRecord").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMemoryRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMemoryRecordRequest));
            CompletableFuture<GetMemoryRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an API Key associated with an API Key Credential Provider
     * </p>
     *
     * @param getResourceApiKeyRequest
     * @return A Java Future containing the result of the GetResourceApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException This exception is thrown when the JWT bearer token is invalid or not found for
     *         OAuth bearer token based access</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetResourceApiKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetResourceApiKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceApiKeyResponse> getResourceApiKey(GetResourceApiKeyRequest getResourceApiKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceApiKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceApiKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourceApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceApiKeyRequest, GetResourceApiKeyResponse>()
                            .withOperationName("GetResourceApiKey").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceApiKeyRequest));
            CompletableFuture<GetResourceApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the OAuth 2.0 token of the provided resource
     * </p>
     *
     * @param getResourceOauth2TokenRequest
     * @return A Java Future containing the result of the GetResourceOauth2Token operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException This exception is thrown when the JWT bearer token is invalid or not found for
     *         OAuth bearer token based access</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetResourceOauth2Token
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetResourceOauth2Token"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceOauth2TokenResponse> getResourceOauth2Token(
            GetResourceOauth2TokenRequest getResourceOauth2TokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceOauth2TokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceOauth2TokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceOauth2Token");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceOauth2TokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceOauth2TokenResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourceOauth2TokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceOauth2TokenRequest, GetResourceOauth2TokenResponse>()
                            .withOperationName("GetResourceOauth2Token").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceOauth2TokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceOauth2TokenRequest));
            CompletableFuture<GetResourceOauth2TokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains an Workload access token for agentic workloads not acting on behalf of user.
     * </p>
     *
     * @param getWorkloadAccessTokenRequest
     * @return A Java Future containing the result of the GetWorkloadAccessToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException This exception is thrown when the JWT bearer token is invalid or not found for
     *         OAuth bearer token based access</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetWorkloadAccessToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetWorkloadAccessToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkloadAccessTokenResponse> getWorkloadAccessToken(
            GetWorkloadAccessTokenRequest getWorkloadAccessTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkloadAccessTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadAccessTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkloadAccessToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkloadAccessTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkloadAccessTokenResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWorkloadAccessTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkloadAccessTokenRequest, GetWorkloadAccessTokenResponse>()
                            .withOperationName("GetWorkloadAccessToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWorkloadAccessTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkloadAccessTokenRequest));
            CompletableFuture<GetWorkloadAccessTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains an Workload access token for agentic workloads acting on behalf of user with JWT token
     * </p>
     *
     * @param getWorkloadAccessTokenForJwtRequest
     * @return A Java Future containing the result of the GetWorkloadAccessTokenForJWT operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException This exception is thrown when the JWT bearer token is invalid or not found for
     *         OAuth bearer token based access</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetWorkloadAccessTokenForJWT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetWorkloadAccessTokenForJWT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkloadAccessTokenForJwtResponse> getWorkloadAccessTokenForJWT(
            GetWorkloadAccessTokenForJwtRequest getWorkloadAccessTokenForJwtRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkloadAccessTokenForJwtRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadAccessTokenForJwtRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkloadAccessTokenForJWT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkloadAccessTokenForJwtResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkloadAccessTokenForJwtResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWorkloadAccessTokenForJwtResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkloadAccessTokenForJwtRequest, GetWorkloadAccessTokenForJwtResponse>()
                            .withOperationName("GetWorkloadAccessTokenForJWT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWorkloadAccessTokenForJwtRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkloadAccessTokenForJwtRequest));
            CompletableFuture<GetWorkloadAccessTokenForJwtResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains an Workload access token for agentic workloads acting on behalf of user with User Id.
     * </p>
     *
     * @param getWorkloadAccessTokenForUserIdRequest
     * @return A Java Future containing the result of the GetWorkloadAccessTokenForUserId operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException This exception is thrown when the JWT bearer token is invalid or not found for
     *         OAuth bearer token based access</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.GetWorkloadAccessTokenForUserId
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/GetWorkloadAccessTokenForUserId"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkloadAccessTokenForUserIdResponse> getWorkloadAccessTokenForUserId(
            GetWorkloadAccessTokenForUserIdRequest getWorkloadAccessTokenForUserIdRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkloadAccessTokenForUserIdRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getWorkloadAccessTokenForUserIdRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkloadAccessTokenForUserId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkloadAccessTokenForUserIdResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkloadAccessTokenForUserIdResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWorkloadAccessTokenForUserIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkloadAccessTokenForUserIdRequest, GetWorkloadAccessTokenForUserIdResponse>()
                            .withOperationName("GetWorkloadAccessTokenForUserId").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWorkloadAccessTokenForUserIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkloadAccessTokenForUserIdRequest));
            CompletableFuture<GetWorkloadAccessTokenForUserIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a request to an agent or tool hosted in an Amazon Bedrock AgentCore Runtime and receives responses in
     * real-time.
     * </p>
     * <p>
     * To invoke an agent you must specify the AgentCore Runtime ARN and provide a payload containing your request. You
     * can optionally specify a qualifier to target a specific version or endpoint of the agent.
     * </p>
     * <p>
     * This operation supports streaming responses, allowing you to receive partial responses as they become available.
     * We recommend using pagination to ensure that the operation returns quickly and successfully when processing large
     * responses.
     * </p>
     * <p>
     * For example code, see <a
     * href="https://docs.aws.amazon.com/bedrock-agentcore/latest/devguide/runtime-invoke-agent.html">Invoke an
     * AgentCore Runtime agent</a>.
     * </p>
     * <p>
     * If you're integrating your agent with OAuth, you can't use the Amazon Web Services SDK to call
     * <code>InvokeAgentRuntime</code>. Instead, make a HTTPS request to <code>InvokeAgentRuntime</code>. For an
     * example, see <a
     * href="https://docs.aws.amazon.com/bedrock-agentcore/latest/devguide/runtime-oauth.html">Authenticate and
     * authorize with Inbound Auth and Outbound Auth</a>.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:InvokeAgentRuntime</code> permission.
     * </p>
     *
     * @param invokeAgentRuntimeRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The response data from the agent runtime. The format of this data depends on the specific agent
     *        configuration and the requested accept type. For most agents, this is a JSON object containing the agent's
     *        response to the user's request.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>RuntimeClientErrorException The exception that occurs when there is an error in the runtime client.
     *         This can happen due to network issues, invalid configuration, or other client-side problems. Check the
     *         error message for specific details about the error.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.InvokeAgentRuntime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/InvokeAgentRuntime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> invokeAgentRuntime(InvokeAgentRuntimeRequest invokeAgentRuntimeRequest,
            AsyncResponseTransformer<InvokeAgentRuntimeResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeAgentRuntimeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeAgentRuntimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeAgentRuntime");
            Pair<AsyncResponseTransformer<InvokeAgentRuntimeResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<InvokeAgentRuntimeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InvokeAgentRuntimeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeAgentRuntimeRequest, InvokeAgentRuntimeResponse>()
                            .withOperationName("InvokeAgentRuntime").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeAgentRuntimeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncResponseTransformer(asyncResponseTransformer).withInput(invokeAgentRuntimeRequest),
                    asyncResponseTransformer);
            AsyncResponseTransformer<InvokeAgentRuntimeResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<InvokeAgentRuntimeResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Executes code within an active code interpreter session in Amazon Bedrock. This operation processes the provided
     * code, runs it in a secure environment, and returns the execution results including output, errors, and generated
     * visualizations.
     * </p>
     * <p>
     * To execute code, you must specify the code interpreter identifier, session ID, and the code to run in the
     * arguments parameter. The operation returns a stream containing the execution results, which can include text
     * output, error messages, and data visualizations.
     * </p>
     * <p>
     * This operation is subject to request rate limiting based on your account's service quotas.
     * </p>
     * <p>
     * The following operations are related to <code>InvokeCodeInterpreter</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartCodeInterpreterSession.html">StartCodeInterpreterSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetCodeInterpreterSession.html">GetCodeInterpreterSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param invokeCodeInterpreterRequest
     * @return A Java Future containing the result of the InvokeCodeInterpreter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.InvokeCodeInterpreter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/InvokeCodeInterpreter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeCodeInterpreter(InvokeCodeInterpreterRequest invokeCodeInterpreterRequest,
            InvokeCodeInterpreterResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeCodeInterpreterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeCodeInterpreterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeCodeInterpreter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeCodeInterpreterResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeCodeInterpreterResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends CodeInterpreterStreamOutput> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier.builder()
                                    .putSdkPojoSupplier("result", CodeInterpreterStreamOutput::resultBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(CodeInterpreterStreamOutput.UNKNOWN))
                                    .build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "conflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "resourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "serviceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeCodeInterpreterResponse, CodeInterpreterStreamOutput> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeCodeInterpreterResponse, CodeInterpreterStreamOutput> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<InvokeCodeInterpreterResponse, CodeInterpreterStreamOutput> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeCodeInterpreterResponse, CodeInterpreterStreamOutput> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeCodeInterpreterRequest, InvokeCodeInterpreterResponse>()
                            .withOperationName("InvokeCodeInterpreter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeCodeInterpreterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeCodeInterpreterRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all actors in an AgentCore Memory resource. We recommend using pagination to ensure that the operation
     * returns quickly and successfully.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:ListActors</code> permission.
     * </p>
     *
     * @param listActorsRequest
     * @return A Java Future containing the result of the ListActors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListActors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListActors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListActorsResponse> listActors(ListActorsRequest listActorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listActorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListActorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListActorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListActorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListActorsRequest, ListActorsResponse>().withOperationName("ListActors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListActorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listActorsRequest));
            CompletableFuture<ListActorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of browser sessions in Amazon Bedrock that match the specified criteria. This operation returns
     * summary information about each session, including identifiers, status, and timestamps.
     * </p>
     * <p>
     * You can filter the results by browser identifier and session status. The operation supports pagination to handle
     * large result sets efficiently.
     * </p>
     * <p>
     * We recommend using pagination to ensure that the operation returns quickly and successfully when retrieving large
     * numbers of sessions.
     * </p>
     * <p>
     * The following operations are related to <code>ListBrowserSessions</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartBrowserSession.html">StartBrowserSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetBrowserSession.html">GetBrowserSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listBrowserSessionsRequest
     * @return A Java Future containing the result of the ListBrowserSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListBrowserSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListBrowserSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBrowserSessionsResponse> listBrowserSessions(
            ListBrowserSessionsRequest listBrowserSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBrowserSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBrowserSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBrowserSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBrowserSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBrowserSessionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListBrowserSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBrowserSessionsRequest, ListBrowserSessionsResponse>()
                            .withOperationName("ListBrowserSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBrowserSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBrowserSessionsRequest));
            CompletableFuture<ListBrowserSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of code interpreter sessions in Amazon Bedrock that match the specified criteria. This operation
     * returns summary information about each session, including identifiers, status, and timestamps.
     * </p>
     * <p>
     * You can filter the results by code interpreter identifier and session status. The operation supports pagination
     * to handle large result sets efficiently.
     * </p>
     * <p>
     * We recommend using pagination to ensure that the operation returns quickly and successfully when retrieving large
     * numbers of sessions.
     * </p>
     * <p>
     * The following operations are related to <code>ListCodeInterpreterSessions</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartCodeInterpreterSession.html">StartCodeInterpreterSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetCodeInterpreterSession.html">GetCodeInterpreterSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listCodeInterpreterSessionsRequest
     * @return A Java Future containing the result of the ListCodeInterpreterSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListCodeInterpreterSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListCodeInterpreterSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCodeInterpreterSessionsResponse> listCodeInterpreterSessions(
            ListCodeInterpreterSessionsRequest listCodeInterpreterSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCodeInterpreterSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCodeInterpreterSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCodeInterpreterSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCodeInterpreterSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCodeInterpreterSessionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListCodeInterpreterSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCodeInterpreterSessionsRequest, ListCodeInterpreterSessionsResponse>()
                            .withOperationName("ListCodeInterpreterSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCodeInterpreterSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCodeInterpreterSessionsRequest));
            CompletableFuture<ListCodeInterpreterSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists events in an AgentCore Memory resource based on specified criteria. We recommend using pagination to ensure
     * that the operation returns quickly and successfully.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:ListEvents</code> permission.
     * </p>
     *
     * @param listEventsRequest
     * @return A Java Future containing the result of the ListEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventsResponse> listEvents(ListEventsRequest listEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListEventsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventsRequest, ListEventsResponse>().withOperationName("ListEvents")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEventsRequest));
            CompletableFuture<ListEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists memory records in an AgentCore Memory resource based on specified criteria. We recommend using pagination
     * to ensure that the operation returns quickly and successfully.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:ListMemoryRecords</code> permission.
     * </p>
     *
     * @param listMemoryRecordsRequest
     * @return A Java Future containing the result of the ListMemoryRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListMemoryRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListMemoryRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMemoryRecordsResponse> listMemoryRecords(ListMemoryRecordsRequest listMemoryRecordsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMemoryRecordsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMemoryRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemoryRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMemoryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMemoryRecordsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMemoryRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMemoryRecordsRequest, ListMemoryRecordsResponse>()
                            .withOperationName("ListMemoryRecords").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMemoryRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMemoryRecordsRequest));
            CompletableFuture<ListMemoryRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists sessions in an AgentCore Memory resource based on specified criteria. We recommend using pagination to
     * ensure that the operation returns quickly and successfully.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:ListSessions</code> permission.
     * </p>
     *
     * @param listSessionsRequest
     * @return A Java Future containing the result of the ListSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.ListSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/ListSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSessionsResponse> listSessions(ListSessionsRequest listSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSessionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSessionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSessionsRequest, ListSessionsResponse>()
                            .withOperationName("ListSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSessionsRequest));
            CompletableFuture<ListSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for and retrieves memory records from an AgentCore Memory resource based on specified search criteria.
     * We recommend using pagination to ensure that the operation returns quickly and successfully.
     * </p>
     * <p>
     * To use this operation, you must have the <code>bedrock-agentcore:RetrieveMemoryRecords</code> permission.
     * </p>
     *
     * @param retrieveMemoryRecordsRequest
     * @return A Java Future containing the result of the RetrieveMemoryRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>InvalidInputException The input fails to satisfy the constraints specified by AgentCore. Check your
     *         input values and try again.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottledException The request was denied due to request throttling. Reduce the frequency of requests
     *         and try again.</li>
     *         <li>ServiceException The service encountered an internal error. Try your request again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.RetrieveMemoryRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/RetrieveMemoryRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveMemoryRecordsResponse> retrieveMemoryRecords(
            RetrieveMemoryRecordsRequest retrieveMemoryRecordsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveMemoryRecordsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveMemoryRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveMemoryRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveMemoryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetrieveMemoryRecordsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RetrieveMemoryRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveMemoryRecordsRequest, RetrieveMemoryRecordsResponse>()
                            .withOperationName("RetrieveMemoryRecords").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveMemoryRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(retrieveMemoryRecordsRequest));
            CompletableFuture<RetrieveMemoryRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates and initializes a browser session in Amazon Bedrock. The session enables agents to navigate and interact
     * with web content, extract information from websites, and perform web-based tasks as part of their response
     * generation.
     * </p>
     * <p>
     * To create a session, you must specify a browser identifier and a name. You can also configure the viewport
     * dimensions to control the visible area of web content. The session remains active until it times out or you
     * explicitly stop it using the <code>StopBrowserSession</code> operation.
     * </p>
     * <p>
     * The following operations are related to <code>StartBrowserSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetBrowserSession.html">GetBrowserSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_UpdateBrowserStream.html">UpdateBrowserStream</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StopBrowserSession.html">StopBrowserSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startBrowserSessionRequest
     * @return A Java Future containing the result of the StartBrowserSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.StartBrowserSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/StartBrowserSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartBrowserSessionResponse> startBrowserSession(
            StartBrowserSessionRequest startBrowserSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startBrowserSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startBrowserSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartBrowserSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartBrowserSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartBrowserSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartBrowserSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartBrowserSessionRequest, StartBrowserSessionResponse>()
                            .withOperationName("StartBrowserSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartBrowserSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startBrowserSessionRequest));
            CompletableFuture<StartBrowserSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates and initializes a code interpreter session in Amazon Bedrock. The session enables agents to execute code
     * as part of their response generation, supporting programming languages such as Python for data analysis,
     * visualization, and computation tasks.
     * </p>
     * <p>
     * To create a session, you must specify a code interpreter identifier and a name. The session remains active until
     * it times out or you explicitly stop it using the <code>StopCodeInterpreterSession</code> operation.
     * </p>
     * <p>
     * The following operations are related to <code>StartCodeInterpreterSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_InvokeCodeInterpreter.html">InvokeCodeInterpreter</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetCodeInterpreterSession.html">GetCodeInterpreterSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StopCodeInterpreterSession.html">StopCodeInterpreterSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startCodeInterpreterSessionRequest
     * @return A Java Future containing the result of the StartCodeInterpreterSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.StartCodeInterpreterSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/StartCodeInterpreterSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCodeInterpreterSessionResponse> startCodeInterpreterSession(
            StartCodeInterpreterSessionRequest startCodeInterpreterSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCodeInterpreterSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCodeInterpreterSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCodeInterpreterSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCodeInterpreterSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCodeInterpreterSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartCodeInterpreterSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCodeInterpreterSessionRequest, StartCodeInterpreterSessionResponse>()
                            .withOperationName("StartCodeInterpreterSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCodeInterpreterSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCodeInterpreterSessionRequest));
            CompletableFuture<StartCodeInterpreterSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates an active browser session in Amazon Bedrock. This operation stops the session, releases associated
     * resources, and makes the session unavailable for further use.
     * </p>
     * <p>
     * To stop a browser session, you must specify both the browser identifier and the session ID. Once stopped, a
     * session cannot be restarted; you must create a new session using <code>StartBrowserSession</code>.
     * </p>
     * <p>
     * The following operations are related to <code>StopBrowserSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartBrowserSession.html">StartBrowserSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetBrowserSession.html">GetBrowserSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopBrowserSessionRequest
     * @return A Java Future containing the result of the StopBrowserSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.StopBrowserSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/StopBrowserSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopBrowserSessionResponse> stopBrowserSession(StopBrowserSessionRequest stopBrowserSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopBrowserSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopBrowserSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopBrowserSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopBrowserSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopBrowserSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopBrowserSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopBrowserSessionRequest, StopBrowserSessionResponse>()
                            .withOperationName("StopBrowserSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopBrowserSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopBrowserSessionRequest));
            CompletableFuture<StopBrowserSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates an active code interpreter session in Amazon Bedrock. This operation stops the session, releases
     * associated resources, and makes the session unavailable for further use.
     * </p>
     * <p>
     * To stop a code interpreter session, you must specify both the code interpreter identifier and the session ID.
     * Once stopped, a session cannot be restarted; you must create a new session using
     * <code>StartCodeInterpreterSession</code>.
     * </p>
     * <p>
     * The following operations are related to <code>StopCodeInterpreterSession</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_StartCodeInterpreterSession.html">StartCodeInterpreterSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/API_GetCodeInterpreterSession.html">GetCodeInterpreterSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopCodeInterpreterSessionRequest
     * @return A Java Future containing the result of the StopCodeInterpreterSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.StopCodeInterpreterSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/StopCodeInterpreterSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopCodeInterpreterSessionResponse> stopCodeInterpreterSession(
            StopCodeInterpreterSessionRequest stopCodeInterpreterSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopCodeInterpreterSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopCodeInterpreterSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopCodeInterpreterSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopCodeInterpreterSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopCodeInterpreterSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopCodeInterpreterSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopCodeInterpreterSessionRequest, StopCodeInterpreterSessionResponse>()
                            .withOperationName("StopCodeInterpreterSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopCodeInterpreterSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopCodeInterpreterSessionRequest));
            CompletableFuture<StopCodeInterpreterSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a browser stream. To use this operation, you must have permissions to perform the
     * bedrock:UpdateBrowserStream action.
     * </p>
     *
     * @param updateBrowserStreamRequest
     * @return A Java Future containing the result of the UpdateBrowserStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The exception that occurs when the request would cause a service quota
     *         to be exceeded. Review your service quotas and either reduce your request rate or request a quota
     *         increase.</li>
     *         <li>AccessDeniedException The exception that occurs when you do not have sufficient permissions to
     *         perform an action. Verify that your IAM policy includes the necessary permissions for the operation you
     *         are trying to perform.</li>
     *         <li>ConflictException The exception that occurs when the request conflicts with the current state of the
     *         resource. This can happen when trying to modify a resource that is currently being modified by another
     *         request, or when trying to create a resource that already exists.</li>
     *         <li>ValidationException The exception that occurs when the input fails to satisfy the constraints
     *         specified by the service. Check the error message for details about which input parameter is invalid and
     *         correct your request.</li>
     *         <li>ResourceNotFoundException The exception that occurs when the specified resource does not exist. This
     *         can happen when using an invalid identifier or when trying to access a resource that has been deleted.</li>
     *         <li>ThrottlingException The exception that occurs when the request was denied due to request throttling.
     *         This happens when you exceed the allowed request rate for an operation. Reduce the frequency of requests
     *         or implement exponential backoff retry logic in your application.</li>
     *         <li>InternalServerException The exception that occurs when the service encounters an unexpected internal
     *         error. This is a temporary condition that will resolve itself with retries. We recommend implementing
     *         exponential backoff retry logic in your application.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentCoreException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentCoreAsyncClient.UpdateBrowserStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agentcore-2024-02-28/UpdateBrowserStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBrowserStreamResponse> updateBrowserStream(
            UpdateBrowserStreamRequest updateBrowserStreamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBrowserStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBrowserStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock AgentCore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBrowserStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBrowserStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBrowserStreamResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ServiceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ThrottledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "RuntimeClientError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RuntimeClientError").httpStatusCode(424)
                            .exceptionBuilderSupplier(RuntimeClientErrorException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateBrowserStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBrowserStreamRequest, UpdateBrowserStreamResponse>()
                            .withOperationName("UpdateBrowserStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateBrowserStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateBrowserStreamRequest));
            CompletableFuture<UpdateBrowserStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockAgentCoreServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockAgentCoreServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockAgentCoreException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BedrockAgentCoreServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockAgentCoreServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
