/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains aggregated scan results across multiple scan operations, providing a summary of scan status and findings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AggregatedScanResult implements SdkPojo, Serializable,
        ToCopyableBuilder<AggregatedScanResult.Builder, AggregatedScanResult> {
    private static final SdkField<Boolean> FAILED_SCAN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("FailedScan").getter(getter(AggregatedScanResult::failedScan)).setter(setter(Builder::failedScan))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedScan").build()).build();

    private static final SdkField<List<String>> FINDINGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Findings")
            .getter(getter(AggregatedScanResult::findingsAsStrings))
            .setter(setter(Builder::findingsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Findings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_COMPUTED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastComputed").getter(getter(AggregatedScanResult::lastComputed)).setter(setter(Builder::lastComputed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastComputed").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FAILED_SCAN_FIELD,
            FINDINGS_FIELD, LAST_COMPUTED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean failedScan;

    private final List<String> findings;

    private final Instant lastComputed;

    private AggregatedScanResult(BuilderImpl builder) {
        this.failedScan = builder.failedScan;
        this.findings = builder.findings;
        this.lastComputed = builder.lastComputed;
    }

    /**
     * <p>
     * A Boolean value indicating whether any of the aggregated scans failed.
     * </p>
     * 
     * @return A Boolean value indicating whether any of the aggregated scans failed.
     */
    public final Boolean failedScan() {
        return failedScan;
    }

    /**
     * <p>
     * An array of findings discovered across all aggregated scans.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindings} method.
     * </p>
     * 
     * @return An array of findings discovered across all aggregated scans.
     */
    public final List<ScanFinding> findings() {
        return ScanFindingsCopier.copyStringToEnum(findings);
    }

    /**
     * For responses, this returns true if the service returned a value for the Findings property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFindings() {
        return findings != null && !(findings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of findings discovered across all aggregated scans.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindings} method.
     * </p>
     * 
     * @return An array of findings discovered across all aggregated scans.
     */
    public final List<String> findingsAsStrings() {
        return findings;
    }

    /**
     * <p>
     * The timestamp when the aggregated scan result was last computed, in Unix format and Coordinated Universal Time
     * (UTC).
     * </p>
     * 
     * @return The timestamp when the aggregated scan result was last computed, in Unix format and Coordinated Universal
     *         Time (UTC).
     */
    public final Instant lastComputed() {
        return lastComputed;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(failedScan());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindings() ? findingsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastComputed());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AggregatedScanResult)) {
            return false;
        }
        AggregatedScanResult other = (AggregatedScanResult) obj;
        return Objects.equals(failedScan(), other.failedScan()) && hasFindings() == other.hasFindings()
                && Objects.equals(findingsAsStrings(), other.findingsAsStrings())
                && Objects.equals(lastComputed(), other.lastComputed());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AggregatedScanResult").add("FailedScan", failedScan())
                .add("Findings", hasFindings() ? findingsAsStrings() : null).add("LastComputed", lastComputed()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FailedScan":
            return Optional.ofNullable(clazz.cast(failedScan()));
        case "Findings":
            return Optional.ofNullable(clazz.cast(findingsAsStrings()));
        case "LastComputed":
            return Optional.ofNullable(clazz.cast(lastComputed()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FailedScan", FAILED_SCAN_FIELD);
        map.put("Findings", FINDINGS_FIELD);
        map.put("LastComputed", LAST_COMPUTED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AggregatedScanResult, T> g) {
        return obj -> g.apply((AggregatedScanResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AggregatedScanResult> {
        /**
         * <p>
         * A Boolean value indicating whether any of the aggregated scans failed.
         * </p>
         * 
         * @param failedScan
         *        A Boolean value indicating whether any of the aggregated scans failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedScan(Boolean failedScan);

        /**
         * <p>
         * An array of findings discovered across all aggregated scans.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered across all aggregated scans.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingsWithStrings(Collection<String> findings);

        /**
         * <p>
         * An array of findings discovered across all aggregated scans.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered across all aggregated scans.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingsWithStrings(String... findings);

        /**
         * <p>
         * An array of findings discovered across all aggregated scans.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered across all aggregated scans.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findings(Collection<ScanFinding> findings);

        /**
         * <p>
         * An array of findings discovered across all aggregated scans.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered across all aggregated scans.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findings(ScanFinding... findings);

        /**
         * <p>
         * The timestamp when the aggregated scan result was last computed, in Unix format and Coordinated Universal
         * Time (UTC).
         * </p>
         * 
         * @param lastComputed
         *        The timestamp when the aggregated scan result was last computed, in Unix format and Coordinated
         *        Universal Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastComputed(Instant lastComputed);
    }

    static final class BuilderImpl implements Builder {
        private Boolean failedScan;

        private List<String> findings = DefaultSdkAutoConstructList.getInstance();

        private Instant lastComputed;

        private BuilderImpl() {
        }

        private BuilderImpl(AggregatedScanResult model) {
            failedScan(model.failedScan);
            findingsWithStrings(model.findings);
            lastComputed(model.lastComputed);
        }

        public final Boolean getFailedScan() {
            return failedScan;
        }

        public final void setFailedScan(Boolean failedScan) {
            this.failedScan = failedScan;
        }

        @Override
        public final Builder failedScan(Boolean failedScan) {
            this.failedScan = failedScan;
            return this;
        }

        public final Collection<String> getFindings() {
            if (findings instanceof SdkAutoConstructList) {
                return null;
            }
            return findings;
        }

        public final void setFindings(Collection<String> findings) {
            this.findings = ScanFindingsCopier.copy(findings);
        }

        @Override
        public final Builder findingsWithStrings(Collection<String> findings) {
            this.findings = ScanFindingsCopier.copy(findings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingsWithStrings(String... findings) {
            findingsWithStrings(Arrays.asList(findings));
            return this;
        }

        @Override
        public final Builder findings(Collection<ScanFinding> findings) {
            this.findings = ScanFindingsCopier.copyEnumToString(findings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findings(ScanFinding... findings) {
            findings(Arrays.asList(findings));
            return this;
        }

        public final Instant getLastComputed() {
            return lastComputed;
        }

        public final void setLastComputed(Instant lastComputed) {
            this.lastComputed = lastComputed;
        }

        @Override
        public final Builder lastComputed(Instant lastComputed) {
            this.lastComputed = lastComputed;
            return this;
        }

        @Override
        public AggregatedScanResult build() {
            return new AggregatedScanResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
