/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This contains metadata about resource selection for tiering configurations.
 * </p>
 * <p>
 * You can specify up to 5 different resource selections per tiering configuration. Data moved to lower-cost tier
 * remains there until deletion (one-way transition).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceSelection implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceSelection.Builder, ResourceSelection> {
    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(ResourceSelection::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> TIERING_DOWN_SETTINGS_IN_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("TieringDownSettingsInDays")
            .getter(getter(ResourceSelection::tieringDownSettingsInDays)).setter(setter(Builder::tieringDownSettingsInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TieringDownSettingsInDays").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceSelection::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCES_FIELD,
            TIERING_DOWN_SETTINGS_IN_DAYS_FIELD, RESOURCE_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> resources;

    private final Integer tieringDownSettingsInDays;

    private final String resourceType;

    private ResourceSelection(BuilderImpl builder) {
        this.resources = builder.resources;
        this.tieringDownSettingsInDays = builder.tieringDownSettingsInDays;
        this.resourceType = builder.resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that either contains ARNs of the associated resources or contains a wildcard <code>*</code>
     * to specify all resources. You can specify up to 100 specific resources per tiering configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return An array of strings that either contains ARNs of the associated resources or contains a wildcard
     *         <code>*</code> to specify all resources. You can specify up to 100 specific resources per tiering
     *         configuration.
     */
    public final List<String> resources() {
        return resources;
    }

    /**
     * <p>
     * The number of days after creation within a backup vault that an object can transition to the low cost warm
     * storage tier. Must be a positive integer between 60 and 36500 days.
     * </p>
     * 
     * @return The number of days after creation within a backup vault that an object can transition to the low cost
     *         warm storage tier. Must be a positive integer between 60 and 36500 days.
     */
    public final Integer tieringDownSettingsInDays() {
        return tieringDownSettingsInDays;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource; for example, <code>S3</code> for Amazon S3. For tiering configurations,
     * this is currently limited to <code>S3</code>.
     * </p>
     * 
     * @return The type of Amazon Web Services resource; for example, <code>S3</code> for Amazon S3. For tiering
     *         configurations, this is currently limited to <code>S3</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tieringDownSettingsInDays());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceSelection)) {
            return false;
        }
        ResourceSelection other = (ResourceSelection) obj;
        return hasResources() == other.hasResources() && Objects.equals(resources(), other.resources())
                && Objects.equals(tieringDownSettingsInDays(), other.tieringDownSettingsInDays())
                && Objects.equals(resourceType(), other.resourceType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceSelection").add("Resources", hasResources() ? resources() : null)
                .add("TieringDownSettingsInDays", tieringDownSettingsInDays()).add("ResourceType", resourceType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "TieringDownSettingsInDays":
            return Optional.ofNullable(clazz.cast(tieringDownSettingsInDays()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Resources", RESOURCES_FIELD);
        map.put("TieringDownSettingsInDays", TIERING_DOWN_SETTINGS_IN_DAYS_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceSelection, T> g) {
        return obj -> g.apply((ResourceSelection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceSelection> {
        /**
         * <p>
         * An array of strings that either contains ARNs of the associated resources or contains a wildcard
         * <code>*</code> to specify all resources. You can specify up to 100 specific resources per tiering
         * configuration.
         * </p>
         * 
         * @param resources
         *        An array of strings that either contains ARNs of the associated resources or contains a wildcard
         *        <code>*</code> to specify all resources. You can specify up to 100 specific resources per tiering
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * An array of strings that either contains ARNs of the associated resources or contains a wildcard
         * <code>*</code> to specify all resources. You can specify up to 100 specific resources per tiering
         * configuration.
         * </p>
         * 
         * @param resources
         *        An array of strings that either contains ARNs of the associated resources or contains a wildcard
         *        <code>*</code> to specify all resources. You can specify up to 100 specific resources per tiering
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * The number of days after creation within a backup vault that an object can transition to the low cost warm
         * storage tier. Must be a positive integer between 60 and 36500 days.
         * </p>
         * 
         * @param tieringDownSettingsInDays
         *        The number of days after creation within a backup vault that an object can transition to the low cost
         *        warm storage tier. Must be a positive integer between 60 and 36500 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tieringDownSettingsInDays(Integer tieringDownSettingsInDays);

        /**
         * <p>
         * The type of Amazon Web Services resource; for example, <code>S3</code> for Amazon S3. For tiering
         * configurations, this is currently limited to <code>S3</code>.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource; for example, <code>S3</code> for Amazon S3. For tiering
         *        configurations, this is currently limited to <code>S3</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);
    }

    static final class BuilderImpl implements Builder {
        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private Integer tieringDownSettingsInDays;

        private String resourceType;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceSelection model) {
            resources(model.resources);
            tieringDownSettingsInDays(model.tieringDownSettingsInDays);
            resourceType(model.resourceType);
        }

        public final Collection<String> getResources() {
            if (resources instanceof SdkAutoConstructList) {
                return null;
            }
            return resources;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final Integer getTieringDownSettingsInDays() {
            return tieringDownSettingsInDays;
        }

        public final void setTieringDownSettingsInDays(Integer tieringDownSettingsInDays) {
            this.tieringDownSettingsInDays = tieringDownSettingsInDays;
        }

        @Override
        public final Builder tieringDownSettingsInDays(Integer tieringDownSettingsInDays) {
            this.tieringDownSettingsInDays = tieringDownSettingsInDays;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public ResourceSelection build() {
            return new ResourceSelection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
