/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a validation rule that modifies the allowed code values for a specific X12 element. This rule allows you to
 * add or remove valid codes from an element's standard code list, providing flexibility to accommodate trading
 * partner-specific requirements or industry variations. You can specify codes to add to expand the allowed values
 * beyond the X12 standard, or codes to remove to restrict the allowed values for stricter validation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class X12CodeListValidationRule implements SdkPojo, Serializable,
        ToCopyableBuilder<X12CodeListValidationRule.Builder, X12CodeListValidationRule> {
    private static final SdkField<String> ELEMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("elementId").getter(getter(X12CodeListValidationRule::elementId)).setter(setter(Builder::elementId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("elementId").build()).build();

    private static final SdkField<List<String>> CODES_TO_ADD_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("codesToAdd")
            .getter(getter(X12CodeListValidationRule::codesToAdd))
            .setter(setter(Builder::codesToAdd))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("codesToAdd").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CODES_TO_REMOVE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("codesToRemove")
            .getter(getter(X12CodeListValidationRule::codesToRemove))
            .setter(setter(Builder::codesToRemove))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("codesToRemove").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ELEMENT_ID_FIELD,
            CODES_TO_ADD_FIELD, CODES_TO_REMOVE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String elementId;

    private final List<String> codesToAdd;

    private final List<String> codesToRemove;

    private X12CodeListValidationRule(BuilderImpl builder) {
        this.elementId = builder.elementId;
        this.codesToAdd = builder.codesToAdd;
        this.codesToRemove = builder.codesToRemove;
    }

    /**
     * <p>
     * Specifies the four-digit element ID to which the code list modifications apply. This identifies which X12 element
     * will have its allowed code values modified.
     * </p>
     * 
     * @return Specifies the four-digit element ID to which the code list modifications apply. This identifies which X12
     *         element will have its allowed code values modified.
     */
    public final String elementId() {
        return elementId;
    }

    /**
     * For responses, this returns true if the service returned a value for the CodesToAdd property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCodesToAdd() {
        return codesToAdd != null && !(codesToAdd instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies a list of code values to add to the element's allowed values. These codes will be considered valid for
     * the specified element in addition to the standard codes defined by the X12 specification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCodesToAdd} method.
     * </p>
     * 
     * @return Specifies a list of code values to add to the element's allowed values. These codes will be considered
     *         valid for the specified element in addition to the standard codes defined by the X12 specification.
     */
    public final List<String> codesToAdd() {
        return codesToAdd;
    }

    /**
     * For responses, this returns true if the service returned a value for the CodesToRemove property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCodesToRemove() {
        return codesToRemove != null && !(codesToRemove instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies a list of code values to remove from the element's allowed values. These codes will be considered
     * invalid for the specified element, even if they are part of the standard codes defined by the X12 specification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCodesToRemove} method.
     * </p>
     * 
     * @return Specifies a list of code values to remove from the element's allowed values. These codes will be
     *         considered invalid for the specified element, even if they are part of the standard codes defined by the
     *         X12 specification.
     */
    public final List<String> codesToRemove() {
        return codesToRemove;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(elementId());
        hashCode = 31 * hashCode + Objects.hashCode(hasCodesToAdd() ? codesToAdd() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCodesToRemove() ? codesToRemove() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof X12CodeListValidationRule)) {
            return false;
        }
        X12CodeListValidationRule other = (X12CodeListValidationRule) obj;
        return Objects.equals(elementId(), other.elementId()) && hasCodesToAdd() == other.hasCodesToAdd()
                && Objects.equals(codesToAdd(), other.codesToAdd()) && hasCodesToRemove() == other.hasCodesToRemove()
                && Objects.equals(codesToRemove(), other.codesToRemove());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("X12CodeListValidationRule").add("ElementId", elementId())
                .add("CodesToAdd", hasCodesToAdd() ? codesToAdd() : null)
                .add("CodesToRemove", hasCodesToRemove() ? codesToRemove() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "elementId":
            return Optional.ofNullable(clazz.cast(elementId()));
        case "codesToAdd":
            return Optional.ofNullable(clazz.cast(codesToAdd()));
        case "codesToRemove":
            return Optional.ofNullable(clazz.cast(codesToRemove()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("elementId", ELEMENT_ID_FIELD);
        map.put("codesToAdd", CODES_TO_ADD_FIELD);
        map.put("codesToRemove", CODES_TO_REMOVE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<X12CodeListValidationRule, T> g) {
        return obj -> g.apply((X12CodeListValidationRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, X12CodeListValidationRule> {
        /**
         * <p>
         * Specifies the four-digit element ID to which the code list modifications apply. This identifies which X12
         * element will have its allowed code values modified.
         * </p>
         * 
         * @param elementId
         *        Specifies the four-digit element ID to which the code list modifications apply. This identifies which
         *        X12 element will have its allowed code values modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elementId(String elementId);

        /**
         * <p>
         * Specifies a list of code values to add to the element's allowed values. These codes will be considered valid
         * for the specified element in addition to the standard codes defined by the X12 specification.
         * </p>
         * 
         * @param codesToAdd
         *        Specifies a list of code values to add to the element's allowed values. These codes will be considered
         *        valid for the specified element in addition to the standard codes defined by the X12 specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codesToAdd(Collection<String> codesToAdd);

        /**
         * <p>
         * Specifies a list of code values to add to the element's allowed values. These codes will be considered valid
         * for the specified element in addition to the standard codes defined by the X12 specification.
         * </p>
         * 
         * @param codesToAdd
         *        Specifies a list of code values to add to the element's allowed values. These codes will be considered
         *        valid for the specified element in addition to the standard codes defined by the X12 specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codesToAdd(String... codesToAdd);

        /**
         * <p>
         * Specifies a list of code values to remove from the element's allowed values. These codes will be considered
         * invalid for the specified element, even if they are part of the standard codes defined by the X12
         * specification.
         * </p>
         * 
         * @param codesToRemove
         *        Specifies a list of code values to remove from the element's allowed values. These codes will be
         *        considered invalid for the specified element, even if they are part of the standard codes defined by
         *        the X12 specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codesToRemove(Collection<String> codesToRemove);

        /**
         * <p>
         * Specifies a list of code values to remove from the element's allowed values. These codes will be considered
         * invalid for the specified element, even if they are part of the standard codes defined by the X12
         * specification.
         * </p>
         * 
         * @param codesToRemove
         *        Specifies a list of code values to remove from the element's allowed values. These codes will be
         *        considered invalid for the specified element, even if they are part of the standard codes defined by
         *        the X12 specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codesToRemove(String... codesToRemove);
    }

    static final class BuilderImpl implements Builder {
        private String elementId;

        private List<String> codesToAdd = DefaultSdkAutoConstructList.getInstance();

        private List<String> codesToRemove = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(X12CodeListValidationRule model) {
            elementId(model.elementId);
            codesToAdd(model.codesToAdd);
            codesToRemove(model.codesToRemove);
        }

        public final String getElementId() {
            return elementId;
        }

        public final void setElementId(String elementId) {
            this.elementId = elementId;
        }

        @Override
        public final Builder elementId(String elementId) {
            this.elementId = elementId;
            return this;
        }

        public final Collection<String> getCodesToAdd() {
            if (codesToAdd instanceof SdkAutoConstructList) {
                return null;
            }
            return codesToAdd;
        }

        public final void setCodesToAdd(Collection<String> codesToAdd) {
            this.codesToAdd = CodeListCopier.copy(codesToAdd);
        }

        @Override
        public final Builder codesToAdd(Collection<String> codesToAdd) {
            this.codesToAdd = CodeListCopier.copy(codesToAdd);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder codesToAdd(String... codesToAdd) {
            codesToAdd(Arrays.asList(codesToAdd));
            return this;
        }

        public final Collection<String> getCodesToRemove() {
            if (codesToRemove instanceof SdkAutoConstructList) {
                return null;
            }
            return codesToRemove;
        }

        public final void setCodesToRemove(Collection<String> codesToRemove) {
            this.codesToRemove = CodeListCopier.copy(codesToRemove);
        }

        @Override
        public final Builder codesToRemove(Collection<String> codesToRemove) {
            this.codesToRemove = CodeListCopier.copy(codesToRemove);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder codesToRemove(String... codesToRemove) {
            codesToRemove(Arrays.asList(codesToRemove));
            return this;
        }

        @Override
        public X12CodeListValidationRule build() {
            return new X12CodeListValidationRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
