/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a scaling instruction for a scalable resource.
 * </p>
 * <p>
 * The scaling instruction is used in combination with a scaling plan, which is a set of instructions for configuring
 * dynamic scaling and predictive scaling for the scalable resources in your application. Each scaling instruction
 * applies to one resource.
 * </p>
 * <p>
 * AWS Auto Scaling creates target tracking scaling policies based on the scaling instructions. Target tracking scaling
 * policies adjust the capacity of your scalable resource as required to maintain resource utilization at the target
 * value that you specified.
 * </p>
 * <p>
 * AWS Auto Scaling also configures predictive scaling for your Amazon EC2 Auto Scaling groups using a subset of
 * parameters, including the load metric, the scaling metric, the target value for the scaling metric, the predictive
 * scaling mode (forecast and scale or forecast only), and the desired behavior when the forecast capacity exceeds the
 * maximum capacity of the resource. With predictive scaling, AWS Auto Scaling generates forecasts with traffic
 * predictions for the two days ahead and schedules scaling actions that proactively add and remove resource capacity to
 * match the forecast.
 * </p>
 * <p>
 * We recommend waiting a minimum of 24 hours after creating an Auto Scaling group to configure predictive scaling. At
 * minimum, there must be 24 hours of historical data to generate a forecast.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/autoscaling/plans/userguide/auto-scaling-getting-started.html">Getting Started with
 * AWS Auto Scaling</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingInstruction implements SdkPojo, Serializable,
        ToCopyableBuilder<ScalingInstruction.Builder, ScalingInstruction> {
    private static final SdkField<String> SERVICE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::serviceNamespaceAsString)).setter(setter(Builder::serviceNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceNamespace").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SCALABLE_DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::scalableDimensionAsString)).setter(setter(Builder::scalableDimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalableDimension").build()).build();

    private static final SdkField<Integer> MIN_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ScalingInstruction::minCapacity)).setter(setter(Builder::minCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinCapacity").build()).build();

    private static final SdkField<Integer> MAX_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ScalingInstruction::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<List<TargetTrackingConfiguration>> TARGET_TRACKING_CONFIGURATIONS_FIELD = SdkField
            .<List<TargetTrackingConfiguration>> builder(MarshallingType.LIST)
            .getter(getter(ScalingInstruction::targetTrackingConfigurations))
            .setter(setter(Builder::targetTrackingConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetTrackingConfigurations")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetTrackingConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetTrackingConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PredefinedLoadMetricSpecification> PREDEFINED_LOAD_METRIC_SPECIFICATION_FIELD = SdkField
            .<PredefinedLoadMetricSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(ScalingInstruction::predefinedLoadMetricSpecification))
            .setter(setter(Builder::predefinedLoadMetricSpecification))
            .constructor(PredefinedLoadMetricSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredefinedLoadMetricSpecification")
                    .build()).build();

    private static final SdkField<CustomizedLoadMetricSpecification> CUSTOMIZED_LOAD_METRIC_SPECIFICATION_FIELD = SdkField
            .<CustomizedLoadMetricSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(ScalingInstruction::customizedLoadMetricSpecification))
            .setter(setter(Builder::customizedLoadMetricSpecification))
            .constructor(CustomizedLoadMetricSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomizedLoadMetricSpecification")
                    .build()).build();

    private static final SdkField<Integer> SCHEDULED_ACTION_BUFFER_TIME_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).getter(getter(ScalingInstruction::scheduledActionBufferTime))
            .setter(setter(Builder::scheduledActionBufferTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduledActionBufferTime").build())
            .build();

    private static final SdkField<String> PREDICTIVE_SCALING_MAX_CAPACITY_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::predictiveScalingMaxCapacityBehaviorAsString))
            .setter(setter(Builder::predictiveScalingMaxCapacityBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PredictiveScalingMaxCapacityBehavior").build()).build();

    private static final SdkField<Integer> PREDICTIVE_SCALING_MAX_CAPACITY_BUFFER_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ScalingInstruction::predictiveScalingMaxCapacityBuffer))
            .setter(setter(Builder::predictiveScalingMaxCapacityBuffer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredictiveScalingMaxCapacityBuffer")
                    .build()).build();

    private static final SdkField<String> PREDICTIVE_SCALING_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::predictiveScalingModeAsString)).setter(setter(Builder::predictiveScalingMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredictiveScalingMode").build())
            .build();

    private static final SdkField<String> SCALING_POLICY_UPDATE_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingInstruction::scalingPolicyUpdateBehaviorAsString))
            .setter(setter(Builder::scalingPolicyUpdateBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPolicyUpdateBehavior")
                    .build()).build();

    private static final SdkField<Boolean> DISABLE_DYNAMIC_SCALING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ScalingInstruction::disableDynamicScaling)).setter(setter(Builder::disableDynamicScaling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableDynamicScaling").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_NAMESPACE_FIELD,
            RESOURCE_ID_FIELD, SCALABLE_DIMENSION_FIELD, MIN_CAPACITY_FIELD, MAX_CAPACITY_FIELD,
            TARGET_TRACKING_CONFIGURATIONS_FIELD, PREDEFINED_LOAD_METRIC_SPECIFICATION_FIELD,
            CUSTOMIZED_LOAD_METRIC_SPECIFICATION_FIELD, SCHEDULED_ACTION_BUFFER_TIME_FIELD,
            PREDICTIVE_SCALING_MAX_CAPACITY_BEHAVIOR_FIELD, PREDICTIVE_SCALING_MAX_CAPACITY_BUFFER_FIELD,
            PREDICTIVE_SCALING_MODE_FIELD, SCALING_POLICY_UPDATE_BEHAVIOR_FIELD, DISABLE_DYNAMIC_SCALING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String serviceNamespace;

    private final String resourceId;

    private final String scalableDimension;

    private final Integer minCapacity;

    private final Integer maxCapacity;

    private final List<TargetTrackingConfiguration> targetTrackingConfigurations;

    private final PredefinedLoadMetricSpecification predefinedLoadMetricSpecification;

    private final CustomizedLoadMetricSpecification customizedLoadMetricSpecification;

    private final Integer scheduledActionBufferTime;

    private final String predictiveScalingMaxCapacityBehavior;

    private final Integer predictiveScalingMaxCapacityBuffer;

    private final String predictiveScalingMode;

    private final String scalingPolicyUpdateBehavior;

    private final Boolean disableDynamicScaling;

    private ScalingInstruction(BuilderImpl builder) {
        this.serviceNamespace = builder.serviceNamespace;
        this.resourceId = builder.resourceId;
        this.scalableDimension = builder.scalableDimension;
        this.minCapacity = builder.minCapacity;
        this.maxCapacity = builder.maxCapacity;
        this.targetTrackingConfigurations = builder.targetTrackingConfigurations;
        this.predefinedLoadMetricSpecification = builder.predefinedLoadMetricSpecification;
        this.customizedLoadMetricSpecification = builder.customizedLoadMetricSpecification;
        this.scheduledActionBufferTime = builder.scheduledActionBufferTime;
        this.predictiveScalingMaxCapacityBehavior = builder.predictiveScalingMaxCapacityBehavior;
        this.predictiveScalingMaxCapacityBuffer = builder.predictiveScalingMaxCapacityBuffer;
        this.predictiveScalingMode = builder.predictiveScalingMode;
        this.scalingPolicyUpdateBehavior = builder.scalingPolicyUpdateBehavior;
        this.disableDynamicScaling = builder.disableDynamicScaling;
    }

    /**
     * <p>
     * The namespace of the AWS service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service.
     * @see ServiceNamespace
     */
    public ServiceNamespace serviceNamespace() {
        return ServiceNamespace.fromValue(serviceNamespace);
    }

    /**
     * <p>
     * The namespace of the AWS service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service.
     * @see ServiceNamespace
     */
    public String serviceNamespaceAsString() {
        return serviceNamespace;
    }

    /**
     * <p>
     * The ID of the resource. This string consists of the resource type and unique identifier.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the name of
     * the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and service
     * name. Example: <code>service/default/sample-webapp</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot
     * Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID. Example:
     * <code>table/my-table</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
     * resource ID. Example: <code>table/my-table/index/my-table-index</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:my-db-cluster</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The ID of the resource. This string consists of the resource type and unique identifier.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the
     *         name of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
     *         service name. Example: <code>service/default/sample-webapp</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
     *         the Spot Fleet request ID. Example:
     *         <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
     *         Example: <code>table/my-table</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
     *         the resource ID. Example: <code>table/my-table/index/my-table-index</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:my-db-cluster</code>.
     *         </p>
     *         </li>
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The scalable dimension associated with the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension associated with the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public ScalableDimension scalableDimension() {
        return ScalableDimension.fromValue(scalableDimension);
    }

    /**
     * <p>
     * The scalable dimension associated with the resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension associated with the resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public String scalableDimensionAsString() {
        return scalableDimension;
    }

    /**
     * <p>
     * The minimum capacity of the resource.
     * </p>
     * 
     * @return The minimum capacity of the resource.
     */
    public Integer minCapacity() {
        return minCapacity;
    }

    /**
     * <p>
     * The maximum capacity of the resource. The exception to this upper limit is if you specify a non-default setting
     * for <b>PredictiveScalingMaxCapacityBehavior</b>.
     * </p>
     * 
     * @return The maximum capacity of the resource. The exception to this upper limit is if you specify a non-default
     *         setting for <b>PredictiveScalingMaxCapacityBehavior</b>.
     */
    public Integer maxCapacity() {
        return maxCapacity;
    }

    /**
     * Returns true if the TargetTrackingConfigurations property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasTargetTrackingConfigurations() {
        return targetTrackingConfigurations != null && !(targetTrackingConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The structure that defines new target tracking configurations (up to 10). Each of these structures includes a
     * specific scaling metric and a target value for the metric, along with various parameters to use with dynamic
     * scaling.
     * </p>
     * <p>
     * With predictive scaling and dynamic scaling, the resource scales based on the target tracking configuration that
     * provides the largest capacity for both scale in and scale out.
     * </p>
     * <p>
     * Condition: The scaling metric must be unique across target tracking configurations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetTrackingConfigurations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The structure that defines new target tracking configurations (up to 10). Each of these structures
     *         includes a specific scaling metric and a target value for the metric, along with various parameters to
     *         use with dynamic scaling. </p>
     *         <p>
     *         With predictive scaling and dynamic scaling, the resource scales based on the target tracking
     *         configuration that provides the largest capacity for both scale in and scale out.
     *         </p>
     *         <p>
     *         Condition: The scaling metric must be unique across target tracking configurations.
     */
    public List<TargetTrackingConfiguration> targetTrackingConfigurations() {
        return targetTrackingConfigurations;
    }

    /**
     * <p>
     * The predefined load metric to use for predictive scaling. This parameter or a
     * <b>CustomizedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
     * otherwise.
     * </p>
     * 
     * @return The predefined load metric to use for predictive scaling. This parameter or a
     *         <b>CustomizedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be
     *         used otherwise.
     */
    public PredefinedLoadMetricSpecification predefinedLoadMetricSpecification() {
        return predefinedLoadMetricSpecification;
    }

    /**
     * <p>
     * The customized load metric to use for predictive scaling. This parameter or a
     * <b>PredefinedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
     * otherwise.
     * </p>
     * 
     * @return The customized load metric to use for predictive scaling. This parameter or a
     *         <b>PredefinedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be
     *         used otherwise.
     */
    public CustomizedLoadMetricSpecification customizedLoadMetricSpecification() {
        return customizedLoadMetricSpecification;
    }

    /**
     * <p>
     * The amount of time, in seconds, to buffer the run time of scheduled scaling actions when scaling out. For
     * example, if the forecast says to add capacity at 10:00 AM, and the buffer time is 5 minutes, then the run time of
     * the corresponding scheduled scaling action will be 9:55 AM. The intention is to give resources time to be
     * provisioned. For example, it can take a few minutes to launch an EC2 instance. The actual amount of time required
     * depends on several factors, such as the size of the instance and whether there are startup scripts to complete.
     * </p>
     * <p>
     * The value must be less than the forecast interval duration of 3600 seconds (60 minutes). The default is 300
     * seconds.
     * </p>
     * <p>
     * Only valid when configuring predictive scaling.
     * </p>
     * 
     * @return The amount of time, in seconds, to buffer the run time of scheduled scaling actions when scaling out. For
     *         example, if the forecast says to add capacity at 10:00 AM, and the buffer time is 5 minutes, then the run
     *         time of the corresponding scheduled scaling action will be 9:55 AM. The intention is to give resources
     *         time to be provisioned. For example, it can take a few minutes to launch an EC2 instance. The actual
     *         amount of time required depends on several factors, such as the size of the instance and whether there
     *         are startup scripts to complete. </p>
     *         <p>
     *         The value must be less than the forecast interval duration of 3600 seconds (60 minutes). The default is
     *         300 seconds.
     *         </p>
     *         <p>
     *         Only valid when configuring predictive scaling.
     */
    public Integer scheduledActionBufferTime() {
        return scheduledActionBufferTime;
    }

    /**
     * <p>
     * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity
     * specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>.
     * </p>
     * <p>
     * The following are possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher than the
     * maximum capacity. The maximum capacity is enforced as a hard limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
     * maximum capacity to equal but not exceed forecast capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
     * maximum capacity by a specified buffer value. The intention is to give the target tracking scaling policy extra
     * capacity if unexpected traffic occurs.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Only valid when configuring predictive scaling.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #predictiveScalingMaxCapacityBehavior} will return
     * {@link PredictiveScalingMaxCapacityBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #predictiveScalingMaxCapacityBehaviorAsString}.
     * </p>
     * 
     * @return Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
     *         capacity specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>
     *         .</p>
     *         <p>
     *         The following are possible values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher
     *         than the maximum capacity. The maximum capacity is enforced as a hard limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than
     *         the maximum capacity to equal but not exceed forecast capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
     *         than the maximum capacity by a specified buffer value. The intention is to give the target tracking
     *         scaling policy extra capacity if unexpected traffic occurs.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Only valid when configuring predictive scaling.
     * @see PredictiveScalingMaxCapacityBehavior
     */
    public PredictiveScalingMaxCapacityBehavior predictiveScalingMaxCapacityBehavior() {
        return PredictiveScalingMaxCapacityBehavior.fromValue(predictiveScalingMaxCapacityBehavior);
    }

    /**
     * <p>
     * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity
     * specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>.
     * </p>
     * <p>
     * The following are possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher than the
     * maximum capacity. The maximum capacity is enforced as a hard limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
     * maximum capacity to equal but not exceed forecast capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
     * maximum capacity by a specified buffer value. The intention is to give the target tracking scaling policy extra
     * capacity if unexpected traffic occurs.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Only valid when configuring predictive scaling.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #predictiveScalingMaxCapacityBehavior} will return
     * {@link PredictiveScalingMaxCapacityBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #predictiveScalingMaxCapacityBehaviorAsString}.
     * </p>
     * 
     * @return Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
     *         capacity specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>
     *         .</p>
     *         <p>
     *         The following are possible values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher
     *         than the maximum capacity. The maximum capacity is enforced as a hard limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than
     *         the maximum capacity to equal but not exceed forecast capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
     *         than the maximum capacity by a specified buffer value. The intention is to give the target tracking
     *         scaling policy extra capacity if unexpected traffic occurs.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Only valid when configuring predictive scaling.
     * @see PredictiveScalingMaxCapacityBehavior
     */
    public String predictiveScalingMaxCapacityBehaviorAsString() {
        return predictiveScalingMaxCapacityBehavior;
    }

    /**
     * <p>
     * The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum capacity.
     * The value is specified as a percentage relative to the forecast capacity. For example, if the buffer is 10, this
     * means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum capacity is 40, then the
     * effective maximum capacity is 55.
     * </p>
     * <p>
     * Only valid when configuring predictive scaling. Required if the <b>PredictiveScalingMaxCapacityBehavior</b> is
     * set to <code>SetMaxCapacityAboveForecastCapacity</code>, and cannot be used otherwise.
     * </p>
     * <p>
     * The range is 1-100.
     * </p>
     * 
     * @return The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
     *         capacity. The value is specified as a percentage relative to the forecast capacity. For example, if the
     *         buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum
     *         capacity is 40, then the effective maximum capacity is 55.</p>
     *         <p>
     *         Only valid when configuring predictive scaling. Required if the
     *         <b>PredictiveScalingMaxCapacityBehavior</b> is set to <code>SetMaxCapacityAboveForecastCapacity</code>,
     *         and cannot be used otherwise.
     *         </p>
     *         <p>
     *         The range is 1-100.
     */
    public Integer predictiveScalingMaxCapacityBuffer() {
        return predictiveScalingMaxCapacityBuffer;
    }

    /**
     * <p>
     * The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto Scaling
     * forecasts capacity but does not create any scheduled scaling actions based on the capacity forecast.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #predictiveScalingMode} will return {@link PredictiveScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #predictiveScalingModeAsString}.
     * </p>
     * 
     * @return The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto
     *         Scaling forecasts capacity but does not create any scheduled scaling actions based on the capacity
     *         forecast.
     * @see PredictiveScalingMode
     */
    public PredictiveScalingMode predictiveScalingMode() {
        return PredictiveScalingMode.fromValue(predictiveScalingMode);
    }

    /**
     * <p>
     * The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto Scaling
     * forecasts capacity but does not create any scheduled scaling actions based on the capacity forecast.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #predictiveScalingMode} will return {@link PredictiveScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #predictiveScalingModeAsString}.
     * </p>
     * 
     * @return The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto
     *         Scaling forecasts capacity but does not create any scheduled scaling actions based on the capacity
     *         forecast.
     * @see PredictiveScalingMode
     */
    public String predictiveScalingModeAsString() {
        return predictiveScalingMode;
    }

    /**
     * <p>
     * Controls whether a resource's externally created scaling policies are kept or replaced.
     * </p>
     * <p>
     * The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
     * <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are deleted and
     * new target tracking scaling policies created.
     * </p>
     * <p>
     * Only valid when configuring dynamic scaling.
     * </p>
     * <p>
     * Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are more
     * than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create new ones.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingPolicyUpdateBehavior} will return {@link ScalingPolicyUpdateBehavior#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #scalingPolicyUpdateBehaviorAsString}.
     * </p>
     * 
     * @return Controls whether a resource's externally created scaling policies are kept or replaced. </p>
     *         <p>
     *         The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
     *         <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are
     *         deleted and new target tracking scaling policies created.
     *         </p>
     *         <p>
     *         Only valid when configuring dynamic scaling.
     *         </p>
     *         <p>
     *         Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are
     *         more than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create
     *         new ones.
     * @see ScalingPolicyUpdateBehavior
     */
    public ScalingPolicyUpdateBehavior scalingPolicyUpdateBehavior() {
        return ScalingPolicyUpdateBehavior.fromValue(scalingPolicyUpdateBehavior);
    }

    /**
     * <p>
     * Controls whether a resource's externally created scaling policies are kept or replaced.
     * </p>
     * <p>
     * The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
     * <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are deleted and
     * new target tracking scaling policies created.
     * </p>
     * <p>
     * Only valid when configuring dynamic scaling.
     * </p>
     * <p>
     * Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are more
     * than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create new ones.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingPolicyUpdateBehavior} will return {@link ScalingPolicyUpdateBehavior#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #scalingPolicyUpdateBehaviorAsString}.
     * </p>
     * 
     * @return Controls whether a resource's externally created scaling policies are kept or replaced. </p>
     *         <p>
     *         The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
     *         <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are
     *         deleted and new target tracking scaling policies created.
     *         </p>
     *         <p>
     *         Only valid when configuring dynamic scaling.
     *         </p>
     *         <p>
     *         Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are
     *         more than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create
     *         new ones.
     * @see ScalingPolicyUpdateBehavior
     */
    public String scalingPolicyUpdateBehaviorAsString() {
        return scalingPolicyUpdateBehavior;
    }

    /**
     * <p>
     * Controls whether dynamic scaling by AWS Auto Scaling is disabled. When dynamic scaling is enabled, AWS Auto
     * Scaling creates target tracking scaling policies based on the specified target tracking configurations.
     * </p>
     * <p>
     * The default is enabled (<code>false</code>).
     * </p>
     * 
     * @return Controls whether dynamic scaling by AWS Auto Scaling is disabled. When dynamic scaling is enabled, AWS
     *         Auto Scaling creates target tracking scaling policies based on the specified target tracking
     *         configurations. </p>
     *         <p>
     *         The default is enabled (<code>false</code>).
     */
    public Boolean disableDynamicScaling() {
        return disableDynamicScaling;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceNamespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(scalableDimensionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(minCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(targetTrackingConfigurations());
        hashCode = 31 * hashCode + Objects.hashCode(predefinedLoadMetricSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(customizedLoadMetricSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(scheduledActionBufferTime());
        hashCode = 31 * hashCode + Objects.hashCode(predictiveScalingMaxCapacityBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(predictiveScalingMaxCapacityBuffer());
        hashCode = 31 * hashCode + Objects.hashCode(predictiveScalingModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scalingPolicyUpdateBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(disableDynamicScaling());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingInstruction)) {
            return false;
        }
        ScalingInstruction other = (ScalingInstruction) obj;
        return Objects.equals(serviceNamespaceAsString(), other.serviceNamespaceAsString())
                && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(scalableDimensionAsString(), other.scalableDimensionAsString())
                && Objects.equals(minCapacity(), other.minCapacity())
                && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(targetTrackingConfigurations(), other.targetTrackingConfigurations())
                && Objects.equals(predefinedLoadMetricSpecification(), other.predefinedLoadMetricSpecification())
                && Objects.equals(customizedLoadMetricSpecification(), other.customizedLoadMetricSpecification())
                && Objects.equals(scheduledActionBufferTime(), other.scheduledActionBufferTime())
                && Objects.equals(predictiveScalingMaxCapacityBehaviorAsString(),
                        other.predictiveScalingMaxCapacityBehaviorAsString())
                && Objects.equals(predictiveScalingMaxCapacityBuffer(), other.predictiveScalingMaxCapacityBuffer())
                && Objects.equals(predictiveScalingModeAsString(), other.predictiveScalingModeAsString())
                && Objects.equals(scalingPolicyUpdateBehaviorAsString(), other.scalingPolicyUpdateBehaviorAsString())
                && Objects.equals(disableDynamicScaling(), other.disableDynamicScaling());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ScalingInstruction").add("ServiceNamespace", serviceNamespaceAsString())
                .add("ResourceId", resourceId()).add("ScalableDimension", scalableDimensionAsString())
                .add("MinCapacity", minCapacity()).add("MaxCapacity", maxCapacity())
                .add("TargetTrackingConfigurations", targetTrackingConfigurations())
                .add("PredefinedLoadMetricSpecification", predefinedLoadMetricSpecification())
                .add("CustomizedLoadMetricSpecification", customizedLoadMetricSpecification())
                .add("ScheduledActionBufferTime", scheduledActionBufferTime())
                .add("PredictiveScalingMaxCapacityBehavior", predictiveScalingMaxCapacityBehaviorAsString())
                .add("PredictiveScalingMaxCapacityBuffer", predictiveScalingMaxCapacityBuffer())
                .add("PredictiveScalingMode", predictiveScalingModeAsString())
                .add("ScalingPolicyUpdateBehavior", scalingPolicyUpdateBehaviorAsString())
                .add("DisableDynamicScaling", disableDynamicScaling()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServiceNamespace":
            return Optional.ofNullable(clazz.cast(serviceNamespaceAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ScalableDimension":
            return Optional.ofNullable(clazz.cast(scalableDimensionAsString()));
        case "MinCapacity":
            return Optional.ofNullable(clazz.cast(minCapacity()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "TargetTrackingConfigurations":
            return Optional.ofNullable(clazz.cast(targetTrackingConfigurations()));
        case "PredefinedLoadMetricSpecification":
            return Optional.ofNullable(clazz.cast(predefinedLoadMetricSpecification()));
        case "CustomizedLoadMetricSpecification":
            return Optional.ofNullable(clazz.cast(customizedLoadMetricSpecification()));
        case "ScheduledActionBufferTime":
            return Optional.ofNullable(clazz.cast(scheduledActionBufferTime()));
        case "PredictiveScalingMaxCapacityBehavior":
            return Optional.ofNullable(clazz.cast(predictiveScalingMaxCapacityBehaviorAsString()));
        case "PredictiveScalingMaxCapacityBuffer":
            return Optional.ofNullable(clazz.cast(predictiveScalingMaxCapacityBuffer()));
        case "PredictiveScalingMode":
            return Optional.ofNullable(clazz.cast(predictiveScalingModeAsString()));
        case "ScalingPolicyUpdateBehavior":
            return Optional.ofNullable(clazz.cast(scalingPolicyUpdateBehaviorAsString()));
        case "DisableDynamicScaling":
            return Optional.ofNullable(clazz.cast(disableDynamicScaling()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingInstruction, T> g) {
        return obj -> g.apply((ScalingInstruction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingInstruction> {
        /**
         * <p>
         * The namespace of the AWS service.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(String serviceNamespace);

        /**
         * <p>
         * The namespace of the AWS service.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(ServiceNamespace serviceNamespace);

        /**
         * <p>
         * The ID of the resource. This string consists of the resource type and unique identifier.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is the name
         * of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
         * service name. Example: <code>service/default/sample-webapp</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the
         * Spot Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
         * Example: <code>table/my-table</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
         * resource ID. Example: <code>table/my-table/index/my-table-index</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:my-db-cluster</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        The ID of the resource. This string consists of the resource type and unique identifier.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Auto Scaling group - The resource type is <code>autoScalingGroup</code> and the unique identifier is
         *        the name of the Auto Scaling group. Example: <code>autoScalingGroup/my-asg</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name
         *        and service name. Example: <code>service/default/sample-webapp</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
         *        the Spot Fleet request ID. Example:
         *        <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB table - The resource type is <code>table</code> and the unique identifier is the resource ID.
         *        Example: <code>table/my-table</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
         *        the resource ID. Example: <code>table/my-table/index/my-table-index</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:my-db-cluster</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The scalable dimension associated with the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension associated with the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(String scalableDimension);

        /**
         * <p>
         * The scalable dimension associated with the resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension associated with the resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>autoscaling:autoScalingGroup:DesiredCapacity</code> - The desired capacity of an Auto Scaling
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(ScalableDimension scalableDimension);

        /**
         * <p>
         * The minimum capacity of the resource.
         * </p>
         * 
         * @param minCapacity
         *        The minimum capacity of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minCapacity(Integer minCapacity);

        /**
         * <p>
         * The maximum capacity of the resource. The exception to this upper limit is if you specify a non-default
         * setting for <b>PredictiveScalingMaxCapacityBehavior</b>.
         * </p>
         * 
         * @param maxCapacity
         *        The maximum capacity of the resource. The exception to this upper limit is if you specify a
         *        non-default setting for <b>PredictiveScalingMaxCapacityBehavior</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Integer maxCapacity);

        /**
         * <p>
         * The structure that defines new target tracking configurations (up to 10). Each of these structures includes a
         * specific scaling metric and a target value for the metric, along with various parameters to use with dynamic
         * scaling.
         * </p>
         * <p>
         * With predictive scaling and dynamic scaling, the resource scales based on the target tracking configuration
         * that provides the largest capacity for both scale in and scale out.
         * </p>
         * <p>
         * Condition: The scaling metric must be unique across target tracking configurations.
         * </p>
         * 
         * @param targetTrackingConfigurations
         *        The structure that defines new target tracking configurations (up to 10). Each of these structures
         *        includes a specific scaling metric and a target value for the metric, along with various parameters to
         *        use with dynamic scaling. </p>
         *        <p>
         *        With predictive scaling and dynamic scaling, the resource scales based on the target tracking
         *        configuration that provides the largest capacity for both scale in and scale out.
         *        </p>
         *        <p>
         *        Condition: The scaling metric must be unique across target tracking configurations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTrackingConfigurations(Collection<TargetTrackingConfiguration> targetTrackingConfigurations);

        /**
         * <p>
         * The structure that defines new target tracking configurations (up to 10). Each of these structures includes a
         * specific scaling metric and a target value for the metric, along with various parameters to use with dynamic
         * scaling.
         * </p>
         * <p>
         * With predictive scaling and dynamic scaling, the resource scales based on the target tracking configuration
         * that provides the largest capacity for both scale in and scale out.
         * </p>
         * <p>
         * Condition: The scaling metric must be unique across target tracking configurations.
         * </p>
         * 
         * @param targetTrackingConfigurations
         *        The structure that defines new target tracking configurations (up to 10). Each of these structures
         *        includes a specific scaling metric and a target value for the metric, along with various parameters to
         *        use with dynamic scaling. </p>
         *        <p>
         *        With predictive scaling and dynamic scaling, the resource scales based on the target tracking
         *        configuration that provides the largest capacity for both scale in and scale out.
         *        </p>
         *        <p>
         *        Condition: The scaling metric must be unique across target tracking configurations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTrackingConfigurations(TargetTrackingConfiguration... targetTrackingConfigurations);

        /**
         * <p>
         * The structure that defines new target tracking configurations (up to 10). Each of these structures includes a
         * specific scaling metric and a target value for the metric, along with various parameters to use with dynamic
         * scaling.
         * </p>
         * <p>
         * With predictive scaling and dynamic scaling, the resource scales based on the target tracking configuration
         * that provides the largest capacity for both scale in and scale out.
         * </p>
         * <p>
         * Condition: The scaling metric must be unique across target tracking configurations.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TargetTrackingConfiguration>.Builder}
         * avoiding the need to create one manually via {@link List<TargetTrackingConfiguration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TargetTrackingConfiguration>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #targetTrackingConfigurations(List<TargetTrackingConfiguration>)}.
         * 
         * @param targetTrackingConfigurations
         *        a consumer that will call methods on {@link List<TargetTrackingConfiguration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTrackingConfigurations(List<TargetTrackingConfiguration>)
         */
        Builder targetTrackingConfigurations(Consumer<TargetTrackingConfiguration.Builder>... targetTrackingConfigurations);

        /**
         * <p>
         * The predefined load metric to use for predictive scaling. This parameter or a
         * <b>CustomizedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
         * otherwise.
         * </p>
         * 
         * @param predefinedLoadMetricSpecification
         *        The predefined load metric to use for predictive scaling. This parameter or a
         *        <b>CustomizedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot
         *        be used otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predefinedLoadMetricSpecification(PredefinedLoadMetricSpecification predefinedLoadMetricSpecification);

        /**
         * <p>
         * The predefined load metric to use for predictive scaling. This parameter or a
         * <b>CustomizedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
         * otherwise.
         * </p>
         * This is a convenience that creates an instance of the {@link PredefinedLoadMetricSpecification.Builder}
         * avoiding the need to create one manually via {@link PredefinedLoadMetricSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link PredefinedLoadMetricSpecification.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #predefinedLoadMetricSpecification(PredefinedLoadMetricSpecification)}.
         * 
         * @param predefinedLoadMetricSpecification
         *        a consumer that will call methods on {@link PredefinedLoadMetricSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #predefinedLoadMetricSpecification(PredefinedLoadMetricSpecification)
         */
        default Builder predefinedLoadMetricSpecification(
                Consumer<PredefinedLoadMetricSpecification.Builder> predefinedLoadMetricSpecification) {
            return predefinedLoadMetricSpecification(PredefinedLoadMetricSpecification.builder()
                    .applyMutation(predefinedLoadMetricSpecification).build());
        }

        /**
         * <p>
         * The customized load metric to use for predictive scaling. This parameter or a
         * <b>PredefinedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
         * otherwise.
         * </p>
         * 
         * @param customizedLoadMetricSpecification
         *        The customized load metric to use for predictive scaling. This parameter or a
         *        <b>PredefinedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot
         *        be used otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizedLoadMetricSpecification(CustomizedLoadMetricSpecification customizedLoadMetricSpecification);

        /**
         * <p>
         * The customized load metric to use for predictive scaling. This parameter or a
         * <b>PredefinedLoadMetricSpecification</b> is required when configuring predictive scaling, and cannot be used
         * otherwise.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomizedLoadMetricSpecification.Builder}
         * avoiding the need to create one manually via {@link CustomizedLoadMetricSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomizedLoadMetricSpecification.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #customizedLoadMetricSpecification(CustomizedLoadMetricSpecification)}.
         * 
         * @param customizedLoadMetricSpecification
         *        a consumer that will call methods on {@link CustomizedLoadMetricSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customizedLoadMetricSpecification(CustomizedLoadMetricSpecification)
         */
        default Builder customizedLoadMetricSpecification(
                Consumer<CustomizedLoadMetricSpecification.Builder> customizedLoadMetricSpecification) {
            return customizedLoadMetricSpecification(CustomizedLoadMetricSpecification.builder()
                    .applyMutation(customizedLoadMetricSpecification).build());
        }

        /**
         * <p>
         * The amount of time, in seconds, to buffer the run time of scheduled scaling actions when scaling out. For
         * example, if the forecast says to add capacity at 10:00 AM, and the buffer time is 5 minutes, then the run
         * time of the corresponding scheduled scaling action will be 9:55 AM. The intention is to give resources time
         * to be provisioned. For example, it can take a few minutes to launch an EC2 instance. The actual amount of
         * time required depends on several factors, such as the size of the instance and whether there are startup
         * scripts to complete.
         * </p>
         * <p>
         * The value must be less than the forecast interval duration of 3600 seconds (60 minutes). The default is 300
         * seconds.
         * </p>
         * <p>
         * Only valid when configuring predictive scaling.
         * </p>
         * 
         * @param scheduledActionBufferTime
         *        The amount of time, in seconds, to buffer the run time of scheduled scaling actions when scaling out.
         *        For example, if the forecast says to add capacity at 10:00 AM, and the buffer time is 5 minutes, then
         *        the run time of the corresponding scheduled scaling action will be 9:55 AM. The intention is to give
         *        resources time to be provisioned. For example, it can take a few minutes to launch an EC2 instance.
         *        The actual amount of time required depends on several factors, such as the size of the instance and
         *        whether there are startup scripts to complete. </p>
         *        <p>
         *        The value must be less than the forecast interval duration of 3600 seconds (60 minutes). The default
         *        is 300 seconds.
         *        </p>
         *        <p>
         *        Only valid when configuring predictive scaling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledActionBufferTime(Integer scheduledActionBufferTime);

        /**
         * <p>
         * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         * capacity specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>.
         * </p>
         * <p>
         * The following are possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher than
         * the maximum capacity. The maximum capacity is enforced as a hard limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
         * maximum capacity to equal but not exceed forecast capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than
         * the maximum capacity by a specified buffer value. The intention is to give the target tracking scaling policy
         * extra capacity if unexpected traffic occurs.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Only valid when configuring predictive scaling.
         * </p>
         * 
         * @param predictiveScalingMaxCapacityBehavior
         *        Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         *        capacity specified for the resource. The default value is
         *        <code>SetForecastCapacityToMaxCapacity</code>.</p>
         *        <p>
         *        The following are possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher
         *        than the maximum capacity. The maximum capacity is enforced as a hard limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
         *        than the maximum capacity to equal but not exceed forecast capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
         *        than the maximum capacity by a specified buffer value. The intention is to give the target tracking
         *        scaling policy extra capacity if unexpected traffic occurs.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Only valid when configuring predictive scaling.
         * @see PredictiveScalingMaxCapacityBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMaxCapacityBehavior
         */
        Builder predictiveScalingMaxCapacityBehavior(String predictiveScalingMaxCapacityBehavior);

        /**
         * <p>
         * Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         * capacity specified for the resource. The default value is <code>SetForecastCapacityToMaxCapacity</code>.
         * </p>
         * <p>
         * The following are possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher than
         * the maximum capacity. The maximum capacity is enforced as a hard limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than the
         * maximum capacity to equal but not exceed forecast capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher than
         * the maximum capacity by a specified buffer value. The intention is to give the target tracking scaling policy
         * extra capacity if unexpected traffic occurs.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Only valid when configuring predictive scaling.
         * </p>
         * 
         * @param predictiveScalingMaxCapacityBehavior
         *        Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum
         *        capacity specified for the resource. The default value is
         *        <code>SetForecastCapacityToMaxCapacity</code>.</p>
         *        <p>
         *        The following are possible values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SetForecastCapacityToMaxCapacity</code> - AWS Auto Scaling cannot scale resource capacity higher
         *        than the maximum capacity. The maximum capacity is enforced as a hard limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SetMaxCapacityToForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
         *        than the maximum capacity to equal but not exceed forecast capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SetMaxCapacityAboveForecastCapacity</code> - AWS Auto Scaling may scale resource capacity higher
         *        than the maximum capacity by a specified buffer value. The intention is to give the target tracking
         *        scaling policy extra capacity if unexpected traffic occurs.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Only valid when configuring predictive scaling.
         * @see PredictiveScalingMaxCapacityBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMaxCapacityBehavior
         */
        Builder predictiveScalingMaxCapacityBehavior(PredictiveScalingMaxCapacityBehavior predictiveScalingMaxCapacityBehavior);

        /**
         * <p>
         * The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
         * capacity. The value is specified as a percentage relative to the forecast capacity. For example, if the
         * buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum
         * capacity is 40, then the effective maximum capacity is 55.
         * </p>
         * <p>
         * Only valid when configuring predictive scaling. Required if the <b>PredictiveScalingMaxCapacityBehavior</b>
         * is set to <code>SetMaxCapacityAboveForecastCapacity</code>, and cannot be used otherwise.
         * </p>
         * <p>
         * The range is 1-100.
         * </p>
         * 
         * @param predictiveScalingMaxCapacityBuffer
         *        The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum
         *        capacity. The value is specified as a percentage relative to the forecast capacity. For example, if
         *        the buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the
         *        maximum capacity is 40, then the effective maximum capacity is 55.</p>
         *        <p>
         *        Only valid when configuring predictive scaling. Required if the
         *        <b>PredictiveScalingMaxCapacityBehavior</b> is set to <code>SetMaxCapacityAboveForecastCapacity</code>
         *        , and cannot be used otherwise.
         *        </p>
         *        <p>
         *        The range is 1-100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictiveScalingMaxCapacityBuffer(Integer predictiveScalingMaxCapacityBuffer);

        /**
         * <p>
         * The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto Scaling
         * forecasts capacity but does not create any scheduled scaling actions based on the capacity forecast.
         * </p>
         * 
         * @param predictiveScalingMode
         *        The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto
         *        Scaling forecasts capacity but does not create any scheduled scaling actions based on the capacity
         *        forecast.
         * @see PredictiveScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMode
         */
        Builder predictiveScalingMode(String predictiveScalingMode);

        /**
         * <p>
         * The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto Scaling
         * forecasts capacity but does not create any scheduled scaling actions based on the capacity forecast.
         * </p>
         * 
         * @param predictiveScalingMode
         *        The predictive scaling mode. The default value is <code>ForecastAndScale</code>. Otherwise, AWS Auto
         *        Scaling forecasts capacity but does not create any scheduled scaling actions based on the capacity
         *        forecast.
         * @see PredictiveScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PredictiveScalingMode
         */
        Builder predictiveScalingMode(PredictiveScalingMode predictiveScalingMode);

        /**
         * <p>
         * Controls whether a resource's externally created scaling policies are kept or replaced.
         * </p>
         * <p>
         * The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
         * <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are deleted
         * and new target tracking scaling policies created.
         * </p>
         * <p>
         * Only valid when configuring dynamic scaling.
         * </p>
         * <p>
         * Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are more
         * than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create new ones.
         * </p>
         * 
         * @param scalingPolicyUpdateBehavior
         *        Controls whether a resource's externally created scaling policies are kept or replaced. </p>
         *        <p>
         *        The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
         *        <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are
         *        deleted and new target tracking scaling policies created.
         *        </p>
         *        <p>
         *        Only valid when configuring dynamic scaling.
         *        </p>
         *        <p>
         *        Condition: The number of existing policies to be replaced must be less than or equal to 50. If there
         *        are more than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not
         *        create new ones.
         * @see ScalingPolicyUpdateBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingPolicyUpdateBehavior
         */
        Builder scalingPolicyUpdateBehavior(String scalingPolicyUpdateBehavior);

        /**
         * <p>
         * Controls whether a resource's externally created scaling policies are kept or replaced.
         * </p>
         * <p>
         * The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
         * <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are deleted
         * and new target tracking scaling policies created.
         * </p>
         * <p>
         * Only valid when configuring dynamic scaling.
         * </p>
         * <p>
         * Condition: The number of existing policies to be replaced must be less than or equal to 50. If there are more
         * than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not create new ones.
         * </p>
         * 
         * @param scalingPolicyUpdateBehavior
         *        Controls whether a resource's externally created scaling policies are kept or replaced. </p>
         *        <p>
         *        The default value is <code>KeepExternalPolicies</code>. If the parameter is set to
         *        <code>ReplaceExternalPolicies</code>, any scaling policies that are external to AWS Auto Scaling are
         *        deleted and new target tracking scaling policies created.
         *        </p>
         *        <p>
         *        Only valid when configuring dynamic scaling.
         *        </p>
         *        <p>
         *        Condition: The number of existing policies to be replaced must be less than or equal to 50. If there
         *        are more than 50 policies to be replaced, AWS Auto Scaling keeps all existing policies and does not
         *        create new ones.
         * @see ScalingPolicyUpdateBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingPolicyUpdateBehavior
         */
        Builder scalingPolicyUpdateBehavior(ScalingPolicyUpdateBehavior scalingPolicyUpdateBehavior);

        /**
         * <p>
         * Controls whether dynamic scaling by AWS Auto Scaling is disabled. When dynamic scaling is enabled, AWS Auto
         * Scaling creates target tracking scaling policies based on the specified target tracking configurations.
         * </p>
         * <p>
         * The default is enabled (<code>false</code>).
         * </p>
         * 
         * @param disableDynamicScaling
         *        Controls whether dynamic scaling by AWS Auto Scaling is disabled. When dynamic scaling is enabled, AWS
         *        Auto Scaling creates target tracking scaling policies based on the specified target tracking
         *        configurations. </p>
         *        <p>
         *        The default is enabled (<code>false</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableDynamicScaling(Boolean disableDynamicScaling);
    }

    static final class BuilderImpl implements Builder {
        private String serviceNamespace;

        private String resourceId;

        private String scalableDimension;

        private Integer minCapacity;

        private Integer maxCapacity;

        private List<TargetTrackingConfiguration> targetTrackingConfigurations = DefaultSdkAutoConstructList.getInstance();

        private PredefinedLoadMetricSpecification predefinedLoadMetricSpecification;

        private CustomizedLoadMetricSpecification customizedLoadMetricSpecification;

        private Integer scheduledActionBufferTime;

        private String predictiveScalingMaxCapacityBehavior;

        private Integer predictiveScalingMaxCapacityBuffer;

        private String predictiveScalingMode;

        private String scalingPolicyUpdateBehavior;

        private Boolean disableDynamicScaling;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingInstruction model) {
            serviceNamespace(model.serviceNamespace);
            resourceId(model.resourceId);
            scalableDimension(model.scalableDimension);
            minCapacity(model.minCapacity);
            maxCapacity(model.maxCapacity);
            targetTrackingConfigurations(model.targetTrackingConfigurations);
            predefinedLoadMetricSpecification(model.predefinedLoadMetricSpecification);
            customizedLoadMetricSpecification(model.customizedLoadMetricSpecification);
            scheduledActionBufferTime(model.scheduledActionBufferTime);
            predictiveScalingMaxCapacityBehavior(model.predictiveScalingMaxCapacityBehavior);
            predictiveScalingMaxCapacityBuffer(model.predictiveScalingMaxCapacityBuffer);
            predictiveScalingMode(model.predictiveScalingMode);
            scalingPolicyUpdateBehavior(model.scalingPolicyUpdateBehavior);
            disableDynamicScaling(model.disableDynamicScaling);
        }

        public final String getServiceNamespace() {
            return serviceNamespace;
        }

        @Override
        public final Builder serviceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
            return this;
        }

        @Override
        public final Builder serviceNamespace(ServiceNamespace serviceNamespace) {
            this.serviceNamespace(serviceNamespace == null ? null : serviceNamespace.toString());
            return this;
        }

        public final void setServiceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getScalableDimension() {
            return scalableDimension;
        }

        @Override
        public final Builder scalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
            return this;
        }

        @Override
        public final Builder scalableDimension(ScalableDimension scalableDimension) {
            this.scalableDimension(scalableDimension == null ? null : scalableDimension.toString());
            return this;
        }

        public final void setScalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
        }

        public final Integer getMinCapacity() {
            return minCapacity;
        }

        @Override
        public final Builder minCapacity(Integer minCapacity) {
            this.minCapacity = minCapacity;
            return this;
        }

        public final void setMinCapacity(Integer minCapacity) {
            this.minCapacity = minCapacity;
        }

        public final Integer getMaxCapacity() {
            return maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Integer maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final void setMaxCapacity(Integer maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        public final Collection<TargetTrackingConfiguration.Builder> getTargetTrackingConfigurations() {
            return targetTrackingConfigurations != null ? targetTrackingConfigurations.stream()
                    .map(TargetTrackingConfiguration::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targetTrackingConfigurations(Collection<TargetTrackingConfiguration> targetTrackingConfigurations) {
            this.targetTrackingConfigurations = TargetTrackingConfigurationsCopier.copy(targetTrackingConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTrackingConfigurations(TargetTrackingConfiguration... targetTrackingConfigurations) {
            targetTrackingConfigurations(Arrays.asList(targetTrackingConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTrackingConfigurations(
                Consumer<TargetTrackingConfiguration.Builder>... targetTrackingConfigurations) {
            targetTrackingConfigurations(Stream.of(targetTrackingConfigurations)
                    .map(c -> TargetTrackingConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargetTrackingConfigurations(
                Collection<TargetTrackingConfiguration.BuilderImpl> targetTrackingConfigurations) {
            this.targetTrackingConfigurations = TargetTrackingConfigurationsCopier.copyFromBuilder(targetTrackingConfigurations);
        }

        public final PredefinedLoadMetricSpecification.Builder getPredefinedLoadMetricSpecification() {
            return predefinedLoadMetricSpecification != null ? predefinedLoadMetricSpecification.toBuilder() : null;
        }

        @Override
        public final Builder predefinedLoadMetricSpecification(PredefinedLoadMetricSpecification predefinedLoadMetricSpecification) {
            this.predefinedLoadMetricSpecification = predefinedLoadMetricSpecification;
            return this;
        }

        public final void setPredefinedLoadMetricSpecification(
                PredefinedLoadMetricSpecification.BuilderImpl predefinedLoadMetricSpecification) {
            this.predefinedLoadMetricSpecification = predefinedLoadMetricSpecification != null ? predefinedLoadMetricSpecification
                    .build() : null;
        }

        public final CustomizedLoadMetricSpecification.Builder getCustomizedLoadMetricSpecification() {
            return customizedLoadMetricSpecification != null ? customizedLoadMetricSpecification.toBuilder() : null;
        }

        @Override
        public final Builder customizedLoadMetricSpecification(CustomizedLoadMetricSpecification customizedLoadMetricSpecification) {
            this.customizedLoadMetricSpecification = customizedLoadMetricSpecification;
            return this;
        }

        public final void setCustomizedLoadMetricSpecification(
                CustomizedLoadMetricSpecification.BuilderImpl customizedLoadMetricSpecification) {
            this.customizedLoadMetricSpecification = customizedLoadMetricSpecification != null ? customizedLoadMetricSpecification
                    .build() : null;
        }

        public final Integer getScheduledActionBufferTime() {
            return scheduledActionBufferTime;
        }

        @Override
        public final Builder scheduledActionBufferTime(Integer scheduledActionBufferTime) {
            this.scheduledActionBufferTime = scheduledActionBufferTime;
            return this;
        }

        public final void setScheduledActionBufferTime(Integer scheduledActionBufferTime) {
            this.scheduledActionBufferTime = scheduledActionBufferTime;
        }

        public final String getPredictiveScalingMaxCapacityBehavior() {
            return predictiveScalingMaxCapacityBehavior;
        }

        @Override
        public final Builder predictiveScalingMaxCapacityBehavior(String predictiveScalingMaxCapacityBehavior) {
            this.predictiveScalingMaxCapacityBehavior = predictiveScalingMaxCapacityBehavior;
            return this;
        }

        @Override
        public final Builder predictiveScalingMaxCapacityBehavior(
                PredictiveScalingMaxCapacityBehavior predictiveScalingMaxCapacityBehavior) {
            this.predictiveScalingMaxCapacityBehavior(predictiveScalingMaxCapacityBehavior == null ? null
                    : predictiveScalingMaxCapacityBehavior.toString());
            return this;
        }

        public final void setPredictiveScalingMaxCapacityBehavior(String predictiveScalingMaxCapacityBehavior) {
            this.predictiveScalingMaxCapacityBehavior = predictiveScalingMaxCapacityBehavior;
        }

        public final Integer getPredictiveScalingMaxCapacityBuffer() {
            return predictiveScalingMaxCapacityBuffer;
        }

        @Override
        public final Builder predictiveScalingMaxCapacityBuffer(Integer predictiveScalingMaxCapacityBuffer) {
            this.predictiveScalingMaxCapacityBuffer = predictiveScalingMaxCapacityBuffer;
            return this;
        }

        public final void setPredictiveScalingMaxCapacityBuffer(Integer predictiveScalingMaxCapacityBuffer) {
            this.predictiveScalingMaxCapacityBuffer = predictiveScalingMaxCapacityBuffer;
        }

        public final String getPredictiveScalingMode() {
            return predictiveScalingMode;
        }

        @Override
        public final Builder predictiveScalingMode(String predictiveScalingMode) {
            this.predictiveScalingMode = predictiveScalingMode;
            return this;
        }

        @Override
        public final Builder predictiveScalingMode(PredictiveScalingMode predictiveScalingMode) {
            this.predictiveScalingMode(predictiveScalingMode == null ? null : predictiveScalingMode.toString());
            return this;
        }

        public final void setPredictiveScalingMode(String predictiveScalingMode) {
            this.predictiveScalingMode = predictiveScalingMode;
        }

        public final String getScalingPolicyUpdateBehavior() {
            return scalingPolicyUpdateBehavior;
        }

        @Override
        public final Builder scalingPolicyUpdateBehavior(String scalingPolicyUpdateBehavior) {
            this.scalingPolicyUpdateBehavior = scalingPolicyUpdateBehavior;
            return this;
        }

        @Override
        public final Builder scalingPolicyUpdateBehavior(ScalingPolicyUpdateBehavior scalingPolicyUpdateBehavior) {
            this.scalingPolicyUpdateBehavior(scalingPolicyUpdateBehavior == null ? null : scalingPolicyUpdateBehavior.toString());
            return this;
        }

        public final void setScalingPolicyUpdateBehavior(String scalingPolicyUpdateBehavior) {
            this.scalingPolicyUpdateBehavior = scalingPolicyUpdateBehavior;
        }

        public final Boolean getDisableDynamicScaling() {
            return disableDynamicScaling;
        }

        @Override
        public final Builder disableDynamicScaling(Boolean disableDynamicScaling) {
            this.disableDynamicScaling = disableDynamicScaling;
            return this;
        }

        public final void setDisableDynamicScaling(Boolean disableDynamicScaling) {
            this.disableDynamicScaling = disableDynamicScaling;
        }

        @Override
        public ScalingInstruction build() {
            return new ScalingInstruction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
