/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a CloudWatch metric of your choosing that can be used for predictive scaling.
 * </p>
 * <p>
 * For predictive scaling to work with a customized load metric specification, AWS Auto Scaling needs access to the
 * <code>Sum</code> and <code>Average</code> statistics that CloudWatch computes from metric data. Statistics are
 * calculations used to aggregate data over specified time periods.
 * </p>
 * <p>
 * When you choose a load metric, make sure that the required <code>Sum</code> and <code>Average</code> statistics for
 * your metric are available in CloudWatch and that they provide relevant data for predictive scaling. The
 * <code>Sum</code> statistic must represent the total load on the resource, and the <code>Average</code> statistic must
 * represent the average load per capacity unit of the resource. For example, there is a metric that counts the number
 * of requests processed by your Auto Scaling group. If the <code>Sum</code> statistic represents the total request
 * count processed by the group, then the <code>Average</code> statistic for the specified metric must represent the
 * average request count processed by each instance of the group.
 * </p>
 * <p>
 * For information about terminology, available metrics, or how to publish new metrics, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html">Amazon CloudWatch
 * Concepts</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomizedLoadMetricSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomizedLoadMetricSpecification.Builder, CustomizedLoadMetricSpecification> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomizedLoadMetricSpecification::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomizedLoadMetricSpecification::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<List<MetricDimension>> DIMENSIONS_FIELD = SdkField
            .<List<MetricDimension>> builder(MarshallingType.LIST)
            .getter(getter(CustomizedLoadMetricSpecification::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomizedLoadMetricSpecification::statisticAsString)).setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistic").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomizedLoadMetricSpecification::unit)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            NAMESPACE_FIELD, DIMENSIONS_FIELD, STATISTIC_FIELD, UNIT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final String namespace;

    private final List<MetricDimension> dimensions;

    private final String statistic;

    private final String unit;

    private CustomizedLoadMetricSpecification(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.dimensions = builder.dimensions;
        this.statistic = builder.statistic;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric.
     * </p>
     * 
     * @return The namespace of the metric.
     */
    public String namespace() {
        return namespace;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dimensions of the metric.
     * </p>
     * <p>
     * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     * customized load metric specification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The dimensions of the metric.</p>
     *         <p>
     *         Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
     *         customized load metric specification.
     */
    public List<MetricDimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The statistic of the metric. Currently, the value must always be <code>Sum</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric. Currently, the value must always be <code>Sum</code>.
     * @see MetricStatistic
     */
    public MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic of the metric. Currently, the value must always be <code>Sum</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the metric. Currently, the value must always be <code>Sum</code>.
     * @see MetricStatistic
     */
    public String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * 
     * @return The unit of the metric.
     */
    public String unit() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomizedLoadMetricSpecification)) {
            return false;
        }
        CustomizedLoadMetricSpecification other = (CustomizedLoadMetricSpecification) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(statisticAsString(), other.statisticAsString()) && Objects.equals(unit(), other.unit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CustomizedLoadMetricSpecification").add("MetricName", metricName())
                .add("Namespace", namespace()).add("Dimensions", dimensions()).add("Statistic", statisticAsString())
                .add("Unit", unit()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomizedLoadMetricSpecification, T> g) {
        return obj -> g.apply((CustomizedLoadMetricSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomizedLoadMetricSpecification> {
        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your customized load metric specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * 
         * @param dimensions
         *        The dimensions of the metric.</p>
         *        <p>
         *        Conditional: If you published your metric with dimensions, you must specify the same dimensions in
         *        your customized load metric specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * The dimensions of the metric.
         * </p>
         * <p>
         * Conditional: If you published your metric with dimensions, you must specify the same dimensions in your
         * customized load metric specification.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricDimension>.Builder} avoiding the need
         * to create one manually via {@link List<MetricDimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricDimension>.Builder#build()} is called immediately and
         * its result is passed to {@link #dimensions(List<MetricDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<MetricDimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<MetricDimension>)
         */
        Builder dimensions(Consumer<MetricDimension.Builder>... dimensions);

        /**
         * <p>
         * The statistic of the metric. Currently, the value must always be <code>Sum</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric. Currently, the value must always be <code>Sum</code>.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic of the metric. Currently, the value must always be <code>Sum</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic of the metric. Currently, the value must always be <code>Sum</code>.
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private String namespace;

        private List<MetricDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String statistic;

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomizedLoadMetricSpecification model) {
            metricName(model.metricName);
            namespace(model.namespace);
            dimensions(model.dimensions);
            statistic(model.statistic);
            unit(model.unit);
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final Collection<MetricDimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(MetricDimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<MetricDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> MetricDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionsCopier.copyFromBuilder(dimensions);
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public CustomizedLoadMetricSpecification build() {
            return new CustomizedLoadMetricSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
