/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.autoscalingplans.model.AutoScalingPlansException;
import software.amazon.awssdk.services.autoscalingplans.model.ConcurrentUpdateException;
import software.amazon.awssdk.services.autoscalingplans.model.CreateScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.CreateScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DeleteScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DeleteScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlanResourcesRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlanResourcesResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlansRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlansResponse;
import software.amazon.awssdk.services.autoscalingplans.model.GetScalingPlanResourceForecastDataRequest;
import software.amazon.awssdk.services.autoscalingplans.model.GetScalingPlanResourceForecastDataResponse;
import software.amazon.awssdk.services.autoscalingplans.model.InternalServiceException;
import software.amazon.awssdk.services.autoscalingplans.model.InvalidNextTokenException;
import software.amazon.awssdk.services.autoscalingplans.model.LimitExceededException;
import software.amazon.awssdk.services.autoscalingplans.model.ObjectNotFoundException;
import software.amazon.awssdk.services.autoscalingplans.model.UpdateScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.UpdateScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.ValidationException;
import software.amazon.awssdk.services.autoscalingplans.transform.CreateScalingPlanRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DeleteScalingPlanRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DescribeScalingPlanResourcesRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DescribeScalingPlansRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.GetScalingPlanResourceForecastDataRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.UpdateScalingPlanRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AutoScalingPlansAsyncClient}.
 *
 * @see AutoScalingPlansAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAutoScalingPlansAsyncClient implements AutoScalingPlansAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAutoScalingPlansAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAutoScalingPlansAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a scaling plan.
     * </p>
     *
     * @param createScalingPlanRequest
     * @return A Java Future containing the result of the CreateScalingPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>LimitExceededException Your account exceeded a limit. This exception is thrown when a per-account
     *         resource limit is exceeded.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to a scaling plan that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.CreateScalingPlan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/CreateScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateScalingPlanResponse> createScalingPlan(CreateScalingPlanRequest createScalingPlanRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateScalingPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateScalingPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateScalingPlanRequest, CreateScalingPlanResponse>()
                            .withOperationName("CreateScalingPlan")
                            .withMarshaller(new CreateScalingPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createScalingPlanRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified scaling plan.
     * </p>
     * <p>
     * Deleting a scaling plan deletes the underlying <a>ScalingInstruction</a> for all of the scalable resources that
     * are covered by the plan.
     * </p>
     * <p>
     * If the plan has launched resources or has scaling activities in progress, you must delete those resources
     * separately.
     * </p>
     *
     * @param deleteScalingPlanRequest
     * @return A Java Future containing the result of the DeleteScalingPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>ObjectNotFoundException The specified object could not be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to a scaling plan that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.DeleteScalingPlan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DeleteScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScalingPlanResponse> deleteScalingPlan(DeleteScalingPlanRequest deleteScalingPlanRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScalingPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScalingPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScalingPlanRequest, DeleteScalingPlanResponse>()
                            .withOperationName("DeleteScalingPlan")
                            .withMarshaller(new DeleteScalingPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteScalingPlanRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the scalable resources in the specified scaling plan.
     * </p>
     *
     * @param describeScalingPlanResourcesRequest
     * @return A Java Future containing the result of the DescribeScalingPlanResources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>InvalidNextTokenException The token provided is not valid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to a scaling plan that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.DescribeScalingPlanResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DescribeScalingPlanResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingPlanResourcesResponse> describeScalingPlanResources(
            DescribeScalingPlanResourcesRequest describeScalingPlanResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalingPlanResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalingPlanResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalingPlanResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingPlanResourcesRequest, DescribeScalingPlanResourcesResponse>()
                            .withOperationName("DescribeScalingPlanResources")
                            .withMarshaller(new DescribeScalingPlanResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalingPlanResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your scaling plans.
     * </p>
     *
     * @param describeScalingPlansRequest
     * @return A Java Future containing the result of the DescribeScalingPlans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>InvalidNextTokenException The token provided is not valid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to a scaling plan that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.DescribeScalingPlans
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DescribeScalingPlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingPlansResponse> describeScalingPlans(
            DescribeScalingPlansRequest describeScalingPlansRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalingPlansResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalingPlansResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalingPlansResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingPlansRequest, DescribeScalingPlansResponse>()
                            .withOperationName("DescribeScalingPlans")
                            .withMarshaller(new DescribeScalingPlansRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalingPlansRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the forecast data for a scalable resource.
     * </p>
     * <p>
     * Capacity forecasts are represented as predicted values, or data points, that are calculated using historical data
     * points from a specified CloudWatch load metric. Data points are available for up to 56 days.
     * </p>
     *
     * @param getScalingPlanResourceForecastDataRequest
     * @return A Java Future containing the result of the GetScalingPlanResourceForecastData operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.GetScalingPlanResourceForecastData
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/GetScalingPlanResourceForecastData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetScalingPlanResourceForecastDataResponse> getScalingPlanResourceForecastData(
            GetScalingPlanResourceForecastDataRequest getScalingPlanResourceForecastDataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetScalingPlanResourceForecastDataResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetScalingPlanResourceForecastDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetScalingPlanResourceForecastDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetScalingPlanResourceForecastDataRequest, GetScalingPlanResourceForecastDataResponse>()
                            .withOperationName("GetScalingPlanResourceForecastData")
                            .withMarshaller(new GetScalingPlanResourceForecastDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getScalingPlanResourceForecastDataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified scaling plan.
     * </p>
     * <p>
     * You cannot update a scaling plan if it is in the process of being created, updated, or deleted.
     * </p>
     *
     * @param updateScalingPlanRequest
     * @return A Java Future containing the result of the UpdateScalingPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the parameters provided.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to a scaling plan that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>ObjectNotFoundException The specified object could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AutoScalingPlansException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AutoScalingPlansAsyncClient.UpdateScalingPlan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/UpdateScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateScalingPlanResponse> updateScalingPlan(UpdateScalingPlanRequest updateScalingPlanRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateScalingPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateScalingPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScalingPlanRequest, UpdateScalingPlanResponse>()
                            .withOperationName("UpdateScalingPlan")
                            .withMarshaller(new UpdateScalingPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateScalingPlanRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AutoScalingPlansException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ObjectNotFoundException")
                                .exceptionBuilderSupplier(ObjectNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentUpdateException")
                                .exceptionBuilderSupplier(ConcurrentUpdateException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
