/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a lifecycle hook. Used in combination with <a>CreateAutoScalingGroup</a>.
 * </p>
 * <p>
 * A lifecycle hook tells Amazon EC2 Auto Scaling to perform an action on an instance when the instance launches (before
 * it is put into service) or as the instance terminates (before it is fully terminated).
 * </p>
 * <p>
 * This step is a part of the procedure for creating a lifecycle hook for an Auto Scaling group:
 * </p>
 * <ol>
 * <li>
 * <p>
 * (Optional) Create a Lambda function and a rule that allows CloudWatch Events to invoke your Lambda function when
 * Amazon EC2 Auto Scaling launches or terminates instances.
 * </p>
 * </li>
 * <li>
 * <p>
 * (Optional) Create a notification target and an IAM role. The target can be either an Amazon SQS queue or an Amazon
 * SNS topic. The role allows Amazon EC2 Auto Scaling to publish lifecycle notifications to the target.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Create the lifecycle hook. Specify whether the hook is used when the instances launch or terminate.</b>
 * </p>
 * </li>
 * <li>
 * <p>
 * If you need more time, record the lifecycle action heartbeat to keep the instance in a pending state using
 * <a>RecordLifecycleActionHeartbeat</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you finish before the timeout period ends, complete the lifecycle action using <a>CompleteLifecycleAction</a>.
 * </p>
 * </li>
 * </ol>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html">Amazon
 * EC2 Auto Scaling Lifecycle Hooks</a> in the <i>Amazon EC2 Auto Scaling User Guide</i>.
 * </p>
 * <p>
 * You can view the lifecycle hooks for an Auto Scaling group using <a>DescribeLifecycleHooks</a>. You can modify an
 * existing lifecycle hook or create new lifecycle hooks using <a>PutLifecycleHook</a>. If you are no longer using a
 * lifecycle hook, you can delete it using <a>DeleteLifecycleHook</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LifecycleHookSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<LifecycleHookSpecification.Builder, LifecycleHookSpecification> {
    private static final SdkField<String> LIFECYCLE_HOOK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::lifecycleHookName)).setter(setter(Builder::lifecycleHookName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LifecycleHookName").build()).build();

    private static final SdkField<String> LIFECYCLE_TRANSITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::lifecycleTransition)).setter(setter(Builder::lifecycleTransition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LifecycleTransition").build())
            .build();

    private static final SdkField<String> NOTIFICATION_METADATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::notificationMetadata)).setter(setter(Builder::notificationMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationMetadata").build())
            .build();

    private static final SdkField<Integer> HEARTBEAT_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(LifecycleHookSpecification::heartbeatTimeout)).setter(setter(Builder::heartbeatTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeartbeatTimeout").build()).build();

    private static final SdkField<String> DEFAULT_RESULT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::defaultResult)).setter(setter(Builder::defaultResult))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultResult").build()).build();

    private static final SdkField<String> NOTIFICATION_TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::notificationTargetARN)).setter(setter(Builder::notificationTargetARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationTargetARN").build())
            .build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleHookSpecification::roleARN)).setter(setter(Builder::roleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LIFECYCLE_HOOK_NAME_FIELD,
            LIFECYCLE_TRANSITION_FIELD, NOTIFICATION_METADATA_FIELD, HEARTBEAT_TIMEOUT_FIELD, DEFAULT_RESULT_FIELD,
            NOTIFICATION_TARGET_ARN_FIELD, ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String lifecycleHookName;

    private final String lifecycleTransition;

    private final String notificationMetadata;

    private final Integer heartbeatTimeout;

    private final String defaultResult;

    private final String notificationTargetARN;

    private final String roleARN;

    private LifecycleHookSpecification(BuilderImpl builder) {
        this.lifecycleHookName = builder.lifecycleHookName;
        this.lifecycleTransition = builder.lifecycleTransition;
        this.notificationMetadata = builder.notificationMetadata;
        this.heartbeatTimeout = builder.heartbeatTimeout;
        this.defaultResult = builder.defaultResult;
        this.notificationTargetARN = builder.notificationTargetARN;
        this.roleARN = builder.roleARN;
    }

    /**
     * <p>
     * The name of the lifecycle hook.
     * </p>
     * 
     * @return The name of the lifecycle hook.
     */
    public String lifecycleHookName() {
        return lifecycleHookName;
    }

    /**
     * <p>
     * The state of the EC2 instance to which you want to attach the lifecycle hook. The valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * autoscaling:EC2_INSTANCE_LAUNCHING
     * </p>
     * </li>
     * <li>
     * <p>
     * autoscaling:EC2_INSTANCE_TERMINATING
     * </p>
     * </li>
     * </ul>
     * 
     * @return The state of the EC2 instance to which you want to attach the lifecycle hook. The valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         autoscaling:EC2_INSTANCE_LAUNCHING
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         autoscaling:EC2_INSTANCE_TERMINATING
     *         </p>
     *         </li>
     */
    public String lifecycleTransition() {
        return lifecycleTransition;
    }

    /**
     * <p>
     * Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the
     * notification target.
     * </p>
     * 
     * @return Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the
     *         notification target.
     */
    public String notificationMetadata() {
        return notificationMetadata;
    }

    /**
     * <p>
     * The maximum time, in seconds, that can elapse before the lifecycle hook times out.
     * </p>
     * <p>
     * If the lifecycle hook times out, Amazon EC2 Auto Scaling performs the action that you specified in the
     * <code>DefaultResult</code> parameter. You can prevent the lifecycle hook from timing out by calling
     * <a>RecordLifecycleActionHeartbeat</a>.
     * </p>
     * 
     * @return The maximum time, in seconds, that can elapse before the lifecycle hook times out.</p>
     *         <p>
     *         If the lifecycle hook times out, Amazon EC2 Auto Scaling performs the action that you specified in the
     *         <code>DefaultResult</code> parameter. You can prevent the lifecycle hook from timing out by calling
     *         <a>RecordLifecycleActionHeartbeat</a>.
     */
    public Integer heartbeatTimeout() {
        return heartbeatTimeout;
    }

    /**
     * <p>
     * Defines the action the Auto Scaling group should take when the lifecycle hook timeout elapses or if an unexpected
     * failure occurs. The valid values are <code>CONTINUE</code> and <code>ABANDON</code>. The default value is
     * <code>ABANDON</code>.
     * </p>
     * 
     * @return Defines the action the Auto Scaling group should take when the lifecycle hook timeout elapses or if an
     *         unexpected failure occurs. The valid values are <code>CONTINUE</code> and <code>ABANDON</code>. The
     *         default value is <code>ABANDON</code>.
     */
    public String defaultResult() {
        return defaultResult;
    }

    /**
     * <p>
     * The ARN of the target that Amazon EC2 Auto Scaling sends notifications to when an instance is in the transition
     * state for the lifecycle hook. The notification target can be either an SQS queue or an SNS topic.
     * </p>
     * 
     * @return The ARN of the target that Amazon EC2 Auto Scaling sends notifications to when an instance is in the
     *         transition state for the lifecycle hook. The notification target can be either an SQS queue or an SNS
     *         topic.
     */
    public String notificationTargetARN() {
        return notificationTargetARN;
    }

    /**
     * <p>
     * The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target, for
     * example, an Amazon SNS topic or an Amazon SQS queue.
     * </p>
     * 
     * @return The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification
     *         target, for example, an Amazon SNS topic or an Amazon SQS queue.
     */
    public String roleARN() {
        return roleARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleHookName());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleTransition());
        hashCode = 31 * hashCode + Objects.hashCode(notificationMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(heartbeatTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(defaultResult());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTargetARN());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LifecycleHookSpecification)) {
            return false;
        }
        LifecycleHookSpecification other = (LifecycleHookSpecification) obj;
        return Objects.equals(lifecycleHookName(), other.lifecycleHookName())
                && Objects.equals(lifecycleTransition(), other.lifecycleTransition())
                && Objects.equals(notificationMetadata(), other.notificationMetadata())
                && Objects.equals(heartbeatTimeout(), other.heartbeatTimeout())
                && Objects.equals(defaultResult(), other.defaultResult())
                && Objects.equals(notificationTargetARN(), other.notificationTargetARN())
                && Objects.equals(roleARN(), other.roleARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LifecycleHookSpecification").add("LifecycleHookName", lifecycleHookName())
                .add("LifecycleTransition", lifecycleTransition()).add("NotificationMetadata", notificationMetadata())
                .add("HeartbeatTimeout", heartbeatTimeout()).add("DefaultResult", defaultResult())
                .add("NotificationTargetARN", notificationTargetARN()).add("RoleARN", roleARN()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LifecycleHookName":
            return Optional.ofNullable(clazz.cast(lifecycleHookName()));
        case "LifecycleTransition":
            return Optional.ofNullable(clazz.cast(lifecycleTransition()));
        case "NotificationMetadata":
            return Optional.ofNullable(clazz.cast(notificationMetadata()));
        case "HeartbeatTimeout":
            return Optional.ofNullable(clazz.cast(heartbeatTimeout()));
        case "DefaultResult":
            return Optional.ofNullable(clazz.cast(defaultResult()));
        case "NotificationTargetARN":
            return Optional.ofNullable(clazz.cast(notificationTargetARN()));
        case "RoleARN":
            return Optional.ofNullable(clazz.cast(roleARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LifecycleHookSpecification, T> g) {
        return obj -> g.apply((LifecycleHookSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LifecycleHookSpecification> {
        /**
         * <p>
         * The name of the lifecycle hook.
         * </p>
         * 
         * @param lifecycleHookName
         *        The name of the lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleHookName(String lifecycleHookName);

        /**
         * <p>
         * The state of the EC2 instance to which you want to attach the lifecycle hook. The valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * autoscaling:EC2_INSTANCE_LAUNCHING
         * </p>
         * </li>
         * <li>
         * <p>
         * autoscaling:EC2_INSTANCE_TERMINATING
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycleTransition
         *        The state of the EC2 instance to which you want to attach the lifecycle hook. The valid values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        autoscaling:EC2_INSTANCE_LAUNCHING
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        autoscaling:EC2_INSTANCE_TERMINATING
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleTransition(String lifecycleTransition);

        /**
         * <p>
         * Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the
         * notification target.
         * </p>
         * 
         * @param notificationMetadata
         *        Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to
         *        the notification target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationMetadata(String notificationMetadata);

        /**
         * <p>
         * The maximum time, in seconds, that can elapse before the lifecycle hook times out.
         * </p>
         * <p>
         * If the lifecycle hook times out, Amazon EC2 Auto Scaling performs the action that you specified in the
         * <code>DefaultResult</code> parameter. You can prevent the lifecycle hook from timing out by calling
         * <a>RecordLifecycleActionHeartbeat</a>.
         * </p>
         * 
         * @param heartbeatTimeout
         *        The maximum time, in seconds, that can elapse before the lifecycle hook times out.</p>
         *        <p>
         *        If the lifecycle hook times out, Amazon EC2 Auto Scaling performs the action that you specified in the
         *        <code>DefaultResult</code> parameter. You can prevent the lifecycle hook from timing out by calling
         *        <a>RecordLifecycleActionHeartbeat</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heartbeatTimeout(Integer heartbeatTimeout);

        /**
         * <p>
         * Defines the action the Auto Scaling group should take when the lifecycle hook timeout elapses or if an
         * unexpected failure occurs. The valid values are <code>CONTINUE</code> and <code>ABANDON</code>. The default
         * value is <code>ABANDON</code>.
         * </p>
         * 
         * @param defaultResult
         *        Defines the action the Auto Scaling group should take when the lifecycle hook timeout elapses or if an
         *        unexpected failure occurs. The valid values are <code>CONTINUE</code> and <code>ABANDON</code>. The
         *        default value is <code>ABANDON</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultResult(String defaultResult);

        /**
         * <p>
         * The ARN of the target that Amazon EC2 Auto Scaling sends notifications to when an instance is in the
         * transition state for the lifecycle hook. The notification target can be either an SQS queue or an SNS topic.
         * </p>
         * 
         * @param notificationTargetARN
         *        The ARN of the target that Amazon EC2 Auto Scaling sends notifications to when an instance is in the
         *        transition state for the lifecycle hook. The notification target can be either an SQS queue or an SNS
         *        topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTargetARN(String notificationTargetARN);

        /**
         * <p>
         * The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target,
         * for example, an Amazon SNS topic or an Amazon SQS queue.
         * </p>
         * 
         * @param roleARN
         *        The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification
         *        target, for example, an Amazon SNS topic or an Amazon SQS queue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);
    }

    static final class BuilderImpl implements Builder {
        private String lifecycleHookName;

        private String lifecycleTransition;

        private String notificationMetadata;

        private Integer heartbeatTimeout;

        private String defaultResult;

        private String notificationTargetARN;

        private String roleARN;

        private BuilderImpl() {
        }

        private BuilderImpl(LifecycleHookSpecification model) {
            lifecycleHookName(model.lifecycleHookName);
            lifecycleTransition(model.lifecycleTransition);
            notificationMetadata(model.notificationMetadata);
            heartbeatTimeout(model.heartbeatTimeout);
            defaultResult(model.defaultResult);
            notificationTargetARN(model.notificationTargetARN);
            roleARN(model.roleARN);
        }

        public final String getLifecycleHookName() {
            return lifecycleHookName;
        }

        @Override
        public final Builder lifecycleHookName(String lifecycleHookName) {
            this.lifecycleHookName = lifecycleHookName;
            return this;
        }

        public final void setLifecycleHookName(String lifecycleHookName) {
            this.lifecycleHookName = lifecycleHookName;
        }

        public final String getLifecycleTransition() {
            return lifecycleTransition;
        }

        @Override
        public final Builder lifecycleTransition(String lifecycleTransition) {
            this.lifecycleTransition = lifecycleTransition;
            return this;
        }

        public final void setLifecycleTransition(String lifecycleTransition) {
            this.lifecycleTransition = lifecycleTransition;
        }

        public final String getNotificationMetadata() {
            return notificationMetadata;
        }

        @Override
        public final Builder notificationMetadata(String notificationMetadata) {
            this.notificationMetadata = notificationMetadata;
            return this;
        }

        public final void setNotificationMetadata(String notificationMetadata) {
            this.notificationMetadata = notificationMetadata;
        }

        public final Integer getHeartbeatTimeout() {
            return heartbeatTimeout;
        }

        @Override
        public final Builder heartbeatTimeout(Integer heartbeatTimeout) {
            this.heartbeatTimeout = heartbeatTimeout;
            return this;
        }

        public final void setHeartbeatTimeout(Integer heartbeatTimeout) {
            this.heartbeatTimeout = heartbeatTimeout;
        }

        public final String getDefaultResult() {
            return defaultResult;
        }

        @Override
        public final Builder defaultResult(String defaultResult) {
            this.defaultResult = defaultResult;
            return this;
        }

        public final void setDefaultResult(String defaultResult) {
            this.defaultResult = defaultResult;
        }

        public final String getNotificationTargetARN() {
            return notificationTargetARN;
        }

        @Override
        public final Builder notificationTargetARN(String notificationTargetARN) {
            this.notificationTargetARN = notificationTargetARN;
            return this;
        }

        public final void setNotificationTargetARN(String notificationTargetARN) {
            this.notificationTargetARN = notificationTargetARN;
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        @Override
        public LifecycleHookSpecification build() {
            return new LifecycleHookSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
