/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The location in Amazon S3 where query results are stored and the encryption option, if any, used for query results.
 * These are known as "client-side settings". If workgroup settings override client-side settings, then the query uses
 * the workgroup settings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResultConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ResultConfiguration.Builder, ResultConfiguration> {
    private static final SdkField<String> OUTPUT_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResultConfiguration::outputLocation)).setter(setter(Builder::outputLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocation").build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(ResultConfiguration::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OUTPUT_LOCATION_FIELD,
            ENCRYPTION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String outputLocation;

    private final EncryptionConfiguration encryptionConfiguration;

    private ResultConfiguration(BuilderImpl builder) {
        this.outputLocation = builder.outputLocation;
        this.encryptionConfiguration = builder.encryptionConfiguration;
    }

    /**
     * <p>
     * The location in Amazon S3 where your query results are stored, such as <code>s3://path/to/query/bucket/</code>.
     * To run the query, you must specify the query results location using one of the ways: either for individual
     * queries using either this setting (client-side), or in the workgroup, using <a>WorkGroupConfiguration</a>. If
     * none of them is set, Athena issues an error that no output location is provided. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup settings
     * override client-side settings, then the query uses the settings specified for the workgroup. See
     * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return The location in Amazon S3 where your query results are stored, such as
     *         <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location
     *         using one of the ways: either for individual queries using either this setting (client-side), or in the
     *         workgroup, using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error that no
     *         output location is provided. For more information, see <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup
     *         settings override client-side settings, then the query uses the settings specified for the workgroup. See
     *         <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public String outputLocation() {
        return outputLocation;
    }

    /**
     * <p>
     * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
     * <code>SSE-KMS</code> or <code>CSE-KMS</code>) and key information. This is a client-side setting. If workgroup
     * settings override client-side settings, then the query uses the encryption configuration that is specified for
     * the workgroup, and also uses the location for storing query results specified in the workgroup. See
     * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings Override
     * Client-Side Settings</a>.
     * </p>
     * 
     * @return If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
     *         <code>SSE-KMS</code> or <code>CSE-KMS</code>) and key information. This is a client-side setting. If
     *         workgroup settings override client-side settings, then the query uses the encryption configuration that
     *         is specified for the workgroup, and also uses the location for storing query results specified in the
     *         workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
     *         Override Client-Side Settings</a>.
     */
    public EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(outputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResultConfiguration)) {
            return false;
        }
        ResultConfiguration other = (ResultConfiguration) obj;
        return Objects.equals(outputLocation(), other.outputLocation())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ResultConfiguration").add("OutputLocation", outputLocation())
                .add("EncryptionConfiguration", encryptionConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OutputLocation":
            return Optional.ofNullable(clazz.cast(outputLocation()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResultConfiguration, T> g) {
        return obj -> g.apply((ResultConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResultConfiguration> {
        /**
         * <p>
         * The location in Amazon S3 where your query results are stored, such as
         * <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location using
         * one of the ways: either for individual queries using either this setting (client-side), or in the workgroup,
         * using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error that no output location
         * is provided. For more information, see <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup settings
         * override client-side settings, then the query uses the settings specified for the workgroup. See
         * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param outputLocation
         *        The location in Amazon S3 where your query results are stored, such as
         *        <code>s3://path/to/query/bucket/</code>. To run the query, you must specify the query results location
         *        using one of the ways: either for individual queries using either this setting (client-side), or in
         *        the workgroup, using <a>WorkGroupConfiguration</a>. If none of them is set, Athena issues an error
         *        that no output location is provided. For more information, see <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a>. If workgroup
         *        settings override client-side settings, then the query uses the settings specified for the workgroup.
         *        See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputLocation(String outputLocation);

        /**
         * <p>
         * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         * <code>SSE-KMS</code> or <code>CSE-KMS</code>) and key information. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the encryption configuration that is
         * specified for the workgroup, and also uses the location for storing query results specified in the workgroup.
         * See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * 
         * @param encryptionConfiguration
         *        If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         *        <code>SSE-KMS</code> or <code>CSE-KMS</code>) and key information. This is a client-side setting. If
         *        workgroup settings override client-side settings, then the query uses the encryption configuration
         *        that is specified for the workgroup, and also uses the location for storing query results specified in
         *        the workgroup. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup
         *        Settings Override Client-Side Settings</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * If query results are encrypted in Amazon S3, indicates the encryption option used (for example,
         * <code>SSE-KMS</code> or <code>CSE-KMS</code>) and key information. This is a client-side setting. If
         * workgroup settings override client-side settings, then the query uses the encryption configuration that is
         * specified for the workgroup, and also uses the location for storing query results specified in the workgroup.
         * See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a> and <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/workgroups-settings-override.html">Workgroup Settings
         * Override Client-Side Settings</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding the
         * need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String outputLocation;

        private EncryptionConfiguration encryptionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ResultConfiguration model) {
            outputLocation(model.outputLocation);
            encryptionConfiguration(model.encryptionConfiguration);
        }

        public final String getOutputLocation() {
            return outputLocation;
        }

        @Override
        public final Builder outputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
            return this;
        }

        public final void setOutputLocation(String outputLocation) {
            this.outputLocation = outputLocation;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public ResultConfiguration build() {
            return new ResultConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
