/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arcregionswitch.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.arcregionswitch.ArcRegionSwitchAsyncClient;
import software.amazon.awssdk.services.arcregionswitch.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.arcregionswitch.model.ArcRegionSwitchRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultArcRegionSwitchAsyncWaiter implements ArcRegionSwitchAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final ArcRegionSwitchAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetPlanEvaluationStatusResponse> planEvaluationStatusPassedWaiter;

    private final AsyncWaiter<GetPlanExecutionResponse> planExecutionCompletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultArcRegionSwitchAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ArcRegionSwitchAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.planEvaluationStatusPassedWaiter = AsyncWaiter.builder(GetPlanEvaluationStatusResponse.class)
                .acceptors(planEvaluationStatusPassedWaiterAcceptors())
                .overrideConfiguration(planEvaluationStatusPassedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.planExecutionCompletedWaiter = AsyncWaiter.builder(GetPlanExecutionResponse.class)
                .acceptors(planExecutionCompletedWaiterAcceptors())
                .overrideConfiguration(planExecutionCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPlanEvaluationStatusResponse>> waitUntilPlanEvaluationStatusPassed(
            GetPlanEvaluationStatusRequest getPlanEvaluationStatusRequest) {
        return planEvaluationStatusPassedWaiter.runAsync(() -> client
                .getPlanEvaluationStatus(applyWaitersUserAgent(getPlanEvaluationStatusRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPlanEvaluationStatusResponse>> waitUntilPlanEvaluationStatusPassed(
            GetPlanEvaluationStatusRequest getPlanEvaluationStatusRequest, WaiterOverrideConfiguration overrideConfig) {
        return planEvaluationStatusPassedWaiter.runAsync(
                () -> client.getPlanEvaluationStatus(applyWaitersUserAgent(getPlanEvaluationStatusRequest)),
                planEvaluationStatusPassedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPlanExecutionResponse>> waitUntilPlanExecutionCompleted(
            GetPlanExecutionRequest getPlanExecutionRequest) {
        return planExecutionCompletedWaiter.runAsync(() -> client
                .getPlanExecution(applyWaitersUserAgent(getPlanExecutionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPlanExecutionResponse>> waitUntilPlanExecutionCompleted(
            GetPlanExecutionRequest getPlanExecutionRequest, WaiterOverrideConfiguration overrideConfig) {
        return planExecutionCompletedWaiter.runAsync(
                () -> client.getPlanExecution(applyWaitersUserAgent(getPlanExecutionRequest)),
                planExecutionCompletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetPlanEvaluationStatusResponse>> planEvaluationStatusPassedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPlanEvaluationStatusResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("evaluationState").value(), "passed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("evaluationState").value(), "actionRequired");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (evaluationState=actionRequired) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("evaluationState").value(), "pendingEvaluation");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPlanExecutionResponse>> planExecutionCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPlanExecutionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("executionState").value(), "completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("executionState").value(), "completedWithExceptions");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("executionState").value(), "failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (executionState=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("executionState").value(), "canceled");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (executionState=canceled) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("executionState").value(), "planExecutionTimedOut");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (executionState=planExecutionTimedOut) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration planEvaluationStatusPassedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration planExecutionCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ArcRegionSwitchAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ArcRegionSwitchRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ArcRegionSwitchAsyncWaiter.Builder {
        private ArcRegionSwitchAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public ArcRegionSwitchAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public ArcRegionSwitchAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ArcRegionSwitchAsyncWaiter.Builder client(ArcRegionSwitchAsyncClient client) {
            this.client = client;
            return this;
        }

        public ArcRegionSwitchAsyncWaiter build() {
            return new DefaultArcRegionSwitchAsyncWaiter(this);
        }
    }
}
