/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appsync;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appsync.model.AccessDeniedException;
import software.amazon.awssdk.services.appsync.model.ApiKeyLimitExceededException;
import software.amazon.awssdk.services.appsync.model.ApiKeyValidityOutOfBoundsException;
import software.amazon.awssdk.services.appsync.model.ApiLimitExceededException;
import software.amazon.awssdk.services.appsync.model.AppSyncException;
import software.amazon.awssdk.services.appsync.model.BadRequestException;
import software.amazon.awssdk.services.appsync.model.ConcurrentModificationException;
import software.amazon.awssdk.services.appsync.model.CreateApiCacheRequest;
import software.amazon.awssdk.services.appsync.model.CreateApiCacheResponse;
import software.amazon.awssdk.services.appsync.model.CreateApiKeyRequest;
import software.amazon.awssdk.services.appsync.model.CreateApiKeyResponse;
import software.amazon.awssdk.services.appsync.model.CreateDataSourceRequest;
import software.amazon.awssdk.services.appsync.model.CreateDataSourceResponse;
import software.amazon.awssdk.services.appsync.model.CreateFunctionRequest;
import software.amazon.awssdk.services.appsync.model.CreateFunctionResponse;
import software.amazon.awssdk.services.appsync.model.CreateGraphqlApiRequest;
import software.amazon.awssdk.services.appsync.model.CreateGraphqlApiResponse;
import software.amazon.awssdk.services.appsync.model.CreateResolverRequest;
import software.amazon.awssdk.services.appsync.model.CreateResolverResponse;
import software.amazon.awssdk.services.appsync.model.CreateTypeRequest;
import software.amazon.awssdk.services.appsync.model.CreateTypeResponse;
import software.amazon.awssdk.services.appsync.model.DeleteApiCacheRequest;
import software.amazon.awssdk.services.appsync.model.DeleteApiCacheResponse;
import software.amazon.awssdk.services.appsync.model.DeleteApiKeyRequest;
import software.amazon.awssdk.services.appsync.model.DeleteApiKeyResponse;
import software.amazon.awssdk.services.appsync.model.DeleteDataSourceRequest;
import software.amazon.awssdk.services.appsync.model.DeleteDataSourceResponse;
import software.amazon.awssdk.services.appsync.model.DeleteFunctionRequest;
import software.amazon.awssdk.services.appsync.model.DeleteFunctionResponse;
import software.amazon.awssdk.services.appsync.model.DeleteGraphqlApiRequest;
import software.amazon.awssdk.services.appsync.model.DeleteGraphqlApiResponse;
import software.amazon.awssdk.services.appsync.model.DeleteResolverRequest;
import software.amazon.awssdk.services.appsync.model.DeleteResolverResponse;
import software.amazon.awssdk.services.appsync.model.DeleteTypeRequest;
import software.amazon.awssdk.services.appsync.model.DeleteTypeResponse;
import software.amazon.awssdk.services.appsync.model.FlushApiCacheRequest;
import software.amazon.awssdk.services.appsync.model.FlushApiCacheResponse;
import software.amazon.awssdk.services.appsync.model.GetApiCacheRequest;
import software.amazon.awssdk.services.appsync.model.GetApiCacheResponse;
import software.amazon.awssdk.services.appsync.model.GetDataSourceRequest;
import software.amazon.awssdk.services.appsync.model.GetDataSourceResponse;
import software.amazon.awssdk.services.appsync.model.GetFunctionRequest;
import software.amazon.awssdk.services.appsync.model.GetFunctionResponse;
import software.amazon.awssdk.services.appsync.model.GetGraphqlApiRequest;
import software.amazon.awssdk.services.appsync.model.GetGraphqlApiResponse;
import software.amazon.awssdk.services.appsync.model.GetIntrospectionSchemaRequest;
import software.amazon.awssdk.services.appsync.model.GetIntrospectionSchemaResponse;
import software.amazon.awssdk.services.appsync.model.GetResolverRequest;
import software.amazon.awssdk.services.appsync.model.GetResolverResponse;
import software.amazon.awssdk.services.appsync.model.GetSchemaCreationStatusRequest;
import software.amazon.awssdk.services.appsync.model.GetSchemaCreationStatusResponse;
import software.amazon.awssdk.services.appsync.model.GetTypeRequest;
import software.amazon.awssdk.services.appsync.model.GetTypeResponse;
import software.amazon.awssdk.services.appsync.model.GraphQlSchemaException;
import software.amazon.awssdk.services.appsync.model.InternalFailureException;
import software.amazon.awssdk.services.appsync.model.LimitExceededException;
import software.amazon.awssdk.services.appsync.model.ListApiKeysRequest;
import software.amazon.awssdk.services.appsync.model.ListApiKeysResponse;
import software.amazon.awssdk.services.appsync.model.ListDataSourcesRequest;
import software.amazon.awssdk.services.appsync.model.ListDataSourcesResponse;
import software.amazon.awssdk.services.appsync.model.ListFunctionsRequest;
import software.amazon.awssdk.services.appsync.model.ListFunctionsResponse;
import software.amazon.awssdk.services.appsync.model.ListGraphqlApisRequest;
import software.amazon.awssdk.services.appsync.model.ListGraphqlApisResponse;
import software.amazon.awssdk.services.appsync.model.ListResolversByFunctionRequest;
import software.amazon.awssdk.services.appsync.model.ListResolversByFunctionResponse;
import software.amazon.awssdk.services.appsync.model.ListResolversRequest;
import software.amazon.awssdk.services.appsync.model.ListResolversResponse;
import software.amazon.awssdk.services.appsync.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appsync.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appsync.model.ListTypesRequest;
import software.amazon.awssdk.services.appsync.model.ListTypesResponse;
import software.amazon.awssdk.services.appsync.model.NotFoundException;
import software.amazon.awssdk.services.appsync.model.StartSchemaCreationRequest;
import software.amazon.awssdk.services.appsync.model.StartSchemaCreationResponse;
import software.amazon.awssdk.services.appsync.model.TagResourceRequest;
import software.amazon.awssdk.services.appsync.model.TagResourceResponse;
import software.amazon.awssdk.services.appsync.model.UnauthorizedException;
import software.amazon.awssdk.services.appsync.model.UntagResourceRequest;
import software.amazon.awssdk.services.appsync.model.UntagResourceResponse;
import software.amazon.awssdk.services.appsync.model.UpdateApiCacheRequest;
import software.amazon.awssdk.services.appsync.model.UpdateApiCacheResponse;
import software.amazon.awssdk.services.appsync.model.UpdateApiKeyRequest;
import software.amazon.awssdk.services.appsync.model.UpdateApiKeyResponse;
import software.amazon.awssdk.services.appsync.model.UpdateDataSourceRequest;
import software.amazon.awssdk.services.appsync.model.UpdateDataSourceResponse;
import software.amazon.awssdk.services.appsync.model.UpdateFunctionRequest;
import software.amazon.awssdk.services.appsync.model.UpdateFunctionResponse;
import software.amazon.awssdk.services.appsync.model.UpdateGraphqlApiRequest;
import software.amazon.awssdk.services.appsync.model.UpdateGraphqlApiResponse;
import software.amazon.awssdk.services.appsync.model.UpdateResolverRequest;
import software.amazon.awssdk.services.appsync.model.UpdateResolverResponse;
import software.amazon.awssdk.services.appsync.model.UpdateTypeRequest;
import software.amazon.awssdk.services.appsync.model.UpdateTypeResponse;
import software.amazon.awssdk.services.appsync.transform.CreateApiCacheRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateApiKeyRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateFunctionRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateGraphqlApiRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateResolverRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.CreateTypeRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteApiCacheRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteApiKeyRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteDataSourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteFunctionRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteGraphqlApiRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteResolverRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.DeleteTypeRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.FlushApiCacheRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetApiCacheRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetDataSourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetFunctionRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetGraphqlApiRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetIntrospectionSchemaRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetResolverRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetSchemaCreationStatusRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.GetTypeRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListApiKeysRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListDataSourcesRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListFunctionsRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListGraphqlApisRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListResolversByFunctionRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListResolversRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.ListTypesRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.StartSchemaCreationRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateApiCacheRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateApiKeyRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateFunctionRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateGraphqlApiRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateResolverRequestMarshaller;
import software.amazon.awssdk.services.appsync.transform.UpdateTypeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppSyncAsyncClient}.
 *
 * @see AppSyncAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppSyncAsyncClient implements AppSyncAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppSyncAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppSyncAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a cache for the GraphQL API.
     * </p>
     *
     * @param createApiCacheRequest
     *        Represents the input of a <code>CreateApiCache</code> operation.
     * @return A Java Future containing the result of the CreateApiCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateApiCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateApiCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApiCacheResponse> createApiCache(CreateApiCacheRequest createApiCacheRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApiCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApiCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApiCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApiCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApiCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApiCacheRequest, CreateApiCacheResponse>()
                            .withOperationName("CreateApiCache")
                            .withMarshaller(new CreateApiCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApiCacheRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createApiCacheRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateApiCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a unique key that you can distribute to clients who are executing your API.
     * </p>
     *
     * @param createApiKeyRequest
     * @return A Java Future containing the result of the CreateApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>ApiKeyLimitExceededException The API key exceeded a limit. Try your request again.</li>
     *         <li>ApiKeyValidityOutOfBoundsException The API key expiration must be set to a value between 1 and 365
     *         days from creation (for <code>CreateApiKey</code>) or from update (for <code>UpdateApiKey</code>).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateApiKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateApiKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApiKeyResponse> createApiKey(CreateApiKeyRequest createApiKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateApiKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApiKeyRequest, CreateApiKeyResponse>()
                            .withOperationName("CreateApiKey").withMarshaller(new CreateApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApiKeyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createApiKeyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> object.
     * </p>
     *
     * @param createDataSourceRequest
     * @return A Java Future containing the result of the CreateDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataSourceResponse> createDataSource(CreateDataSourceRequest createDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceRequest, CreateDataSourceResponse>()
                            .withOperationName("CreateDataSource")
                            .withMarshaller(new CreateDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDataSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>Function</code> object.
     * </p>
     * <p>
     * A function is a reusable entity. Multiple functions can be used to compose the resolver logic.
     * </p>
     *
     * @param createFunctionRequest
     * @return A Java Future containing the result of the CreateFunction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFunctionResponse> createFunction(CreateFunctionRequest createFunctionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFunctionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFunction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFunctionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFunctionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFunctionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFunctionRequest, CreateFunctionResponse>()
                            .withOperationName("CreateFunction")
                            .withMarshaller(new CreateFunctionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFunctionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createFunctionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateFunctionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>GraphqlApi</code> object.
     * </p>
     *
     * @param createGraphqlApiRequest
     * @return A Java Future containing the result of the CreateGraphqlApi operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>ApiLimitExceededException The GraphQL API exceeded a limit. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateGraphqlApi
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateGraphqlApi" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphqlApiResponse> createGraphqlApi(CreateGraphqlApiRequest createGraphqlApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphqlApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraphqlApi");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphqlApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGraphqlApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphqlApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphqlApiRequest, CreateGraphqlApiResponse>()
                            .withOperationName("CreateGraphqlApi")
                            .withMarshaller(new CreateGraphqlApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGraphqlApiRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGraphqlApiRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateGraphqlApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>Resolver</code> object.
     * </p>
     * <p>
     * A resolver converts incoming requests into a format that a data source can understand and converts the data
     * source's responses into GraphQL.
     * </p>
     *
     * @param createResolverRequest
     * @return A Java Future containing the result of the CreateResolver operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateResolver
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateResolver" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverResponse> createResolver(CreateResolverRequest createResolverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolver");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverRequest, CreateResolverResponse>()
                            .withOperationName("CreateResolver")
                            .withMarshaller(new CreateResolverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createResolverRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createResolverRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateResolverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>Type</code> object.
     * </p>
     *
     * @param createTypeRequest
     * @return A Java Future containing the result of the CreateType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.CreateType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/CreateType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTypeResponse> createType(CreateTypeRequest createTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTypeRequest, CreateTypeResponse>().withOperationName("CreateType")
                            .withMarshaller(new CreateTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTypeRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>ApiCache</code> object.
     * </p>
     *
     * @param deleteApiCacheRequest
     *        Represents the input of a <code>DeleteApiCache</code> operation.
     * @return A Java Future containing the result of the DeleteApiCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteApiCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteApiCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApiCacheResponse> deleteApiCache(DeleteApiCacheRequest deleteApiCacheRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApiCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApiCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApiCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApiCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApiCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApiCacheRequest, DeleteApiCacheResponse>()
                            .withOperationName("DeleteApiCache")
                            .withMarshaller(new DeleteApiCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApiCacheRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApiCacheRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApiCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an API key.
     * </p>
     *
     * @param deleteApiKeyRequest
     * @return A Java Future containing the result of the DeleteApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteApiKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteApiKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApiKeyResponse> deleteApiKey(DeleteApiKeyRequest deleteApiKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteApiKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApiKeyRequest, DeleteApiKeyResponse>()
                            .withOperationName("DeleteApiKey").withMarshaller(new DeleteApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApiKeyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApiKeyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>DataSource</code> object.
     * </p>
     *
     * @param deleteDataSourceRequest
     * @return A Java Future containing the result of the DeleteDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataSourceResponse> deleteDataSource(DeleteDataSourceRequest deleteDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataSourceRequest, DeleteDataSourceResponse>()
                            .withOperationName("DeleteDataSource")
                            .withMarshaller(new DeleteDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDataSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Function</code>.
     * </p>
     *
     * @param deleteFunctionRequest
     * @return A Java Future containing the result of the DeleteFunction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFunctionResponse> deleteFunction(DeleteFunctionRequest deleteFunctionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFunctionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFunction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFunctionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFunctionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFunctionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFunctionRequest, DeleteFunctionResponse>()
                            .withOperationName("DeleteFunction")
                            .withMarshaller(new DeleteFunctionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFunctionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteFunctionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteFunctionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>GraphqlApi</code> object.
     * </p>
     *
     * @param deleteGraphqlApiRequest
     * @return A Java Future containing the result of the DeleteGraphqlApi operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteGraphqlApi
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteGraphqlApi" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGraphqlApiResponse> deleteGraphqlApi(DeleteGraphqlApiRequest deleteGraphqlApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphqlApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraphqlApi");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGraphqlApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGraphqlApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGraphqlApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGraphqlApiRequest, DeleteGraphqlApiResponse>()
                            .withOperationName("DeleteGraphqlApi")
                            .withMarshaller(new DeleteGraphqlApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGraphqlApiRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGraphqlApiRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteGraphqlApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Resolver</code> object.
     * </p>
     *
     * @param deleteResolverRequest
     * @return A Java Future containing the result of the DeleteResolver operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteResolver
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteResolver" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverResponse> deleteResolver(DeleteResolverRequest deleteResolverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolver");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverRequest, DeleteResolverResponse>()
                            .withOperationName("DeleteResolver")
                            .withMarshaller(new DeleteResolverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResolverRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteResolverRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteResolverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Type</code> object.
     * </p>
     *
     * @param deleteTypeRequest
     * @return A Java Future containing the result of the DeleteType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.DeleteType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/DeleteType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTypeResponse> deleteType(DeleteTypeRequest deleteTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTypeRequest, DeleteTypeResponse>().withOperationName("DeleteType")
                            .withMarshaller(new DeleteTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTypeRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Flushes an <code>ApiCache</code> object.
     * </p>
     *
     * @param flushApiCacheRequest
     *        Represents the input of a <code>FlushApiCache</code> operation.
     * @return A Java Future containing the result of the FlushApiCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.FlushApiCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/FlushApiCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<FlushApiCacheResponse> flushApiCache(FlushApiCacheRequest flushApiCacheRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, flushApiCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "FlushApiCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<FlushApiCacheResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    FlushApiCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<FlushApiCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<FlushApiCacheRequest, FlushApiCacheResponse>()
                            .withOperationName("FlushApiCache")
                            .withMarshaller(new FlushApiCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(flushApiCacheRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = flushApiCacheRequest.overrideConfiguration().orElse(null);
            CompletableFuture<FlushApiCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an <code>ApiCache</code> object.
     * </p>
     *
     * @param getApiCacheRequest
     *        Represents the input of a <code>GetApiCache</code> operation.
     * @return A Java Future containing the result of the GetApiCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetApiCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetApiCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetApiCacheResponse> getApiCache(GetApiCacheRequest getApiCacheRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApiCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApiCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApiCacheResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetApiCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApiCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApiCacheRequest, GetApiCacheResponse>()
                            .withOperationName("GetApiCache").withMarshaller(new GetApiCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApiCacheRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getApiCacheRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetApiCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a <code>DataSource</code> object.
     * </p>
     *
     * @param getDataSourceRequest
     * @return A Java Future containing the result of the GetDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataSourceResponse> getDataSource(GetDataSourceRequest getDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataSourceRequest, GetDataSourceResponse>()
                            .withOperationName("GetDataSource")
                            .withMarshaller(new GetDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDataSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get a <code>Function</code>.
     * </p>
     *
     * @param getFunctionRequest
     * @return A Java Future containing the result of the GetFunction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFunctionResponse> getFunction(GetFunctionRequest getFunctionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFunctionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFunction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFunctionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFunctionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFunctionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFunctionRequest, GetFunctionResponse>()
                            .withOperationName("GetFunction").withMarshaller(new GetFunctionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFunctionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getFunctionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetFunctionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a <code>GraphqlApi</code> object.
     * </p>
     *
     * @param getGraphqlApiRequest
     * @return A Java Future containing the result of the GetGraphqlApi operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetGraphqlApi
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetGraphqlApi" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGraphqlApiResponse> getGraphqlApi(GetGraphqlApiRequest getGraphqlApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphqlApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraphqlApi");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGraphqlApiResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGraphqlApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGraphqlApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGraphqlApiRequest, GetGraphqlApiResponse>()
                            .withOperationName("GetGraphqlApi")
                            .withMarshaller(new GetGraphqlApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGraphqlApiRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getGraphqlApiRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetGraphqlApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the introspection schema for a GraphQL API.
     * </p>
     *
     * @param getIntrospectionSchemaRequest
     * @return A Java Future containing the result of the GetIntrospectionSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GraphQlSchemaException The GraphQL schema is not valid.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetIntrospectionSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetIntrospectionSchema" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIntrospectionSchemaResponse> getIntrospectionSchema(
            GetIntrospectionSchemaRequest getIntrospectionSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIntrospectionSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIntrospectionSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetIntrospectionSchemaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIntrospectionSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIntrospectionSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIntrospectionSchemaRequest, GetIntrospectionSchemaResponse>()
                            .withOperationName("GetIntrospectionSchema")
                            .withMarshaller(new GetIntrospectionSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIntrospectionSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIntrospectionSchemaRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetIntrospectionSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a <code>Resolver</code> object.
     * </p>
     *
     * @param getResolverRequest
     * @return A Java Future containing the result of the GetResolver operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetResolver
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetResolver" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverResponse> getResolver(GetResolverRequest getResolverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolver");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetResolverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRequest, GetResolverResponse>()
                            .withOperationName("GetResolver").withMarshaller(new GetResolverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetResolverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current status of a schema creation operation.
     * </p>
     *
     * @param getSchemaCreationStatusRequest
     * @return A Java Future containing the result of the GetSchemaCreationStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetSchemaCreationStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetSchemaCreationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSchemaCreationStatusResponse> getSchemaCreationStatus(
            GetSchemaCreationStatusRequest getSchemaCreationStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaCreationStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchemaCreationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSchemaCreationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSchemaCreationStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSchemaCreationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSchemaCreationStatusRequest, GetSchemaCreationStatusResponse>()
                            .withOperationName("GetSchemaCreationStatus")
                            .withMarshaller(new GetSchemaCreationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSchemaCreationStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getSchemaCreationStatusRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetSchemaCreationStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a <code>Type</code> object.
     * </p>
     *
     * @param getTypeRequest
     * @return A Java Future containing the result of the GetType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.GetType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/GetType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTypeResponse> getType(GetTypeRequest getTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTypeRequest, GetTypeResponse>().withOperationName("GetType")
                            .withMarshaller(new GetTypeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getTypeRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the API keys for a given API.
     * </p>
     * <note>
     * <p>
     * API keys are deleted automatically 60 days after they expire. However, they may still be included in the response
     * until they have actually been deleted. You can safely call <code>DeleteApiKey</code> to manually delete a key
     * before it's automatically deleted.
     * </p>
     * </note>
     *
     * @param listApiKeysRequest
     * @return A Java Future containing the result of the ListApiKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListApiKeys
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListApiKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListApiKeysResponse> listApiKeys(ListApiKeysRequest listApiKeysRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApiKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApiKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApiKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListApiKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApiKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApiKeysRequest, ListApiKeysResponse>()
                            .withOperationName("ListApiKeys").withMarshaller(new ListApiKeysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApiKeysRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listApiKeysRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListApiKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the data sources for a given API.
     * </p>
     *
     * @param listDataSourcesRequest
     * @return A Java Future containing the result of the ListDataSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListDataSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListDataSources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataSourcesResponse> listDataSources(ListDataSourcesRequest listDataSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataSourcesRequest, ListDataSourcesResponse>()
                            .withOperationName("ListDataSources")
                            .withMarshaller(new ListDataSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDataSourcesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDataSourcesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDataSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List multiple functions.
     * </p>
     *
     * @param listFunctionsRequest
     * @return A Java Future containing the result of the ListFunctions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListFunctions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListFunctions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFunctionsResponse> listFunctions(ListFunctionsRequest listFunctionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFunctionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFunctions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFunctionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFunctionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFunctionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFunctionsRequest, ListFunctionsResponse>()
                            .withOperationName("ListFunctions")
                            .withMarshaller(new ListFunctionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFunctionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFunctionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListFunctionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your GraphQL APIs.
     * </p>
     *
     * @param listGraphqlApisRequest
     * @return A Java Future containing the result of the ListGraphqlApis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListGraphqlApis
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListGraphqlApis" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGraphqlApisResponse> listGraphqlApis(ListGraphqlApisRequest listGraphqlApisRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphqlApisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphqlApis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGraphqlApisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGraphqlApisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGraphqlApisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGraphqlApisRequest, ListGraphqlApisResponse>()
                            .withOperationName("ListGraphqlApis")
                            .withMarshaller(new ListGraphqlApisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGraphqlApisRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGraphqlApisRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListGraphqlApisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resolvers for a given API and type.
     * </p>
     *
     * @param listResolversRequest
     * @return A Java Future containing the result of the ListResolvers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListResolvers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListResolvers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolversResponse> listResolvers(ListResolversRequest listResolversRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolversRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolvers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolversResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResolversResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolversResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolversRequest, ListResolversResponse>()
                            .withOperationName("ListResolvers")
                            .withMarshaller(new ListResolversRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolversRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolversRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListResolversResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the resolvers that are associated with a specific function.
     * </p>
     *
     * @param listResolversByFunctionRequest
     * @return A Java Future containing the result of the ListResolversByFunction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListResolversByFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListResolversByFunction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolversByFunctionResponse> listResolversByFunction(
            ListResolversByFunctionRequest listResolversByFunctionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolversByFunctionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolversByFunction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolversByFunctionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolversByFunctionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolversByFunctionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolversByFunctionRequest, ListResolversByFunctionResponse>()
                            .withOperationName("ListResolversByFunction")
                            .withMarshaller(new ListResolversByFunctionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolversByFunctionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolversByFunctionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListResolversByFunctionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the types for a given API.
     * </p>
     *
     * @param listTypesRequest
     * @return A Java Future containing the result of the ListTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.ListTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/ListTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTypesResponse> listTypes(ListTypesRequest listTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTypesRequest, ListTypesResponse>().withOperationName("ListTypes")
                            .withMarshaller(new ListTypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTypesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTypesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a new schema to your GraphQL API.
     * </p>
     * <p>
     * This operation is asynchronous. Use to determine when it has completed.
     * </p>
     *
     * @param startSchemaCreationRequest
     * @return A Java Future containing the result of the StartSchemaCreation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.StartSchemaCreation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/StartSchemaCreation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSchemaCreationResponse> startSchemaCreation(
            StartSchemaCreationRequest startSchemaCreationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSchemaCreationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSchemaCreation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSchemaCreationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSchemaCreationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSchemaCreationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSchemaCreationRequest, StartSchemaCreationResponse>()
                            .withOperationName("StartSchemaCreation")
                            .withMarshaller(new StartSchemaCreationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startSchemaCreationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startSchemaCreationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<StartSchemaCreationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a resource with user-supplied tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the cache for the GraphQL API.
     * </p>
     *
     * @param updateApiCacheRequest
     *        Represents the input of a <code>UpdateApiCache</code> operation.
     * @return A Java Future containing the result of the UpdateApiCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateApiCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateApiCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApiCacheResponse> updateApiCache(UpdateApiCacheRequest updateApiCacheRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApiCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApiCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApiCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApiCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApiCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApiCacheRequest, UpdateApiCacheResponse>()
                            .withOperationName("UpdateApiCache")
                            .withMarshaller(new UpdateApiCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApiCacheRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateApiCacheRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateApiCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an API key. The key can be updated while it is not deleted.
     * </p>
     *
     * @param updateApiKeyRequest
     * @return A Java Future containing the result of the UpdateApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>LimitExceededException The request exceeded a limit. Try your request again.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>ApiKeyValidityOutOfBoundsException The API key expiration must be set to a value between 1 and 365
     *         days from creation (for <code>CreateApiKey</code>) or from update (for <code>UpdateApiKey</code>).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateApiKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateApiKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApiKeyResponse> updateApiKey(UpdateApiKeyRequest updateApiKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateApiKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApiKeyRequest, UpdateApiKeyResponse>()
                            .withOperationName("UpdateApiKey").withMarshaller(new UpdateApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApiKeyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateApiKeyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <code>DataSource</code> object.
     * </p>
     *
     * @param updateDataSourceRequest
     * @return A Java Future containing the result of the UpdateDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataSourceResponse> updateDataSource(UpdateDataSourceRequest updateDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataSourceRequest, UpdateDataSourceResponse>()
                            .withOperationName("UpdateDataSource")
                            .withMarshaller(new UpdateDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDataSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <code>Function</code> object.
     * </p>
     *
     * @param updateFunctionRequest
     * @return A Java Future containing the result of the UpdateFunction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFunctionResponse> updateFunction(UpdateFunctionRequest updateFunctionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFunctionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFunction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFunctionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFunctionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFunctionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFunctionRequest, UpdateFunctionResponse>()
                            .withOperationName("UpdateFunction")
                            .withMarshaller(new UpdateFunctionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFunctionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFunctionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateFunctionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <code>GraphqlApi</code> object.
     * </p>
     *
     * @param updateGraphqlApiRequest
     * @return A Java Future containing the result of the UpdateGraphqlApi operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>AccessDeniedException You do not have access to perform this operation on this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateGraphqlApi
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateGraphqlApi" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGraphqlApiResponse> updateGraphqlApi(UpdateGraphqlApiRequest updateGraphqlApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGraphqlApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGraphqlApi");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGraphqlApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGraphqlApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGraphqlApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGraphqlApiRequest, UpdateGraphqlApiResponse>()
                            .withOperationName("UpdateGraphqlApi")
                            .withMarshaller(new UpdateGraphqlApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGraphqlApiRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGraphqlApiRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateGraphqlApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <code>Resolver</code> object.
     * </p>
     *
     * @param updateResolverRequest
     * @return A Java Future containing the result of the UpdateResolver operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateResolver
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateResolver" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverResponse> updateResolver(UpdateResolverRequest updateResolverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolver");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverRequest, UpdateResolverResponse>()
                            .withOperationName("UpdateResolver")
                            .withMarshaller(new UpdateResolverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResolverRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResolverRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateResolverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a <code>Type</code> object.
     * </p>
     *
     * @param updateTypeRequest
     * @return A Java Future containing the result of the UpdateType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request is not well formed. For example, a value is invalid or a required
     *         field is missing. Check the field values, and then try again.</li>
     *         <li>ConcurrentModificationException Another modification is in progress at this time and it must complete
     *         before you can make your change.</li>
     *         <li>NotFoundException The resource specified in the request was not found. Check the resource, and then
     *         try again.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>InternalFailureException An internal AWS AppSync error occurred. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppSyncAsyncClient.UpdateType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/UpdateType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTypeResponse> updateType(UpdateTypeRequest updateTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTypeRequest, UpdateTypeResponse>().withOperationName("UpdateType")
                            .withMarshaller(new UpdateTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTypeRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppSyncException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GraphQLSchemaException")
                                .exceptionBuilderSupplier(GraphQlSchemaException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApiKeyLimitExceededException")
                                .exceptionBuilderSupplier(ApiKeyLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApiKeyValidityOutOfBoundsException")
                                .exceptionBuilderSupplier(ApiKeyValidityOutOfBoundsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApiLimitExceededException")
                                .exceptionBuilderSupplier(ApiLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
