/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains information about a change event that occurred for a service, such as a deployment or
 * configuration change.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ChangeEvent implements SdkPojo, Serializable, ToCopyableBuilder<ChangeEvent.Builder, ChangeEvent> {
    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(ChangeEvent::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(ChangeEvent::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(ChangeEvent::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<Map<String, String>> ENTITY_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Entity")
            .getter(getter(ChangeEvent::entity))
            .setter(setter(Builder::entity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Entity").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CHANGE_EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChangeEventType").getter(getter(ChangeEvent::changeEventTypeAsString))
            .setter(setter(Builder::changeEventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeEventType").build()).build();

    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(ChangeEvent::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(ChangeEvent::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> EVENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventName").getter(getter(ChangeEvent::eventName)).setter(setter(Builder::eventName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMP_FIELD,
            ACCOUNT_ID_FIELD, REGION_FIELD, ENTITY_FIELD, CHANGE_EVENT_TYPE_FIELD, EVENT_ID_FIELD, USER_NAME_FIELD,
            EVENT_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant timestamp;

    private final String accountId;

    private final String region;

    private final Map<String, String> entity;

    private final String changeEventType;

    private final String eventId;

    private final String userName;

    private final String eventName;

    private ChangeEvent(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.accountId = builder.accountId;
        this.region = builder.region;
        this.entity = builder.entity;
        this.changeEventType = builder.changeEventType;
        this.eventId = builder.eventId;
        this.userName = builder.userName;
        this.eventName = builder.eventName;
    }

    /**
     * <p>
     * The timestamp when this change event occurred. When used in a raw HTTP Query API, it is formatted as epoch time
     * in seconds.
     * </p>
     * 
     * @return The timestamp when this change event occurred. When used in a raw HTTP Query API, it is formatted as
     *         epoch time in seconds.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The Amazon Web Services account ID where this change event occurred.
     * </p>
     * 
     * @return The Amazon Web Services account ID where this change event occurred.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Web Services region where this change event occurred.
     * </p>
     * 
     * @return The Amazon Web Services region where this change event occurred.
     */
    public final String region() {
        return region;
    }

    /**
     * For responses, this returns true if the service returned a value for the Entity property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntity() {
        return entity != null && !(entity instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The entity (service or resource) that was affected by this change event, including its key attributes.
     * </p>
     * <p>
     * This is a string-to-string map. It can include the following fields.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Type</code> designates the type of object this is.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of the
     * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Name</code> specifies the name of the object. This is used only if the value of the <code>Type</code> field
     * is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value of the
     * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AwsAccountId</code> specifies the account where this object is in.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Below is an example of a service.
     * </p>
     * <p>
     * <code>{ "Type": "Service", "Name": "visits-service", "Environment": "petclinic-test" }</code>
     * </p>
     * <p>
     * Below is an example of a resource.
     * </p>
     * <p>
     * <code>{ "Type": "AWS::Resource", "ResourceType": "AWS::DynamoDB::Table", "Identifier": "Customers" }</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntity} method.
     * </p>
     * 
     * @return The entity (service or resource) that was affected by this change event, including its key
     *         attributes.</p>
     *         <p>
     *         This is a string-to-string map. It can include the following fields.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Type</code> designates the type of object this is.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of
     *         the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Name</code> specifies the name of the object. This is used only if the value of the
     *         <code>Type</code> field is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>
     *         .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value
     *         of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AwsAccountId</code> specifies the account where this object is in.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Below is an example of a service.
     *         </p>
     *         <p>
     *         <code>{ "Type": "Service", "Name": "visits-service", "Environment": "petclinic-test" }</code>
     *         </p>
     *         <p>
     *         Below is an example of a resource.
     *         </p>
     *         <p>
     *         <code>{ "Type": "AWS::Resource", "ResourceType": "AWS::DynamoDB::Table", "Identifier": "Customers" }</code>
     */
    public final Map<String, String> entity() {
        return entity;
    }

    /**
     * <p>
     * The type of change event that occurred, such as <code>DEPLOYMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeEventType}
     * will return {@link ChangeEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #changeEventTypeAsString}.
     * </p>
     * 
     * @return The type of change event that occurred, such as <code>DEPLOYMENT</code>.
     * @see ChangeEventType
     */
    public final ChangeEventType changeEventType() {
        return ChangeEventType.fromValue(changeEventType);
    }

    /**
     * <p>
     * The type of change event that occurred, such as <code>DEPLOYMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeEventType}
     * will return {@link ChangeEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #changeEventTypeAsString}.
     * </p>
     * 
     * @return The type of change event that occurred, such as <code>DEPLOYMENT</code>.
     * @see ChangeEventType
     */
    public final String changeEventTypeAsString() {
        return changeEventType;
    }

    /**
     * <p>
     * A unique identifier for this change event. For CloudTrail-based events, this is the CloudTrail event id. For
     * other events, this will be <code>Unknown</code>.
     * </p>
     * 
     * @return A unique identifier for this change event. For CloudTrail-based events, this is the CloudTrail event id.
     *         For other events, this will be <code>Unknown</code>.
     */
    public final String eventId() {
        return eventId;
    }

    /**
     * <p>
     * The name of the user who initiated this change event, if available.
     * </p>
     * 
     * @return The name of the user who initiated this change event, if available.
     */
    public final String userName() {
        return userName;
    }

    /**
     * <p>
     * The name or description of this change event.
     * </p>
     * 
     * @return The name or description of this change event.
     */
    public final String eventName() {
        return eventName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntity() ? entity() : null);
        hashCode = 31 * hashCode + Objects.hashCode(changeEventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(eventName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ChangeEvent)) {
            return false;
        }
        ChangeEvent other = (ChangeEvent) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(region(), other.region()) && hasEntity() == other.hasEntity()
                && Objects.equals(entity(), other.entity())
                && Objects.equals(changeEventTypeAsString(), other.changeEventTypeAsString())
                && Objects.equals(eventId(), other.eventId()) && Objects.equals(userName(), other.userName())
                && Objects.equals(eventName(), other.eventName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ChangeEvent").add("Timestamp", timestamp()).add("AccountId", accountId())
                .add("Region", region()).add("Entity", hasEntity() ? entity() : null)
                .add("ChangeEventType", changeEventTypeAsString()).add("EventId", eventId()).add("UserName", userName())
                .add("EventName", eventName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "Entity":
            return Optional.ofNullable(clazz.cast(entity()));
        case "ChangeEventType":
            return Optional.ofNullable(clazz.cast(changeEventTypeAsString()));
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "EventName":
            return Optional.ofNullable(clazz.cast(eventName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Timestamp", TIMESTAMP_FIELD);
        map.put("AccountId", ACCOUNT_ID_FIELD);
        map.put("Region", REGION_FIELD);
        map.put("Entity", ENTITY_FIELD);
        map.put("ChangeEventType", CHANGE_EVENT_TYPE_FIELD);
        map.put("EventId", EVENT_ID_FIELD);
        map.put("UserName", USER_NAME_FIELD);
        map.put("EventName", EVENT_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ChangeEvent, T> g) {
        return obj -> g.apply((ChangeEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ChangeEvent> {
        /**
         * <p>
         * The timestamp when this change event occurred. When used in a raw HTTP Query API, it is formatted as epoch
         * time in seconds.
         * </p>
         * 
         * @param timestamp
         *        The timestamp when this change event occurred. When used in a raw HTTP Query API, it is formatted as
         *        epoch time in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The Amazon Web Services account ID where this change event occurred.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID where this change event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Web Services region where this change event occurred.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services region where this change event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The entity (service or resource) that was affected by this change event, including its key attributes.
         * </p>
         * <p>
         * This is a string-to-string map. It can include the following fields.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Type</code> designates the type of object this is.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of the
         * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Name</code> specifies the name of the object. This is used only if the value of the <code>Type</code>
         * field is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value of
         * the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AwsAccountId</code> specifies the account where this object is in.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Below is an example of a service.
         * </p>
         * <p>
         * <code>{ "Type": "Service", "Name": "visits-service", "Environment": "petclinic-test" }</code>
         * </p>
         * <p>
         * Below is an example of a resource.
         * </p>
         * <p>
         * <code>{ "Type": "AWS::Resource", "ResourceType": "AWS::DynamoDB::Table", "Identifier": "Customers" }</code>
         * </p>
         * 
         * @param entity
         *        The entity (service or resource) that was affected by this change event, including its key
         *        attributes.</p>
         *        <p>
         *        This is a string-to-string map. It can include the following fields.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Type</code> designates the type of object this is.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType</code> specifies the type of the resource. This field is used only when the value
         *        of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Name</code> specifies the name of the object. This is used only if the value of the
         *        <code>Type</code> field is <code>Service</code>, <code>RemoteService</code>, or
         *        <code>AWS::Service</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Identifier</code> identifies the resource objects of this resource. This is used only if the
         *        value of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AwsAccountId</code> specifies the account where this object is in.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Below is an example of a service.
         *        </p>
         *        <p>
         *        <code>{ "Type": "Service", "Name": "visits-service", "Environment": "petclinic-test" }</code>
         *        </p>
         *        <p>
         *        Below is an example of a resource.
         *        </p>
         *        <p>
         *        <code>{ "Type": "AWS::Resource", "ResourceType": "AWS::DynamoDB::Table", "Identifier": "Customers" }</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entity(Map<String, String> entity);

        /**
         * <p>
         * The type of change event that occurred, such as <code>DEPLOYMENT</code>.
         * </p>
         * 
         * @param changeEventType
         *        The type of change event that occurred, such as <code>DEPLOYMENT</code>.
         * @see ChangeEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeEventType
         */
        Builder changeEventType(String changeEventType);

        /**
         * <p>
         * The type of change event that occurred, such as <code>DEPLOYMENT</code>.
         * </p>
         * 
         * @param changeEventType
         *        The type of change event that occurred, such as <code>DEPLOYMENT</code>.
         * @see ChangeEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeEventType
         */
        Builder changeEventType(ChangeEventType changeEventType);

        /**
         * <p>
         * A unique identifier for this change event. For CloudTrail-based events, this is the CloudTrail event id. For
         * other events, this will be <code>Unknown</code>.
         * </p>
         * 
         * @param eventId
         *        A unique identifier for this change event. For CloudTrail-based events, this is the CloudTrail event
         *        id. For other events, this will be <code>Unknown</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * The name of the user who initiated this change event, if available.
         * </p>
         * 
         * @param userName
         *        The name of the user who initiated this change event, if available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * The name or description of this change event.
         * </p>
         * 
         * @param eventName
         *        The name or description of this change event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventName(String eventName);
    }

    static final class BuilderImpl implements Builder {
        private Instant timestamp;

        private String accountId;

        private String region;

        private Map<String, String> entity = DefaultSdkAutoConstructMap.getInstance();

        private String changeEventType;

        private String eventId;

        private String userName;

        private String eventName;

        private BuilderImpl() {
        }

        private BuilderImpl(ChangeEvent model) {
            timestamp(model.timestamp);
            accountId(model.accountId);
            region(model.region);
            entity(model.entity);
            changeEventType(model.changeEventType);
            eventId(model.eventId);
            userName(model.userName);
            eventName(model.eventName);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final Map<String, String> getEntity() {
            if (entity instanceof SdkAutoConstructMap) {
                return null;
            }
            return entity;
        }

        public final void setEntity(Map<String, String> entity) {
            this.entity = AttributesCopier.copy(entity);
        }

        @Override
        public final Builder entity(Map<String, String> entity) {
            this.entity = AttributesCopier.copy(entity);
            return this;
        }

        public final String getChangeEventType() {
            return changeEventType;
        }

        public final void setChangeEventType(String changeEventType) {
            this.changeEventType = changeEventType;
        }

        @Override
        public final Builder changeEventType(String changeEventType) {
            this.changeEventType = changeEventType;
            return this;
        }

        @Override
        public final Builder changeEventType(ChangeEventType changeEventType) {
            this.changeEventType(changeEventType == null ? null : changeEventType.toString());
            return this;
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final String getEventName() {
            return eventName;
        }

        public final void setEventName(String eventName) {
            this.eventName = eventName;
        }

        @Override
        public final Builder eventName(String eventName) {
            this.eventName = eventName;
            return this;
        }

        @Override
        public ChangeEvent build() {
            return new ChangeEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
