/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationdiscovery.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of continuous export descriptions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContinuousExportDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<ContinuousExportDescription.Builder, ContinuousExportDescription> {
    private static final SdkField<String> EXPORT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinuousExportDescription::exportId)).setter(setter(Builder::exportId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exportId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinuousExportDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_DETAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinuousExportDescription::statusDetail)).setter(setter(Builder::statusDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusDetail").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinuousExportDescription::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Bucket").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ContinuousExportDescription::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> STOP_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ContinuousExportDescription::stopTime)).setter(setter(Builder::stopTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stopTime").build()).build();

    private static final SdkField<String> DATA_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinuousExportDescription::dataSourceAsString)).setter(setter(Builder::dataSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSource").build()).build();

    private static final SdkField<Map<String, String>> SCHEMA_STORAGE_CONFIG_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ContinuousExportDescription::schemaStorageConfig))
            .setter(setter(Builder::schemaStorageConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schemaStorageConfig").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPORT_ID_FIELD, STATUS_FIELD,
            STATUS_DETAIL_FIELD, S3_BUCKET_FIELD, START_TIME_FIELD, STOP_TIME_FIELD, DATA_SOURCE_FIELD,
            SCHEMA_STORAGE_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String exportId;

    private final String status;

    private final String statusDetail;

    private final String s3Bucket;

    private final Instant startTime;

    private final Instant stopTime;

    private final String dataSource;

    private final Map<String, String> schemaStorageConfig;

    private ContinuousExportDescription(BuilderImpl builder) {
        this.exportId = builder.exportId;
        this.status = builder.status;
        this.statusDetail = builder.statusDetail;
        this.s3Bucket = builder.s3Bucket;
        this.startTime = builder.startTime;
        this.stopTime = builder.stopTime;
        this.dataSource = builder.dataSource;
        this.schemaStorageConfig = builder.schemaStorageConfig;
    }

    /**
     * <p>
     * The unique ID assigned to this export.
     * </p>
     * 
     * @return The unique ID assigned to this export.
     */
    public String exportId() {
        return exportId;
    }

    /**
     * <p>
     * Describes the status of the export. Can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * START_IN_PROGRESS - setting up resources to start continuous export.
     * </p>
     * </li>
     * <li>
     * <p>
     * START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export again.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE - data is being exported to the customer bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
     * start-continuous-export.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOP_IN_PROGRESS - stopping the export.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
     * </p>
     * </li>
     * <li>
     * <p>
     * INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContinuousExportStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the status of the export. Can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         START_IN_PROGRESS - setting up resources to start continuous export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export
     *         again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE - data is being exported to the customer bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
     *         start-continuous-export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOP_IN_PROGRESS - stopping the export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer
     *         bucket.
     *         </p>
     *         </li>
     * @see ContinuousExportStatus
     */
    public ContinuousExportStatus status() {
        return ContinuousExportStatus.fromValue(status);
    }

    /**
     * <p>
     * Describes the status of the export. Can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * START_IN_PROGRESS - setting up resources to start continuous export.
     * </p>
     * </li>
     * <li>
     * <p>
     * START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export again.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE - data is being exported to the customer bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
     * start-continuous-export.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOP_IN_PROGRESS - stopping the export.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
     * </p>
     * </li>
     * <li>
     * <p>
     * INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContinuousExportStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the status of the export. Can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         START_IN_PROGRESS - setting up resources to start continuous export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export
     *         again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE - data is being exported to the customer bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
     *         start-continuous-export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOP_IN_PROGRESS - stopping the export.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer
     *         bucket.
     *         </p>
     *         </li>
     * @see ContinuousExportStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Contains information about any errors that have occurred. This data type can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACCESS_DENIED - You don’t have permission to start Data Exploration in Amazon Athena. Contact your AWS
     * administrator for help. For more information, see <a
     * href="http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html">Setting Up AWS
     * Application Discovery Service</a> in the Application Discovery Service User Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELIVERY_STREAM_LIMIT_FAILURE - You reached the limit for Amazon Kinesis Data Firehose delivery streams. Reduce
     * the number of streams or request a limit increase and try again. For more information, see <a
     * href="http://docs.aws.amazon.com/streams/latest/dev/service-sizes-and-limits.html">Kinesis Data Streams
     * Limits</a> in the Amazon Kinesis Data Streams Developer Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * FIREHOSE_ROLE_MISSING - The Data Exploration feature is in an error state because your IAM User is missing the
     * AWSApplicationDiscoveryServiceFirehose role. Turn on Data Exploration in Amazon Athena and try again. For more
     * information, see <a
     * href="http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html#setting-up-user-policy"
     * >Step 3: Provide Application Discovery Service Access to Non-Administrator Users by Attaching Policies</a> in the
     * Application Discovery Service User Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * FIREHOSE_STREAM_DOES_NOT_EXIST - The Data Exploration feature is in an error state because your IAM User is
     * missing one or more of the Kinesis data delivery streams.
     * </p>
     * </li>
     * <li>
     * <p>
     * INTERNAL_FAILURE - The Data Exploration feature is in an error state because of an internal failure. Try again
     * later. If this problem persists, contact AWS Support.
     * </p>
     * </li>
     * <li>
     * <p>
     * S3_BUCKET_LIMIT_FAILURE - You reached the limit for Amazon S3 buckets. Reduce the number of Amazon S3 buckets or
     * request a limit increase and try again. For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
     * Limitations</a> in the Amazon Simple Storage Service Developer Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * S3_NOT_SIGNED_UP - Your account is not signed up for the Amazon S3 service. You must sign up before you can use
     * Amazon S3. You can sign up at the following URL: <a
     * href="https://aws.amazon.com/s3">https://aws.amazon.com/s3</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Contains information about any errors that have occurred. This data type can have the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACCESS_DENIED - You don’t have permission to start Data Exploration in Amazon Athena. Contact your AWS
     *         administrator for help. For more information, see <a
     *         href="http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html">Setting Up AWS
     *         Application Discovery Service</a> in the Application Discovery Service User Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELIVERY_STREAM_LIMIT_FAILURE - You reached the limit for Amazon Kinesis Data Firehose delivery streams.
     *         Reduce the number of streams or request a limit increase and try again. For more information, see <a
     *         href="http://docs.aws.amazon.com/streams/latest/dev/service-sizes-and-limits.html">Kinesis Data Streams
     *         Limits</a> in the Amazon Kinesis Data Streams Developer Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FIREHOSE_ROLE_MISSING - The Data Exploration feature is in an error state because your IAM User is
     *         missing the AWSApplicationDiscoveryServiceFirehose role. Turn on Data Exploration in Amazon Athena and
     *         try again. For more information, see <a href=
     *         "http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html#setting-up-user-policy"
     *         >Step 3: Provide Application Discovery Service Access to Non-Administrator Users by Attaching
     *         Policies</a> in the Application Discovery Service User Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FIREHOSE_STREAM_DOES_NOT_EXIST - The Data Exploration feature is in an error state because your IAM User
     *         is missing one or more of the Kinesis data delivery streams.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INTERNAL_FAILURE - The Data Exploration feature is in an error state because of an internal failure. Try
     *         again later. If this problem persists, contact AWS Support.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         S3_BUCKET_LIMIT_FAILURE - You reached the limit for Amazon S3 buckets. Reduce the number of Amazon S3
     *         buckets or request a limit increase and try again. For more information, see <a
     *         href="http://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
     *         Limitations</a> in the Amazon Simple Storage Service Developer Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         S3_NOT_SIGNED_UP - Your account is not signed up for the Amazon S3 service. You must sign up before you
     *         can use Amazon S3. You can sign up at the following URL: <a
     *         href="https://aws.amazon.com/s3">https://aws.amazon.com/s3</a>.
     *         </p>
     *         </li>
     */
    public String statusDetail() {
        return statusDetail;
    }

    /**
     * <p>
     * The name of the s3 bucket where the export data parquet files are stored.
     * </p>
     * 
     * @return The name of the s3 bucket where the export data parquet files are stored.
     */
    public String s3Bucket() {
        return s3Bucket;
    }

    /**
     * <p>
     * The timestamp representing when the continuous export was started.
     * </p>
     * 
     * @return The timestamp representing when the continuous export was started.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The timestamp that represents when this continuous export was stopped.
     * </p>
     * 
     * @return The timestamp that represents when this continuous export was stopped.
     */
    public Instant stopTime() {
        return stopTime;
    }

    /**
     * <p>
     * The type of data collector used to gather this data (currently only offered for AGENT).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataSource} will
     * return {@link DataSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dataSourceAsString}.
     * </p>
     * 
     * @return The type of data collector used to gather this data (currently only offered for AGENT).
     * @see DataSource
     */
    public DataSource dataSource() {
        return DataSource.fromValue(dataSource);
    }

    /**
     * <p>
     * The type of data collector used to gather this data (currently only offered for AGENT).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataSource} will
     * return {@link DataSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dataSourceAsString}.
     * </p>
     * 
     * @return The type of data collector used to gather this data (currently only offered for AGENT).
     * @see DataSource
     */
    public String dataSourceAsString() {
        return dataSource;
    }

    /**
     * Returns true if the SchemaStorageConfig property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSchemaStorageConfig() {
        return schemaStorageConfig != null && !(schemaStorageConfig instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An object which describes how the data is stored.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>databaseName</code> - the name of the Glue database used to store the schema.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSchemaStorageConfig()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An object which describes how the data is stored.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>databaseName</code> - the name of the Glue database used to store the schema.
     *         </p>
     *         </li>
     */
    public Map<String, String> schemaStorageConfig() {
        return schemaStorageConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(exportId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetail());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(stopTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaStorageConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContinuousExportDescription)) {
            return false;
        }
        ContinuousExportDescription other = (ContinuousExportDescription) obj;
        return Objects.equals(exportId(), other.exportId()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetail(), other.statusDetail()) && Objects.equals(s3Bucket(), other.s3Bucket())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(stopTime(), other.stopTime())
                && Objects.equals(dataSourceAsString(), other.dataSourceAsString())
                && Objects.equals(schemaStorageConfig(), other.schemaStorageConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContinuousExportDescription").add("ExportId", exportId()).add("Status", statusAsString())
                .add("StatusDetail", statusDetail()).add("S3Bucket", s3Bucket()).add("StartTime", startTime())
                .add("StopTime", stopTime()).add("DataSource", dataSourceAsString())
                .add("SchemaStorageConfig", schemaStorageConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "exportId":
            return Optional.ofNullable(clazz.cast(exportId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusDetail":
            return Optional.ofNullable(clazz.cast(statusDetail()));
        case "s3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "stopTime":
            return Optional.ofNullable(clazz.cast(stopTime()));
        case "dataSource":
            return Optional.ofNullable(clazz.cast(dataSourceAsString()));
        case "schemaStorageConfig":
            return Optional.ofNullable(clazz.cast(schemaStorageConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContinuousExportDescription, T> g) {
        return obj -> g.apply((ContinuousExportDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContinuousExportDescription> {
        /**
         * <p>
         * The unique ID assigned to this export.
         * </p>
         * 
         * @param exportId
         *        The unique ID assigned to this export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportId(String exportId);

        /**
         * <p>
         * Describes the status of the export. Can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * START_IN_PROGRESS - setting up resources to start continuous export.
         * </p>
         * </li>
         * <li>
         * <p>
         * START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export
         * again.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE - data is being exported to the customer bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
         * start-continuous-export.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOP_IN_PROGRESS - stopping the export.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
         * </p>
         * </li>
         * <li>
         * <p>
         * INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes the status of the export. Can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        START_IN_PROGRESS - setting up resources to start continuous export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        START_FAILED - an error occurred setting up continuous export. To recover, call
         *        start-continuous-export again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE - data is being exported to the customer bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
         *        start-continuous-export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOP_IN_PROGRESS - stopping the export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer
         *        bucket.
         *        </p>
         *        </li>
         * @see ContinuousExportStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContinuousExportStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Describes the status of the export. Can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * START_IN_PROGRESS - setting up resources to start continuous export.
         * </p>
         * </li>
         * <li>
         * <p>
         * START_FAILED - an error occurred setting up continuous export. To recover, call start-continuous-export
         * again.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE - data is being exported to the customer bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
         * start-continuous-export.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOP_IN_PROGRESS - stopping the export.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
         * </p>
         * </li>
         * <li>
         * <p>
         * INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes the status of the export. Can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        START_IN_PROGRESS - setting up resources to start continuous export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        START_FAILED - an error occurred setting up continuous export. To recover, call
         *        start-continuous-export again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE - data is being exported to the customer bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR - an error occurred during export. To fix the issue, call stop-continuous-export and
         *        start-continuous-export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOP_IN_PROGRESS - stopping the export.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOP_FAILED - an error occurred stopping the export. To recover, call stop-continuous-export again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INACTIVE - the continuous export has been stopped. Data is no longer being exported to the customer
         *        bucket.
         *        </p>
         *        </li>
         * @see ContinuousExportStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContinuousExportStatus
         */
        Builder status(ContinuousExportStatus status);

        /**
         * <p>
         * Contains information about any errors that have occurred. This data type can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACCESS_DENIED - You don’t have permission to start Data Exploration in Amazon Athena. Contact your AWS
         * administrator for help. For more information, see <a
         * href="http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html">Setting Up AWS
         * Application Discovery Service</a> in the Application Discovery Service User Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELIVERY_STREAM_LIMIT_FAILURE - You reached the limit for Amazon Kinesis Data Firehose delivery streams.
         * Reduce the number of streams or request a limit increase and try again. For more information, see <a
         * href="http://docs.aws.amazon.com/streams/latest/dev/service-sizes-and-limits.html">Kinesis Data Streams
         * Limits</a> in the Amazon Kinesis Data Streams Developer Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * FIREHOSE_ROLE_MISSING - The Data Exploration feature is in an error state because your IAM User is missing
         * the AWSApplicationDiscoveryServiceFirehose role. Turn on Data Exploration in Amazon Athena and try again. For
         * more information, see <a href=
         * "http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html#setting-up-user-policy"
         * >Step 3: Provide Application Discovery Service Access to Non-Administrator Users by Attaching Policies</a> in
         * the Application Discovery Service User Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * FIREHOSE_STREAM_DOES_NOT_EXIST - The Data Exploration feature is in an error state because your IAM User is
         * missing one or more of the Kinesis data delivery streams.
         * </p>
         * </li>
         * <li>
         * <p>
         * INTERNAL_FAILURE - The Data Exploration feature is in an error state because of an internal failure. Try
         * again later. If this problem persists, contact AWS Support.
         * </p>
         * </li>
         * <li>
         * <p>
         * S3_BUCKET_LIMIT_FAILURE - You reached the limit for Amazon S3 buckets. Reduce the number of Amazon S3 buckets
         * or request a limit increase and try again. For more information, see <a
         * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
         * Limitations</a> in the Amazon Simple Storage Service Developer Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * S3_NOT_SIGNED_UP - Your account is not signed up for the Amazon S3 service. You must sign up before you can
         * use Amazon S3. You can sign up at the following URL: <a
         * href="https://aws.amazon.com/s3">https://aws.amazon.com/s3</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetail
         *        Contains information about any errors that have occurred. This data type can have the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACCESS_DENIED - You don’t have permission to start Data Exploration in Amazon Athena. Contact your AWS
         *        administrator for help. For more information, see <a
         *        href="http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html">Setting Up
         *        AWS Application Discovery Service</a> in the Application Discovery Service User Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELIVERY_STREAM_LIMIT_FAILURE - You reached the limit for Amazon Kinesis Data Firehose delivery
         *        streams. Reduce the number of streams or request a limit increase and try again. For more information,
         *        see <a href="http://docs.aws.amazon.com/streams/latest/dev/service-sizes-and-limits.html">Kinesis Data
         *        Streams Limits</a> in the Amazon Kinesis Data Streams Developer Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FIREHOSE_ROLE_MISSING - The Data Exploration feature is in an error state because your IAM User is
         *        missing the AWSApplicationDiscoveryServiceFirehose role. Turn on Data Exploration in Amazon Athena and
         *        try again. For more information, see <a href=
         *        "http://docs.aws.amazon.com/application-discovery/latest/userguide/setting-up.html#setting-up-user-policy"
         *        >Step 3: Provide Application Discovery Service Access to Non-Administrator Users by Attaching
         *        Policies</a> in the Application Discovery Service User Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FIREHOSE_STREAM_DOES_NOT_EXIST - The Data Exploration feature is in an error state because your IAM
         *        User is missing one or more of the Kinesis data delivery streams.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INTERNAL_FAILURE - The Data Exploration feature is in an error state because of an internal failure.
         *        Try again later. If this problem persists, contact AWS Support.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        S3_BUCKET_LIMIT_FAILURE - You reached the limit for Amazon S3 buckets. Reduce the number of Amazon S3
         *        buckets or request a limit increase and try again. For more information, see <a
         *        href="http://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html">Bucket Restrictions and
         *        Limitations</a> in the Amazon Simple Storage Service Developer Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        S3_NOT_SIGNED_UP - Your account is not signed up for the Amazon S3 service. You must sign up before
         *        you can use Amazon S3. You can sign up at the following URL: <a
         *        href="https://aws.amazon.com/s3">https://aws.amazon.com/s3</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetail(String statusDetail);

        /**
         * <p>
         * The name of the s3 bucket where the export data parquet files are stored.
         * </p>
         * 
         * @param s3Bucket
         *        The name of the s3 bucket where the export data parquet files are stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The timestamp representing when the continuous export was started.
         * </p>
         * 
         * @param startTime
         *        The timestamp representing when the continuous export was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The timestamp that represents when this continuous export was stopped.
         * </p>
         * 
         * @param stopTime
         *        The timestamp that represents when this continuous export was stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopTime(Instant stopTime);

        /**
         * <p>
         * The type of data collector used to gather this data (currently only offered for AGENT).
         * </p>
         * 
         * @param dataSource
         *        The type of data collector used to gather this data (currently only offered for AGENT).
         * @see DataSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSource
         */
        Builder dataSource(String dataSource);

        /**
         * <p>
         * The type of data collector used to gather this data (currently only offered for AGENT).
         * </p>
         * 
         * @param dataSource
         *        The type of data collector used to gather this data (currently only offered for AGENT).
         * @see DataSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSource
         */
        Builder dataSource(DataSource dataSource);

        /**
         * <p>
         * An object which describes how the data is stored.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>databaseName</code> - the name of the Glue database used to store the schema.
         * </p>
         * </li>
         * </ul>
         * 
         * @param schemaStorageConfig
         *        An object which describes how the data is stored.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>databaseName</code> - the name of the Glue database used to store the schema.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaStorageConfig(Map<String, String> schemaStorageConfig);
    }

    static final class BuilderImpl implements Builder {
        private String exportId;

        private String status;

        private String statusDetail;

        private String s3Bucket;

        private Instant startTime;

        private Instant stopTime;

        private String dataSource;

        private Map<String, String> schemaStorageConfig = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContinuousExportDescription model) {
            exportId(model.exportId);
            status(model.status);
            statusDetail(model.statusDetail);
            s3Bucket(model.s3Bucket);
            startTime(model.startTime);
            stopTime(model.stopTime);
            dataSource(model.dataSource);
            schemaStorageConfig(model.schemaStorageConfig);
        }

        public final String getExportId() {
            return exportId;
        }

        @Override
        public final Builder exportId(String exportId) {
            this.exportId = exportId;
            return this;
        }

        public final void setExportId(String exportId) {
            this.exportId = exportId;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ContinuousExportStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetail() {
            return statusDetail;
        }

        @Override
        public final Builder statusDetail(String statusDetail) {
            this.statusDetail = statusDetail;
            return this;
        }

        public final void setStatusDetail(String statusDetail) {
            this.statusDetail = statusDetail;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getStopTime() {
            return stopTime;
        }

        @Override
        public final Builder stopTime(Instant stopTime) {
            this.stopTime = stopTime;
            return this;
        }

        public final void setStopTime(Instant stopTime) {
            this.stopTime = stopTime;
        }

        public final String getDataSourceAsString() {
            return dataSource;
        }

        @Override
        public final Builder dataSource(String dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        @Override
        public final Builder dataSource(DataSource dataSource) {
            this.dataSource(dataSource == null ? null : dataSource.toString());
            return this;
        }

        public final void setDataSource(String dataSource) {
            this.dataSource = dataSource;
        }

        public final Map<String, String> getSchemaStorageConfig() {
            return schemaStorageConfig;
        }

        @Override
        public final Builder schemaStorageConfig(Map<String, String> schemaStorageConfig) {
            this.schemaStorageConfig = SchemaStorageConfigCopier.copy(schemaStorageConfig);
            return this;
        }

        public final void setSchemaStorageConfig(Map<String, String> schemaStorageConfig) {
            this.schemaStorageConfig = SchemaStorageConfigCopier.copy(schemaStorageConfig);
        }

        @Override
        public ContinuousExportDescription build() {
            return new ContinuousExportDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
