/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Permissions designate which private CA actions can be performed by an AWS service or entity. In order for ACM to
 * automatically renew private certificates, you must give the ACM service principal all available permissions (
 * <code>IssueCertificate</code>, <code>GetCertificate</code>, and <code>ListPermissions</code>). Permissions can be
 * assigned with the <a>CreatePermission</a> action, removed with the <a>DeletePermission</a> action, and listed with
 * the <a>ListPermissions</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Permission implements SdkPojo, Serializable, ToCopyableBuilder<Permission.Builder, Permission> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Permission::certificateAuthorityArn)).setter(setter(Builder::certificateAuthorityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateAuthorityArn").build())
            .build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Permission::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<String> PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Permission::principal)).setter(setter(Builder::principal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Principal").build()).build();

    private static final SdkField<String> SOURCE_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Permission::sourceAccount)).setter(setter(Builder::sourceAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceAccount").build()).build();

    private static final SdkField<List<String>> ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Permission::actionsAsStrings))
            .setter(setter(Builder::actionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Permission::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(CERTIFICATE_AUTHORITY_ARN_FIELD, CREATED_AT_FIELD, PRINCIPAL_FIELD, SOURCE_ACCOUNT_FIELD, ACTIONS_FIELD,
                    POLICY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityArn;

    private final Instant createdAt;

    private final String principal;

    private final String sourceAccount;

    private final List<String> actions;

    private final String policy;

    private Permission(BuilderImpl builder) {
        this.certificateAuthorityArn = builder.certificateAuthorityArn;
        this.createdAt = builder.createdAt;
        this.principal = builder.principal;
        this.sourceAccount = builder.sourceAccount;
        this.actions = builder.actions;
        this.policy = builder.policy;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
     */
    public String certificateAuthorityArn() {
        return certificateAuthorityArn;
    }

    /**
     * <p>
     * The time at which the permission was created.
     * </p>
     * 
     * @return The time at which the permission was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The AWS service or entity that holds the permission. At this time, the only valid principal is
     * <code>acm.amazonaws.com</code>.
     * </p>
     * 
     * @return The AWS service or entity that holds the permission. At this time, the only valid principal is
     *         <code>acm.amazonaws.com</code>.
     */
    public String principal() {
        return principal;
    }

    /**
     * <p>
     * The ID of the account that assigned the permission.
     * </p>
     * 
     * @return The ID of the account that assigned the permission.
     */
    public String sourceAccount() {
        return sourceAccount;
    }

    /**
     * <p>
     * The private CA actions that can be performed by the designated AWS service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The private CA actions that can be performed by the designated AWS service.
     */
    public List<ActionType> actions() {
        return ActionListCopier.copyStringToEnum(actions);
    }

    /**
     * Returns true if the Actions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The private CA actions that can be performed by the designated AWS service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The private CA actions that can be performed by the designated AWS service.
     */
    public List<String> actionsAsStrings() {
        return actions;
    }

    /**
     * <p>
     * The name of the policy that is associated with the permission.
     * </p>
     * 
     * @return The name of the policy that is associated with the permission.
     */
    public String policy() {
        return policy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(principal());
        hashCode = 31 * hashCode + Objects.hashCode(sourceAccount());
        hashCode = 31 * hashCode + Objects.hashCode(actionsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Permission)) {
            return false;
        }
        Permission other = (Permission) obj;
        return Objects.equals(certificateAuthorityArn(), other.certificateAuthorityArn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(principal(), other.principal())
                && Objects.equals(sourceAccount(), other.sourceAccount())
                && Objects.equals(actionsAsStrings(), other.actionsAsStrings()) && Objects.equals(policy(), other.policy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Permission").add("CertificateAuthorityArn", certificateAuthorityArn())
                .add("CreatedAt", createdAt()).add("Principal", principal()).add("SourceAccount", sourceAccount())
                .add("Actions", actionsAsStrings()).add("Policy", policy()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityArn":
            return Optional.ofNullable(clazz.cast(certificateAuthorityArn()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "Principal":
            return Optional.ofNullable(clazz.cast(principal()));
        case "SourceAccount":
            return Optional.ofNullable(clazz.cast(sourceAccount()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actionsAsStrings()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Permission, T> g) {
        return obj -> g.apply((Permission) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Permission> {
        /**
         * <p>
         * The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
         * </p>
         * 
         * @param certificateAuthorityArn
         *        The Amazon Resource Number (ARN) of the private CA from which the permission was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityArn(String certificateAuthorityArn);

        /**
         * <p>
         * The time at which the permission was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the permission was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The AWS service or entity that holds the permission. At this time, the only valid principal is
         * <code>acm.amazonaws.com</code>.
         * </p>
         * 
         * @param principal
         *        The AWS service or entity that holds the permission. At this time, the only valid principal is
         *        <code>acm.amazonaws.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principal(String principal);

        /**
         * <p>
         * The ID of the account that assigned the permission.
         * </p>
         * 
         * @param sourceAccount
         *        The ID of the account that assigned the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceAccount(String sourceAccount);

        /**
         * <p>
         * The private CA actions that can be performed by the designated AWS service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated AWS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(Collection<String> actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated AWS service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated AWS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(String... actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated AWS service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated AWS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ActionType> actions);

        /**
         * <p>
         * The private CA actions that can be performed by the designated AWS service.
         * </p>
         * 
         * @param actions
         *        The private CA actions that can be performed by the designated AWS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ActionType... actions);

        /**
         * <p>
         * The name of the policy that is associated with the permission.
         * </p>
         * 
         * @param policy
         *        The name of the policy that is associated with the permission.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityArn;

        private Instant createdAt;

        private String principal;

        private String sourceAccount;

        private List<String> actions = DefaultSdkAutoConstructList.getInstance();

        private String policy;

        private BuilderImpl() {
        }

        private BuilderImpl(Permission model) {
            certificateAuthorityArn(model.certificateAuthorityArn);
            createdAt(model.createdAt);
            principal(model.principal);
            sourceAccount(model.sourceAccount);
            actionsWithStrings(model.actions);
            policy(model.policy);
        }

        public final String getCertificateAuthorityArn() {
            return certificateAuthorityArn;
        }

        @Override
        public final Builder certificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
            return this;
        }

        public final void setCertificateAuthorityArn(String certificateAuthorityArn) {
            this.certificateAuthorityArn = certificateAuthorityArn;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final String getPrincipal() {
            return principal;
        }

        @Override
        public final Builder principal(String principal) {
            this.principal = principal;
            return this;
        }

        public final void setPrincipal(String principal) {
            this.principal = principal;
        }

        public final String getSourceAccount() {
            return sourceAccount;
        }

        @Override
        public final Builder sourceAccount(String sourceAccount) {
            this.sourceAccount = sourceAccount;
            return this;
        }

        public final void setSourceAccount(String sourceAccount) {
            this.sourceAccount = sourceAccount;
        }

        public final Collection<String> getActionsAsStrings() {
            return actions;
        }

        @Override
        public final Builder actionsWithStrings(Collection<String> actions) {
            this.actions = ActionListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actionsWithStrings(String... actions) {
            actionsWithStrings(Arrays.asList(actions));
            return this;
        }

        @Override
        public final Builder actions(Collection<ActionType> actions) {
            this.actions = ActionListCopier.copyEnumToString(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ActionType... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        public final void setActionsWithStrings(Collection<String> actions) {
            this.actions = ActionListCopier.copy(actions);
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        @Override
        public Permission build() {
            return new Permission(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
