/**
 * <h1>Amazon Redshift Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always stable and safe to use.
 * <p>
 * </blockquote>
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development.
 * They are subject to non-backward compatible changes or removal in any future version. These are
 * not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be
 * announced in the release notes. This means that while you may use them, you may need to update
 * your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <h2>Starting a Redshift Cluster Database</h2>
 * <p>
 * To set up a Redshift cluster, define a <code>Cluster</code>. It will be launched in a VPC.
 * You can specify a VPC, otherwise one will be created. The nodes are always launched in private subnets and are encrypted by default.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "Vpc");
 * Cluster cluster = Cluster.Builder.create(this, "Redshift")
 *         .masterUser(Login.builder()
 *                 .masterUsername("admin")
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager.
 * <p>
 * A default database named <code>default_db</code> will be created in the cluster. To change the name of this database set the <code>defaultDatabaseName</code> attribute in the constructor properties.
 * <p>
 * By default, the cluster will not be publicly accessible.
 * Depending on your use case, you can make the cluster publicly accessible with the <code>publiclyAccessible</code> property.
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster, use the <code>.connections</code> attribute. Redshift Clusters have
 * a default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * cluster.connections.allowDefaultPortFromAnyIpv4("Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoint to access your database cluster will be available as the <code>.clusterEndpoint</code> attribute:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * cluster.addRotationSingleUser();
 * </pre></blockquote>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.secretsmanager.*;
 * 
 * 
 * cluster.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder()
 *         .secret(Secret.fromSecretNameV2(this, "Imported Secret", "my-secret"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Database Resources</h2>
 * <p>
 * This module allows for the creation of non-CloudFormation database resources such as users
 * and tables. This allows you to manage identities, permissions, and stateful resources
 * within your Redshift cluster from your CDK application.
 * <p>
 * Because these resources are not available in CloudFormation, this library leverages
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html">custom
 * resources</a>
 * to manage them. In addition to the IAM permissions required to make Redshift service
 * calls, the execution role for the custom resource handler requires database credentials to
 * create resources within the cluster.
 * <p>
 * These database credentials can be supplied explicitly through the <code>adminUser</code> properties
 * of the various database resource constructs. Alternatively, the credentials can be
 * automatically pulled from the Redshift cluster's default administrator
 * credentials. However, this option is only available if the password for the credentials
 * was generated by the CDK application (ie., no value vas provided for <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-redshift.Login.html#masterpasswordspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan">the <code>masterPassword</code>
 * property</a>
 * of
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-redshift.Cluster.html#masteruserspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan"><code>Cluster.masterUser</code></a>).
 * <p>
 * <h3>Creating Users</h3>
 * <p>
 * Create a user within a Redshift cluster database by instantiating a <code>User</code> construct. This
 * will generate a username and password, store the credentials in a <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-secretsmanager.Secret.html">AWS Secrets Manager
 * <code>Secret</code></a>,
 * and make a query to the Redshift cluster to create a new database user with the
 * credentials.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * User.Builder.create(this, "User")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the user credentials are encrypted with your AWS account's default Secrets
 * Manager encryption key. You can specify the encryption key used for this purpose by
 * supplying a key in the <code>encryptionKey</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Key encryptionKey = new Key(this, "Key");
 * User.Builder.create(this, "User")
 *         .encryptionKey(encryptionKey)
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, a username is automatically generated from the user construct ID and its path
 * in the construct tree. You can specify a particular username by providing a value for the
 * <code>username</code> property. Usernames must be valid identifiers; see: <a href="https://docs.aws.amazon.com/redshift/latest/dg/r_names.html">Names and
 * identifiers</a> in the <em>Amazon
 * Redshift Database Developer Guide</em>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * User.Builder.create(this, "User")
 *         .username("myuser")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The user password is generated by AWS Secrets Manager using the default configuration
 * found in
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-secretsmanager.SecretStringGenerator.html"><code>secretsmanager.SecretStringGenerator</code></a>,
 * except with password length <code>30</code> and some SQL-incompliant characters excluded. The
 * plaintext for the password will never be present in the CDK application; instead, a
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">CloudFormation Dynamic
 * Reference</a>
 * will be used wherever the password value is required.
 * <p>
 * <h3>Creating Tables</h3>
 * <p>
 * Create a table within a Redshift cluster database by instantiating a <code>Table</code>
 * construct. This will make a query to the Redshift cluster to create a new database table
 * with the supplied schema.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The table can be configured to have distStyle attribute and a distKey column:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").distKey(true).build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .distStyle(TableDistStyle.KEY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The table can also be configured to have sortStyle attribute and sortKey columns:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").sortKey(true).build(), Column.builder().name("col2").dataType("float").sortKey(true).build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .sortStyle(TableSortStyle.COMPOUND)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Granting Privileges</h3>
 * <p>
 * You can give a user privileges to perform certain actions on a table by using the
 * <code>Table.grant()</code> method.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * User user = User.Builder.create(this, "User")
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * Table table = Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build();
 * 
 * table.grant(user, TableAction.DROP, TableAction.SELECT);
 * </pre></blockquote>
 * <p>
 * Take care when managing privileges via the CDK, as attempting to manage a user's
 * privileges on the same table in multiple CDK applications could lead to accidentally
 * overriding these permissions. Consider the following two CDK applications which both refer
 * to the same user and table. In application 1, the resources are created and the user is
 * given <code>INSERT</code> permissions on the table:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String databaseName = "databaseName";
 * String username = "myuser";
 * String tableName = "mytable";
 * 
 * User user = User.Builder.create(this, "User")
 *         .username(username)
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build();
 * Table table = Table.Builder.create(this, "Table")
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build();
 * table.grant(user, TableAction.INSERT);
 * </pre></blockquote>
 * <p>
 * In application 2, the resources are imported and the user is given <code>INSERT</code> permissions on
 * the table:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String databaseName = "databaseName";
 * String username = "myuser";
 * String tableName = "mytable";
 * 
 * IUser user = User.fromUserAttributes(this, "User", UserAttributes.builder()
 *         .username(username)
 *         .password(SecretValue.plainText("NOT_FOR_PRODUCTION"))
 *         .cluster(cluster)
 *         .databaseName(databaseName)
 *         .build());
 * ITable table = Table.fromTableAttributes(this, "Table", TableAttributes.builder()
 *         .tableName(tableName)
 *         .tableColumns(List.of(Column.builder().name("col1").dataType("varchar(4)").build(), Column.builder().name("col2").dataType("float").build()))
 *         .cluster(cluster)
 *         .databaseName("databaseName")
 *         .build());
 * table.grant(user, TableAction.INSERT);
 * </pre></blockquote>
 * <p>
 * Both applications attempt to grant the user the appropriate privilege on the table by
 * submitting a <code>GRANT USER</code> SQL query to the Redshift cluster. Note that the latter of these
 * two calls will have no effect since the user has already been granted the privilege.
 * <p>
 * Now, if application 1 were to remove the call to <code>grant</code>, a <code>REVOKE USER</code> SQL query is
 * submitted to the Redshift cluster. In general, application 1 does not know that
 * application 2 has also granted this permission and thus cannot decide not to issue the
 * revocation. This leads to the undesirable state where application 2 still contains the
 * call to <code>grant</code> but the user does not have the specified permission.
 * <p>
 * Note that this does not occur when duplicate privileges are granted within the same
 * application, as such privileges are de-duplicated before any SQL query is submitted.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.redshift;
