/**
 * <h2>Amazon CloudWatch Events Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Amazon CloudWatch Events delivers a near real-time stream of system events that
 * describe changes in AWS resources. For example, an AWS CodePipeline emits the
 * <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#codepipeline_event_type">State
 * Change</a>
 * event when the pipeline changes it's state.</p>
 * <ul>
 * <li><strong>Events</strong>: An event indicates a change in your AWS environment. AWS resources
 * can generate events when their state changes. For example, Amazon EC2
 * generates an event when the state of an EC2 instance changes from pending to
 * running, and Amazon EC2 Auto Scaling generates events when it launches or
 * terminates instances. AWS CloudTrail publishes events when you make API calls.
 * You can generate custom application-level events and publish them to
 * CloudWatch Events. You can also set up scheduled events that are generated on
 * a periodic basis. For a list of services that generate events, and sample
 * events from each service, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html">CloudWatch Events Event Examples From Each
 * Supported
 * Service</a>.</li>
 * <li><strong>Targets</strong>: A target processes events. Targets can include Amazon EC2
 * instances, AWS Lambda functions, Kinesis streams, Amazon ECS tasks, Step
 * Functions state machines, Amazon SNS topics, Amazon SQS queues, and built-in
 * targets. A target receives events in JSON format.</li>
 * <li><strong>Rules</strong>: A rule matches incoming events and routes them to targets for
 * processing. A single rule can route to multiple targets, all of which are
 * processed in parallel. Rules are not processed in a particular order. This
 * enables different parts of an organization to look for and process the events
 * that are of interest to them. A rule can customize the JSON sent to the
 * target, by passing only certain parts or by overwriting it with a constant.</li>
 * <li><strong>EventBuses</strong>: An event bus can receive events from your own custom applications
 * or it can receive events from applications and services created by AWS SaaS partners.
 * See <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/create-event-bus.html">Creating an Event Bus</a>.</li>
 * </ul>
 * <p>The <code>Rule</code> construct defines a CloudWatch events rule which monitors an
 * event based on an <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/CloudWatchEventsandEventPatterns.html">event
 * pattern</a>
 * and invoke <strong>event targets</strong> when the pattern is matched against a triggered
 * event. Event targets are objects that implement the <code>IRuleTarget</code> interface.</p>
 * <p>Normally, you will use one of the <code>source.onXxx(name[, target[, options]]) -&gt; Rule</code> methods on the event source to define an event rule associated with
 * the specific activity. You can targets either via props, or add targets using
 * <code>rule.addTarget</code>.</p>
 * <p>For example, to define an rule that triggers a CodeBuild project build when a
 * commit is pushed to the &quot;master&quot; branch of a CodeCommit repository:</p>
 * <pre><code class="language-ts">// This example is in TypeScript, examples in Java are coming soon.
 * const onCommitRule = repo.onCommit('OnCommit', {
 *   target: new targets.CodeBuildProject(project),
 *   branches: ['master']
 * });
 * </code></pre>
 * <p>You can add additional targets, with optional <a href="https://docs.aws.amazon.com/AmazonCloudWatchEvents/latest/APIReference/API_InputTransformer.html">input
 * transformer</a>
 * using <code>eventRule.addTarget(target[, input])</code>. For example, we can add a SNS
 * topic target which formats a human-readable message for the commit.</p>
 * <p>For example, this adds an SNS topic as a target:</p>
 * <pre><code class="language-ts">onCommitRule.addTarget(new targets.SnsTopic(topic, {
 *   message: events.RuleTargetInput.fromText(
 *     `A commit was pushed to the repository ${codecommit.ReferenceEvent.repositoryName} on branch ${codecommit.ReferenceEvent.referenceName}`
 *   )
 * }));
 * </code></pre>
 * <h2>Event Targets</h2>
 * <p>The <code>@aws-cdk/aws-events-targets</code> module includes classes that implement the <code>IRuleTarget</code>
 * interface for various AWS services.</p>
 * <p>The following targets are supported:</p>
 * <ul>
 * <li><code>targets.CodeBuildProject</code>: Start an AWS CodeBuild build</li>
 * <li><code>targets.CodePipeline</code>: Start an AWS CodePipeline pipeline execution</li>
 * <li><code>targets.EcsTask</code>: Start a task on an Amazon ECS cluster</li>
 * <li><code>targets.LambdaFunction</code>: Invoke an AWS Lambda function</li>
 * <li><code>targets.SnsTopic</code>: Publish into an SNS topic</li>
 * <li><code>targets.SqsQueue</code>: Send a message to an Amazon SQS Queue</li>
 * <li><code>targets.SfnStateMachine</code>: Trigger an AWS Step Functions state machine</li>
 * <li><code>targets.AwsApi</code>: Make an AWS API call</li>
 * </ul>
 * <h3>Cross-account targets</h3>
 * <p>It's possible to have the source of the event and a target in separate AWS accounts:</p>
 * <pre><code class="language-typescript">import { App, Stack } from '@aws-cdk/core';
 * import codebuild = require('@aws-cdk/aws-codebuild');
 * import codecommit = require('@aws-cdk/aws-codecommit');
 * import targets = require('@aws-cdk/aws-events-targets');
 * 
 * const app = new App();
 * 
 * const stack1 = new Stack(app, 'Stack1', { env: { account: account1, region: 'us-east-1' } });
 * const repo = new codecommit.Repository(stack1, 'Repository', {
 *   // ...
 * });
 * 
 * const stack2 = new Stack(app, 'Stack2', { env: { account: account2, region: 'us-east-1' } });
 * const project = new codebuild.Project(stack2, 'Project', {
 *   // ...
 * });
 * 
 * repo.onCommit('OnCommit', {
 *   target: new targets.CodeBuildProject(project),
 * });
 * </code></pre>
 * <p>In this situation, the CDK will wire the 2 accounts together:</p>
 * <ul>
 * <li>It will generate a rule in the source stack with the event bus of the target account as the target</li>
 * <li>It will generate a rule in the target stack, with the provided target</li>
 * <li>It will generate a separate stack that gives the source account permissions to publish events
 * to the event bus of the target account in the given region,
 * and make sure its deployed before the source stack</li>
 * </ul>
 * <p><strong>Note</strong>: while events can span multiple accounts, they <em>cannot</em> span different regions
 * (that is a CloudWatch, not CDK, limitation).</p>
 * <p>For more information, see the
 * <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/CloudWatchEvents-CrossAccountEventDelivery.html">AWS documentation on cross-account events</a>.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.events;
