/**
 * <h1>Amazon EventBridge Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * Amazon EventBridge delivers a near real-time stream of system events that
 * describe changes in AWS resources. For example, an AWS CodePipeline emits the
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/event-types.html#codepipeline-event-type">State
 * Change</a>
 * event when the pipeline changes its state.
 * <p>
 * <ul>
 * <li><strong>Events</strong>: An event indicates a change in your AWS environment. AWS resources
 * can generate events when their state changes. For example, Amazon EC2
 * generates an event when the state of an EC2 instance changes from pending to
 * running, and Amazon EC2 Auto Scaling generates events when it launches or
 * terminates instances. AWS CloudTrail publishes events when you make API calls.
 * You can generate custom application-level events and publish them to
 * EventBridge. You can also set up scheduled events that are generated on
 * a periodic basis. For a list of services that generate events, and sample
 * events from each service, see <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/event-types.html">EventBridge Event Examples From Each
 * Supported
 * Service</a>.</li>
 * <li><strong>Targets</strong>: A target processes events. Targets can include Amazon EC2
 * instances, AWS Lambda functions, Kinesis streams, Amazon ECS tasks, Step
 * Functions state machines, Amazon SNS topics, Amazon SQS queues, Amazon CloudWatch LogGroups, and built-in
 * targets. A target receives events in JSON format.</li>
 * <li><strong>Rules</strong>: A rule matches incoming events and routes them to targets for
 * processing. A single rule can route to multiple targets, all of which are
 * processed in parallel. Rules are not processed in a particular order. This
 * enables different parts of an organization to look for and process the events
 * that are of interest to them. A rule can customize the JSON sent to the
 * target, by passing only certain parts or by overwriting it with a constant.</li>
 * <li><strong>EventBuses</strong>: An event bus can receive events from your own custom applications
 * or it can receive events from applications and services created by AWS SaaS partners.
 * See <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/create-event-bus.html">Creating an Event Bus</a>.</li>
 * </ul>
 * <p>
 * <h2>Rule</h2>
 * <p>
 * The <code>Rule</code> construct defines an EventBridge rule which monitors an
 * event based on an <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/filtering-examples-structure.html">event
 * pattern</a>
 * and invoke <strong>event targets</strong> when the pattern is matched against a triggered
 * event. Event targets are objects that implement the <code>IRuleTarget</code> interface.
 * <p>
 * Normally, you will use one of the <code>source.onXxx(name[, target[, options]]) -&gt; Rule</code> methods on the event source to define an event rule associated with
 * the specific activity. You can targets either via props, or add targets using
 * <code>rule.addTarget</code>.
 * <p>
 * For example, to define an rule that triggers a CodeBuild project build when a
 * commit is pushed to the "master" branch of a CodeCommit repository:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object onCommitRule = repo.onCommit("OnCommit", Map.of(
 *         "target", new CodeBuildProject(project),
 *         "branches", asList("master")));
 * </pre></blockquote>
 * <p>
 * You can add additional targets, with optional <a href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_InputTransformer.html">input
 * transformer</a>
 * using <code>eventRule.addTarget(target[, input])</code>. For example, we can add a SNS
 * topic target which formats a human-readable message for the commit.
 * <p>
 * For example, this adds an SNS topic as a target:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * onCommitRule.addTarget(SnsTopic.Builder.create(topic)
 *         .message(events.RuleTargetInput.fromText(String.format("A commit was pushed to the repository %s on branch %s", codecommit.ReferenceEvent.getRepositoryName(), codecommit.ReferenceEvent.getReferenceName())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Scheduling</h2>
 * <p>
 * You can configure a Rule to run on a schedule (cron or rate).
 * Rate must be specified in minutes, hours or days.
 * <p>
 * The following example runs a task every day at 4am:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.events.Rule;
 * import software.amazon.awscdk.services.events.Schedule;
 * import software.amazon.awscdk.services.events.targets.EcsTask;
 * 
 * EcsTask ecsTaskTarget = new EcsTask(new EcsTaskProps().cluster(cluster).taskDefinition(taskDefinition).role(role));
 * 
 * new Rule(this, "ScheduleRule", new RuleProps()
 *         .schedule(Schedule.cron(new CronOptions().minute("0").hour("4")))
 *         .targets(asList(ecsTaskTarget)));
 * </pre></blockquote>
 * <p>
 * If you want to specify Fargate platform version, set <code>platformVersion</code> in EcsTask's props like the following example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object platformVersion = ecs.FargatePlatformVersion.getVERSION1_4();
 * Object ecsTaskTarget = EcsTask.Builder.create().cluster(cluster).taskDefinition(taskDefinition).role(role).platformVersion(platformVersion).build();
 * </pre></blockquote>
 * <p>
 * <h2>Event Targets</h2>
 * <p>
 * The <code>&#64;aws-cdk/aws-events-targets</code> module includes classes that implement the <code>IRuleTarget</code>
 * interface for various AWS services.
 * <p>
 * The following targets are supported:
 * <p>
 * <ul>
 * <li><code>targets.CodeBuildProject</code>: Start an AWS CodeBuild build</li>
 * <li><code>targets.CodePipeline</code>: Start an AWS CodePipeline pipeline execution</li>
 * <li><code>targets.EcsTask</code>: Start a task on an Amazon ECS cluster</li>
 * <li><code>targets.LambdaFunction</code>: Invoke an AWS Lambda function</li>
 * <li><code>targets.SnsTopic</code>: Publish into an SNS topic</li>
 * <li><code>targets.SqsQueue</code>: Send a message to an Amazon SQS Queue</li>
 * <li><code>targets.SfnStateMachine</code>: Trigger an AWS Step Functions state machine</li>
 * <li><code>targets.BatchJob</code>: Queue an AWS Batch Job</li>
 * <li><code>targets.AwsApi</code>: Make an AWS API call</li>
 * </ul>
 * <p>
 * <h3>Cross-account targets</h3>
 * <p>
 * It's possible to have the source of the event and a target in separate AWS accounts:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.core.App;
 * import software.amazon.awscdk.core.Stack;
 * import software.amazon.awscdk.services.codebuild.*;
 * import software.amazon.awscdk.services.codecommit.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * App app = new App();
 * 
 * Stack stack1 = new Stack(app, "Stack1", new StackProps().env(new Environment().account(account1).region("us-east-1")));
 * Repository repo = new Repository(stack1, "Repository", new RepositoryProps());
 * 
 * Stack stack2 = new Stack(app, "Stack2", new StackProps().env(new Environment().account(account2).region("us-east-1")));
 * Project project = new Project(stack2, "Project", new ProjectProps());
 * 
 * repo.onCommit("OnCommit", new OnCommitOptions()
 *         .target(new CodeBuildProject(project)));
 * </pre></blockquote>
 * <p>
 * In this situation, the CDK will wire the 2 accounts together:
 * <p>
 * <ul>
 * <li>It will generate a rule in the source stack with the event bus of the target account as the target</li>
 * <li>It will generate a rule in the target stack, with the provided target</li>
 * <li>It will generate a separate stack that gives the source account permissions to publish events
 * to the event bus of the target account in the given region,
 * and make sure its deployed before the source stack</li>
 * </ul>
 * <p>
 * <strong>Note</strong>: while events can span multiple accounts, they <em>cannot</em> span different regions
 * (that is an EventBridge, not CDK, limitation).
 * <p>
 * For more information, see the
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html">AWS documentation on cross-account events</a>.
 * <p>
 * <h2>Archiving</h2>
 * <p>
 * It is possible to archive all or some events sent to an event bus. It is then possible to <a href="https://aws.amazon.com/blogs/aws/new-archive-and-replay-events-with-amazon-eventbridge/">replay these events</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.core.*;
 * 
 * Object stack = new stack();
 * 
 * Object bus = EventBus.Builder.create(stack, "bus")
 *         .eventBusName("MyCustomEventBus")
 *         .build();
 * 
 * bus.archive("MyArchive", Map.of(
 *         "archiveName", "MyCustomEventBusArchive",
 *         "description", "MyCustomerEventBus Archive",
 *         "eventPattern", Map.of(
 *                 "account", asList(stack.getAccount())),
 *         "retention", cdk.Duration.days(365)));
 * </pre></blockquote>
 * <p>
 * <h2>Granting PutEvents to an existing EventBus</h2>
 * <p>
 * To import an existing EventBus into your CDK application, use <code>EventBus.fromEventBusArn</code> or <code>EventBus.fromEventBusAttributes</code>
 * factory method.
 * <p>
 * Then, you can use the <code>grantPutEventsTo</code> method to grant <code>event:PutEvents</code> to the eventBus.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object eventBus = EventBus.fromEventBusArn(this, "ImportedEventBus", "arn:aws:events:us-east-1:111111111:event-bus/my-event-bus");
 * 
 * // now you can just call methods on the eventbus
 * eventBus.grantPutEventsTo(lambdaFunction);
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.events;
