/**
 * <h2>Amazon ECS Construct Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This package contains constructs for working with <strong>Amazon Elastic Container
 * Service</strong> (Amazon ECS).
 * <p>
 * Amazon ECS is a highly scalable, fast, container management service
 * that makes it easy to run, stop,
 * and manage Docker containers on a cluster of Amazon EC2 instances.
 * <p>
 * For further information on Amazon ECS,
 * see the <a href="https://docs.aws.amazon.com/ecs">Amazon ECS documentation</a>
 * <p>
 * The following example creates an Amazon ECS cluster,
 * adds capacity to it,
 * and instantiates the Amazon ECS Service with an automatic load balancer.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * // Create an ECS cluster
 * Cluster cluster = new Cluster(this, "Cluster", new ClusterProps()
 *         .vpc(vpc));
 * 
 * // Add capacity to it
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", new AddCapacityOptions()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3));
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("DefaultContainer", new ContainerDefinitionOptions()
 *         .image(ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(512));
 * 
 * // Instantiate an Amazon ECS Service
 * Ec2Service ecsService = new Ec2Service(this, "Service", new Ec2ServiceProps()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition));
 * </pre></blockquote>
 * <p>
 * For a set of constructs defining common ECS architectural patterns, see the <code>&#64;aws-cdk/aws-ecs-patterns</code> package.
 * <p>
 * <h2>Launch Types: AWS Fargate vs Amazon EC2</h2>
 * <p>
 * There are two sets of constructs in this library; one to run tasks on Amazon EC2 and
 * one to run tasks on AWS Fargate.
 * <p>
 * <ul>
 * <li>Use the <code>Ec2TaskDefinition</code> and <code>Ec2Service</code> constructs to run tasks on Amazon EC2 instances running in your account.</li>
 * <li>Use the <code>FargateTaskDefinition</code> and <code>FargateService</code> constructs to run tasks on
 * instances that are managed for you by AWS.</li>
 * </ul>
 * <p>
 * Here are the main differences:
 * <p>
 * <ul>
 * <li><strong>Amazon EC2</strong>: instances are under your control. Complete control of task to host
 * allocation. Required to specify at least a memory reseration or limit for
 * every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
 * Classic Load Balancer. Can share volumes between container and host.</li>
 * <li><strong>AWS Fargate</strong>: tasks run on AWS-managed instances, AWS manages task to host
 * allocation for you. Requires specification of memory and cpu sizes at the
 * taskdefinition level. Only supports AwsVpc networking modes and
 * Application/Network Load Balancers. Only the AWS log driver is supported.
 * Many host features are not supported such as adding kernel capabilities
 * and mounting host devices/volumes inside the container.</li>
 * </ul>
 * <p>
 * For more information on Amazon EC2 vs AWS Fargate and networking see the AWS Documentation:
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate</a> and
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>.
 * <p>
 * <h2>Clusters</h2>
 * <p>
 * A <code>Cluster</code> defines the infrastructure to run your
 * tasks on. You can run many tasks on a single cluster.
 * <p>
 * The following code creates a cluster that can run AWS Fargate tasks:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use tasks with Amazon EC2 launch-type, you have to add capacity to
 * the cluster in order for tasks to be scheduled on your instances.  Typically,
 * you add an AutoScalingGroup with instances running the latest
 * Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
 * AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
 * that you construct yourself. It's possible to add multiple AutoScalingGroups
 * with various instance types.
 * <p>
 * The following example creates an Amazon ECS cluster and adds capacity to it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * // Either add default capacity
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", Map.of(
 *         "instanceType", new InstanceType("t2.xlarge"),
 *         "desiredCapacity", 3));
 * 
 * // Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
 * var autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .machineImage(EcsOptimizedImage.amazonLinux())
 *         // Or use Amazon ECS-Optimized Amazon Linux 2 AMI
 *         // machineImage: EcsOptimizedImage.amazonLinux2(),
 *         .desiredCapacity(3)
 *         .build();
 * 
 * cluster.addAutoScalingGroup(autoScalingGroup);
 * </pre></blockquote>
 * <p>
 * If you omit the property <code>vpc</code>, the construct will create a new VPC with two AZs.
 * <p>
 * <h3>Spot Instances</h3>
 * <p>
 * To add spot instances into the cluster, you must specify the <code>spotPrice</code> in the <code>ecs.AddCapacityOptions</code> and optionally enable the <code>spotInstanceDraining</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Add an AutoScalingGroup with spot instances to the existing cluster
 * cluster.addCapacity("AsgSpot", Map.of(
 *         "maxCapacity", 2,
 *         "minCapacity", 2,
 *         "desiredCapacity", 2,
 *         "instanceType", new InstanceType("c5.xlarge"),
 *         "spotPrice", "0.0735",
 *         // Enable the Automated Spot Draining support for Amazon ECS
 *         "spotInstanceDraining", true));
 * </pre></blockquote>
 * <p>
 * <h3>SNS Topic Encryption</h3>
 * <p>
 * When the <code>ecs.AddCapacityOptions</code> that you provide has a non-zero <code>taskDrainTime</code> (the default) then an SNS topic and Lambda are created to ensure that the
 * cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
 * and the Lambda acts on that event. If you wish to engage <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html">server-side encryption</a> for this SNS Topic
 * then you may do so by providing a KMS key for the <code>topicEncryptionKey</code> propery of <code>ecs.AddCapacityOptions</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Given
 * var key = kms.Key(...);
 * // Then, use that key to encrypt the lifecycle-event SNS Topic.
 * cluster.addCapacity("ASGEncryptedSNS", Map.of(
 *         "instanceType", new InstanceType("t2.xlarge"),
 *         "desiredCapacity", 3,
 *         "topicEncryptionKey", key));
 * </pre></blockquote>
 * <p>
 * <h2>Task definitions</h2>
 * <p>
 * A task Definition describes what a single copy of a <strong>task</strong> should look like.
 * A task definition has one or more containers; typically, it has one
 * main container (the <em>default container</em> is the first one that's added
 * to the task definition, and it is marked <em>essential</em>) and optionally
 * some supporting containers which are used to support the main container,
 * doings things like upload logs or metrics to monitoring services.
 * <p>
 * To run a task or service with Amazon EC2 launch type, use the <code>Ec2TaskDefinition</code>. For AWS Fargate tasks/services, use the
 * <code>FargateTaskDefinition</code>. These classes provide a simplified API that only contain
 * properties relevant for that specific launch type.
 * <p>
 * For a <code>FargateTaskDefinition</code>, specify the task size (<code>memoryLimitMiB</code> and <code>cpu</code>):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add containers to a task definition, call <code>addContainer()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var container = fargateTaskDefinition.addContainer("WebContainer", Map.of(
 *         // Use an image from DockerHub
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")));
 * </pre></blockquote>
 * <p>
 * For a <code>Ec2TaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var ec2TaskDefinition = Ec2TaskDefinition.Builder.create(this, "TaskDef")
 *         .networkMode(NetworkMode.getBRIDGE())
 *         .build();
 * 
 * var container = ec2TaskDefinition.addContainer("WebContainer", Map.of(
 *         // Use an image from DockerHub
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
 *         "memoryLimitMiB", 1024));
 * </pre></blockquote>
 * <p>
 * You can specify container properties when you add them to the task definition, or with various methods, e.g.:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * container.addPortMappings(Map.of(
 *         "containerPort", 3000));
 * </pre></blockquote>
 * <p>
 * To add data volumes to a task definition, call <code>addVolume()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var volume = ecs.Volume("Volume", Map.of(
 *         // Use an Elastic FileSystem
 *         "name", "mydatavolume",
 *         "efsVolumeConfiguration", ecs.EfsVolumeConfiguration(Map.of(
 *                 "fileSystemId", "EFS"))));
 * 
 * var container = fargateTaskDefinition.addVolume("mydatavolume");
 * </pre></blockquote>
 * <p>
 * To use a TaskDefinition that can be used with either Amazon EC2 or
 * AWS Fargate launch types, use the <code>TaskDefinition</code> construct.
 * <p>
 * When creating a task definition you have to specify what kind of
 * tasks you intend to run: Amazon EC2, AWS Fargate, or both.
 * The following example uses both:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var taskDefinition = TaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .networkMode(NetworkMode.getAWS_VPC())
 *         .compatibility(ecs.Compatibility.getEC2_AND_FARGATE())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Images</h3>
 * <p>
 * Images supply the software that runs inside the container. Images can be
 * obtained from either DockerHub or from ECR repositories, or built directly from a local Dockerfile.
 * <p>
 * <ul>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName)</code>: use a public image.</li>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })</code>: use a private image that requires credentials.</li>
 * <li><code>ecs.ContainerImage.fromEcrRepository(repo, tag)</code>: use the given ECR repository as the image
 * to start. If no tag is provided, "latest" is assumed.</li>
 * <li><code>ecs.ContainerImage.fromAsset('./image')</code>: build and upload an
 * image directly from a <code>Dockerfile</code> in your source directory.</li>
 * <li><code>ecs.ContainerImage.fromDockerImageAsset(asset)</code>: uses an existing
 * <code>&#64;aws-cdk/aws-ecr-assets.DockerImageAsset</code> as a container image.</li>
 * </ul>
 * <p>
 * <h3>Environment variables</h3>
 * <p>
 * To pass environment variables to the container, use the <code>environment</code> and <code>secrets</code> props.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * taskDefinition.addContainer("container", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
 *         "memoryLimitMiB", 1024,
 *         "environment", Map.of(// clear text, not for sensitive data
 *                 "STAGE", "prod"),
 *         "secrets", Map.of(// Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
 *                 "SECRET", ecs.Secret.fromSecretsManager(secret),
 *                 "DB_PASSWORD", ecs.Secret.fromSecretsManager(dbSecret, "password"), // Reference a specific JSON field
 *                 "PARAMETER", ecs.Secret.fromSsmParameter(parameter))));
 * </pre></blockquote>
 * <p>
 * The task execution role is automatically granted read permissions on the secrets/parameters.
 * <p>
 * <h2>Service</h2>
 * <p>
 * A <code>Service</code> instantiates a <code>TaskDefinition</code> on a <code>Cluster</code> a given number of
 * times, optionally associating them with a load balancer.
 * If a task fails,
 * Amazon ECS automatically restarts the task.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var taskDefinition = ;
 * 
 * var service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>Services</code> by default will create a security group if not provided.
 * If you'd like to specify which security groups to use you can override the <code>securityGroups</code> property.
 * <p>
 * <h3>Include an application/network load balancer</h3>
 * <p>
 * <code>Services</code> are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * var service = FargateService.Builder.create(this, "Service").build();
 * 
 * ApplicationLoadBalancer lb = new ApplicationLoadBalancer(this, "LB", new ApplicationLoadBalancerProps().vpc(vpc).internetFacing(true));
 * ApplicationListener listener = lb.addListener("Listener", new BaseApplicationListenerProps().port(80));
 * ApplicationTargetGroup targetGroup1 = listener.addTargets("ECS1", new AddApplicationTargetsProps()
 *         .port(80)
 *         .targets(asList(service)));
 * ApplicationTargetGroup targetGroup2 = listener.addTargets("ECS2", new AddApplicationTargetsProps()
 *         .port(80)
 *         .targets(asList(service.loadBalancerTarget(Map.of(
 *                 "containerName", "MyContainer",
 *                 "containerPort", 8080)))));
 * </pre></blockquote>
 * <p>
 * Note that in the example above, the default <code>service</code> only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use <code>service.loadBalancerTarget()</code> to return a load balancing target for a specific container and port.
 * <p>
 * Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * var service = FargateService.Builder.create(this, "Service").build();
 * 
 * ApplicationLoadBalancer lb = new ApplicationLoadBalancer(this, "LB", new ApplicationLoadBalancerProps().vpc(vpc).internetFacing(true));
 * ApplicationListener listener = lb.addListener("Listener", new BaseApplicationListenerProps().port(80));
 * service.registerLoadBalancerTargets(Map.of(
 *         "containerName", "web",
 *         "containerPort", 80,
 *         "newTargetGroupId", "ECS",
 *         "listener", ecs.ListenerConfig.applicationListener(listener, Map.of(
 *                 "protocol", elbv2.ApplicationProtocol.getHTTPS()))));
 * </pre></blockquote>
 * <p>
 * <h3>Using a Load Balancer from a different Stack</h3>
 * <p>
 * If you want to put your Load Balancer and the Service it is load balancing to in
 * different stacks, you may not be able to use the convenience methods
 * <code>loadBalancer.addListener()</code> and <code>listener.addTargets()</code>.
 * <p>
 * The reason is that these methods will create resources in the same Stack as the
 * object they're called on, which may lead to cyclic references between stacks.
 * Instead, you will have to create an <code>ApplicationListener</code> in the service stack,
 * or an empty <code>TargetGroup</code> in the load balancer stack that you attach your
 * service to.
 * <p>
 * See the <a href="https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/">ecs/cross-stack-load-balancer example</a>
 * for the alternatives.
 * <p>
 * <h3>Include a classic load balancer</h3>
 * <p>
 * <code>Services</code> can also be directly attached to a classic load balancer as targets:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.elasticloadbalancing.*;
 * 
 * var service = Ec2Service.Builder.create(this, "Service").build();
 * 
 * LoadBalancer lb = new LoadBalancer(stack, "LB", new LoadBalancerProps().vpc(vpc));
 * lb.addListener(new LoadBalancerListener().externalPort(80));
 * lb.addTarget(service);
 * </pre></blockquote>
 * <p>
 * Similarly, if you want to have more control over load balancer targeting:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.elasticloadbalancing.*;
 * 
 * var service = Ec2Service.Builder.create(this, "Service").build();
 * 
 * LoadBalancer lb = new LoadBalancer(stack, "LB", new LoadBalancerProps().vpc(vpc));
 * lb.addListener(new LoadBalancerListener().externalPort(80));
 * lb.addTarget(service.getLoadBalancerTarget(), Map.of(
 *         "containerName", "MyContainer",
 *         "containerPort", 80));
 * </pre></blockquote>
 * <p>
 * There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:
 * <p>
 * <ul>
 * <li><code>LoadBalancedFargateService</code></li>
 * <li><code>LoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <h2>Task Auto-Scaling</h2>
 * <p>
 * You can configure the task count of a service to match demand. Task auto-scaling is
 * configured by calling <code>autoScaleTaskCount()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var scaling = service.autoScaleTaskCount(Map.of("maxCapacity", 10));
 * scaling.scaleOnCpuUtilization("CpuScaling", Map.of(
 *         "targetUtilizationPercent", 50));
 * 
 * scaling.scaleOnRequestCount("RequestScaling", Map.of(
 *         "requestsPerTarget", 10000,
 *         "targetGroup", target));
 * </pre></blockquote>
 * <p>
 * Task auto-scaling is powered by <em>Application Auto-Scaling</em>.
 * See that section for details.
 * <p>
 * <h2>Instance Auto-Scaling</h2>
 * <p>
 * If you're running on AWS Fargate, AWS manages the physical machines that your
 * containers are running on for you. If you're running an Amazon ECS cluster however,
 * your Amazon EC2 instances might fill up as your number of Tasks goes up.
 * <p>
 * To avoid placement errors, configure auto-scaling for your
 * Amazon EC2 instance group so that your instance count scales with demand. To keep
 * your Amazon EC2 instances halfway loaded, scaling up to a maximum of 30 instances
 * if required:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var autoScalingGroup = cluster.addCapacity("DefaultAutoScalingGroup", Map.of(
 *         "instanceType", new InstanceType("t2.xlarge"),
 *         "minCapacity", 3,
 *         "maxCapacity", 30,
 *         "desiredCapacity", 3,
 * 
 *         // Give instances 5 minutes to drain running tasks when an instance is
 *         // terminated. This is the default, turn this off by specifying 0 or
 *         // change the timeout up to 900 seconds.
 *         "taskDrainTime", Duration.seconds(300)));
 * 
 * autoScalingGroup.scaleOnCpuUtilization("KeepCpuHalfwayLoaded", Map.of(
 *         "targetUtilizationPercent", 50));
 * </pre></blockquote>
 * <p>
 * See the <code>&#64;aws-cdk/aws-autoscaling</code> library for more autoscaling options
 * you can configure on your instances.
 * <p>
 * <h2>Integration with CloudWatch Events</h2>
 * <p>
 * To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
 * <code>&#64;aws-cdk/aws-events-targets.EcsTask</code> instead of an <code>Ec2Service</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromAsset(path.resolve(__dirname, "..", "eventhandler-image")),
 *         "memoryLimitMiB", 256,
 *         "logging", AwsLogDriver.Builder.create().streamPrefix("EventDemo").build()));
 * 
 * // An Rule that describes the event trigger (in this case a scheduled run)
 * var rule = Rule.Builder.create(this, "Rule")
 *         .schedule(events.Schedule.expression("rate(1 min)"))
 *         .build();
 * 
 * // Pass an environment variable to the container 'TheContainer' in the task
 * rule.addTarget(new EcsTask(new EcsTaskProps()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 *         .containerOverrides(asList(new ContainerOverride()
 *                 .containerName("TheContainer")
 *                 .environment(asList(new TaskEnvironmentVariable()
 *                         .name("I_WAS_TRIGGERED")
 *                         .value("From CloudWatch Events")))))));
 * </pre></blockquote>
 * <p>
 * <h2>Log Drivers</h2>
 * <p>
 * Currently Supported Log Drivers:
 * <p>
 * <ul>
 * <li>awslogs</li>
 * <li>fluentd</li>
 * <li>gelf</li>
 * <li>journald</li>
 * <li>json-file</li>
 * <li>splunk</li>
 * <li>syslog</li>
 * <li>awsfirelens</li>
 * </ul>
 * <p>
 * <h3>awslogs Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.awsLogs(Map.of("streamPrefix", "EventDemo"))));
 * </pre></blockquote>
 * <p>
 * <h3>fluentd Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.fluentd()));
 * </pre></blockquote>
 * <p>
 * <h3>gelf Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.gelf(Map.of("address", "my-gelf-address"))));
 * </pre></blockquote>
 * <p>
 * <h3>journald Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.journald()));
 * </pre></blockquote>
 * <p>
 * <h3>json-file Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.jsonFile()));
 * </pre></blockquote>
 * <p>
 * <h3>splunk Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.splunk(Map.of(
 *                 "token", cdk.SecretValue.secretsManager("my-splunk-token"),
 *                 "url", "my-splunk-url"))));
 * </pre></blockquote>
 * <p>
 * <h3>syslog Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.syslog()));
 * </pre></blockquote>
 * <p>
 * <h3>firelens Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.firelens(Map.of(
 *                 "options", Map.of(
 *                         "Name", "firehose",
 *                         "region", "us-west-2",
 *                         "delivery_stream", "my-stream")))));
 * </pre></blockquote>
 * <p>
 * <h3>Generic Log Driver</h3>
 * <p>
 * A generic log driver object exists to provide a lower level abstraction of the log driver configuration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * var taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", GenericLogDriver.Builder.create()
 *                 .logDriver("fluentd")
 *                 .options(Map.of(
 *                         "tag", "example-tag"))
 *                 .build()));
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ecs;
