/**
 * <h1>AWS Cloud Development Kit Core Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This library includes the basic building blocks of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> (AWS CDK). It defines the core classes that are used in the rest of the
 * AWS Construct Library.
 * <p>
 * See the <a href="https://docs.aws.amazon.com/cdk/latest/guide/home.html">AWS CDK Developer
 * Guide</a> for
 * information of most of the capabilities of this library. The rest of this
 * README will only cover topics not already covered in the Developer Guide.
 * <p>
 * <h2>Stacks and Stages</h2>
 * <p>
 * A <code>Stack</code> is the smallest physical unit of deployment, and maps directly onto
 * a CloudFormation Stack. You define a Stack by defining a subclass of <code>Stack</code>
 * -- let's call it <code>MyStack</code> -- and instantiating the constructs that make up
 * your application in <code>MyStack</code>'s constructor. You then instantiate this stack
 * one or more times to define different instances of your application. For example,
 * you can instantiate it once using few and cheap EC2 instances for testing,
 * and once again using more and bigger EC2 instances for production.
 * <p>
 * When your application grows, you may decide that it makes more sense to split it
 * out across multiple <code>Stack</code> classes. This can happen for a number of reasons:
 * <p>
 * <ul>
 * <li>You could be starting to reach the maximum number of resources allowed in a single
 * stack (this is currently 500).</li>
 * <li>You could decide you want to separate out stateful resources and stateless resources
 * into separate stacks, so that it becomes easy to tear down and recreate the stacks
 * that don't have stateful resources.</li>
 * <li>There could be a single stack with resources (like a VPC) that are shared
 * between multiple instances of other stacks containing your applications.</li>
 * </ul>
 * <p>
 * As soon as your conceptual application starts to encompass multiple stacks,
 * it is convenient to wrap them in another construct that represents your
 * logical application. You can then treat that new unit the same way you used
 * to be able to treat a single stack: by instantiating it multiple times
 * for different instances of your application.
 * <p>
 * You can define a custom subclass of <code>Construct</code>, holding one or more
 * <code>Stack</code>s, to represent a single logical instance of your application.
 * <p>
 * As a final note: <code>Stack</code>s are not a unit of reuse. They describe physical
 * deployment layouts, and as such are best left to application builders to
 * organize their deployments with. If you want to vend a reusable construct,
 * define it as a subclasses of <code>Construct</code>: the consumers of your construct
 * will decide where to place it in their own stacks.
 * <p>
 * <h2>Nested Stacks</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Nested stacks</a> are stacks created as part of other stacks. You create a nested stack within another stack by using the <code>NestedStack</code> construct.
 * <p>
 * As your infrastructure grows, common patterns can emerge in which you declare the same components in multiple templates. You can separate out these common components and create dedicated templates for them. Then use the resource in your template to reference other templates, creating nested stacks.
 * <p>
 * For example, assume that you have a load balancer configuration that you use for most of your stacks. Instead of copying and pasting the same configurations into your templates, you can create a dedicated template for the load balancer. Then, you just use the resource to reference that template from within other templates.
 * <p>
 * The following example will define a single top-level stack that contains two nested stacks: each one with a single Amazon S3 bucket:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * public class MyNestedStack extends NestedStack {
 *     public MyNestedStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyNestedStack(Construct scope, String id, NestedStackProps props) {
 *         super(scope, id, props);
 * 
 *         new Bucket(this, "NestedBucket");
 *     }
 * }
 * 
 * public class MyParentStack extends Stack {
 *     public MyParentStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyParentStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         new MyNestedStack(this, "Nested1");
 *         new MyNestedStack(this, "Nested2");
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Resources references across nested/parent boundaries (even with multiple levels of nesting) will be wired by the AWS CDK
 * through CloudFormation parameters and outputs. When a resource from a parent stack is referenced by a nested stack,
 * a CloudFormation parameter will automatically be added to the nested stack and assigned from the parent; when a resource
 * from a nested stack is referenced by a parent stack, a CloudFormation output will be automatically be added to the
 * nested stack and referenced using <code>Fn::GetAtt "Outputs.Xxx"</code> from the parent.
 * <p>
 * Nested stacks also support the use of Docker image and file assets.
 * <p>
 * <h2>Accessing resources in a different stack</h2>
 * <p>
 * You can access resources in a different stack, as long as they are in the
 * same account and AWS Region. The following example defines the stack <code>stack1</code>,
 * which defines an Amazon S3 bucket. Then it defines a second stack, <code>stack2</code>,
 * which takes the bucket from stack1 as a constructor property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Map&lt;String, String&gt; prod = Map.of("account", "123456789012", "region", "us-east-1");
 * 
 * Object stack1 = StackThatProvidesABucket.Builder.create(app, "Stack1").env(prod).build();
 * 
 * // stack2 will take a property { bucket: IBucket }
 * Object stack2 = StackThatExpectsABucket.Builder.create(app, "Stack2")
 *         .bucket(stack1.getBucket())
 *         .env(prod)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If the AWS CDK determines that the resource is in the same account and
 * Region, but in a different stack, it automatically synthesizes AWS
 * CloudFormation
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-exports.html">Exports</a>
 * in the producing stack and an
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html">Fn::ImportValue</a>
 * in the consuming stack to transfer that information from one stack to the
 * other.
 * <p>
 * <h3>Removing automatic cross-stack references</h3>
 * <p>
 * The automatic references created by CDK when you use resources across stacks
 * are convenient, but may block your deployments if you want to remove the
 * resources that are referenced in this way. You will see an error like:
 * <p>
 * <blockquote><pre>
 * Export Stack1:ExportsOutputFnGetAtt-****** cannot be deleted as it is in use by Stack1
 * </pre></blockquote>
 * <p>
 * Let's say there is a Bucket in the <code>stack1</code>, and the <code>stack2</code> references its
 * <code>bucket.bucketName</code>. You now want to remove the bucket and run into the error above.
 * <p>
 * It's not safe to remove <code>stack1.bucket</code> while <code>stack2</code> is still using it, so
 * unblocking yourself from this is a two-step process. This is how it works:
 * <p>
 * DEPLOYMENT 1: break the relationship
 * <p>
 * <ul>
 * <li>Make sure <code>stack2</code> no longer references <code>bucket.bucketName</code> (maybe the consumer
 * stack now uses its own bucket, or it writes to an AWS DynamoDB table, or maybe you just
 * remove the Lambda Function altogether).</li>
 * <li>In the <code>stack1</code> class, call <code>this.exportAttribute(this.bucket.bucketName)</code>. This
 * will make sure the CloudFormation Export continues to exist while the relationship
 * between the two stacks is being broken.</li>
 * <li>Deploy (this will effectively only change the <code>stack2</code>, but it's safe to deploy both).</li>
 * </ul>
 * <p>
 * DEPLOYMENT 2: remove the resource
 * <p>
 * <ul>
 * <li>You are now free to remove the <code>bucket</code> resource from <code>stack1</code>.</li>
 * <li>Don't forget to remove the <code>exportAttribute()</code> call as well.</li>
 * <li>Deploy again (this time only the <code>stack1</code> will be changed -- the bucket will be deleted).</li>
 * </ul>
 * <p>
 * <h2>Durations</h2>
 * <p>
 * To make specifications of time intervals unambiguous, a single class called
 * <code>Duration</code> is used throughout the AWS Construct Library by all constructs
 * that that take a time interval as a parameter (be it for a timeout, a
 * rate, or something else).
 * <p>
 * An instance of Duration is constructed by using one of the static factory
 * methods on it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Duration.seconds(300);// 5 minutes
 * Duration.minutes(5);// 5 minutes
 * Duration.hours(1);// 1 hour
 * Duration.days(7);// 7 days
 * Duration.parse("PT5M");
 * </pre></blockquote>
 * <p>
 * <h2>Size (Digital Information Quantity)</h2>
 * <p>
 * To make specification of digital storage quantities unambiguous, a class called
 * <code>Size</code> is available.
 * <p>
 * An instance of <code>Size</code> is initialized through one of its static factory methods:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Size.kibibytes(200);// 200 KiB
 * Size.mebibytes(5);// 5 MiB
 * Size.gibibytes(40);// 40 GiB
 * Size.tebibytes(200);// 200 TiB
 * Size.pebibytes(3);
 * </pre></blockquote>
 * <p>
 * Instances of <code>Size</code> created with one of the units can be converted into others.
 * By default, conversion to a higher unit will fail if the conversion does not produce
 * a whole number. This can be overridden by unsetting <code>integral</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Size.mebibytes(2).toKibibytes();// yields 2048
 * Size.kibibytes(2050).toMebibytes(new SizeConversionOptions().rounding(SizeRoundingBehavior.getFLOOR()));
 * </pre></blockquote>
 * <p>
 * <h2>Secrets</h2>
 * <p>
 * To help avoid accidental storage of secrets as plain text, we use the <code>SecretValue</code> type to
 * represent secrets. Any construct that takes a value that should be a secret (such as
 * a password or an access key) will take a parameter of type <code>SecretValue</code>.
 * <p>
 * The best practice is to store secrets in AWS Secrets Manager and reference them using <code>SecretValue.secretsManager</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SecretValue secret = SecretValue.secretsManager("secretId", new SecretsManagerSecretOptions()
 *         .jsonField("password")// optional: key of a JSON field to retrieve (defaults to all content),
 *         .versionId("id")// optional: id of the version (default AWSCURRENT)
 *         .versionStage("stage"));
 * </pre></blockquote>
 * <p>
 * Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
 * <code>SecretValue</code> also supports the following secret sources:
 * <p>
 * <ul>
 * <li><code>SecretValue.plainText(secret)</code>: stores the secret as plain text in your app and the resulting template (not recommended).</li>
 * <li><code>SecretValue.ssmSecure(param, version)</code>: refers to a secret stored as a SecureString in the SSM Parameter Store.</li>
 * <li><code>SecretValue.cfnParameter(param)</code>: refers to a secret passed through a CloudFormation parameter (must have <code>NoEcho: true</code>).</li>
 * <li><code>SecretValue.cfnDynamicReference(dynref)</code>: refers to a secret described by a CloudFormation dynamic reference (used by <code>ssmSecure</code> and <code>secretsManager</code>).</li>
 * </ul>
 * <p>
 * <h2>ARN manipulation</h2>
 * <p>
 * Sometimes you will need to put together or pick apart Amazon Resource Names
 * (ARNs). The functions <code>stack.formatArn()</code> and <code>stack.parseArn()</code> exist for
 * this purpose.
 * <p>
 * <code>formatArn()</code> can be used to build an ARN from components. It will automatically
 * use the region and account of the stack you're calling it on:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Builds "arn:&lt;PARTITION&gt;:lambda:&lt;REGION&gt;:&lt;ACCOUNT&gt;:function:MyFunction"
 * stack.formatArn(new ArnComponents()
 *         .service("lambda")
 *         .resource("function")
 *         .sep(":")
 *         .resourceName("MyFunction"));
 * </pre></blockquote>
 * <p>
 * <code>parseArn()</code> can be used to get a single component from an ARN. <code>parseArn()</code>
 * will correctly deal with both literal ARNs and deploy-time values (tokens),
 * but in case of a deploy-time value be aware that the result will be another
 * deploy-time value which cannot be inspected in the CDK application.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Extracts the function name out of an AWS Lambda Function ARN
 * ArnComponents arnComponents = stack.parseArn(arn, ":");
 * String functionName = arnComponents.getResourceName();
 * </pre></blockquote>
 * <p>
 * Note that depending on the service, the resource separator can be either
 * <code>:</code> or <code>/</code>, and the resource name can be either the 6th or 7th
 * component in the ARN. When using these functions, you will need to know
 * the format of the ARN you are dealing with.
 * <p>
 * For an exhaustive list of ARN formats used in AWS, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">AWS ARNs and
 * Namespaces</a>
 * in the AWS General Reference.
 * <p>
 * <h2>Dependencies</h2>
 * <p>
 * <h3>Construct Dependencies</h3>
 * <p>
 * Sometimes AWS resources depend on other resources, and the creation of one
 * resource must be completed before the next one can be started.
 * <p>
 * In general, CloudFormation will correctly infer the dependency relationship
 * between resources based on the property values that are used. In the cases where
 * it doesn't, the AWS Construct Library will add the dependency relationship for
 * you.
 * <p>
 * If you need to add an ordering dependency that is not automatically inferred,
 * you do so by adding a dependency relationship using
 * <code>constructA.node.addDependency(constructB)</code>. This will add a dependency
 * relationship between all resources in the scope of <code>constructA</code> and all
 * resources in the scope of <code>constructB</code>.
 * <p>
 * If you want a single object to represent a set of constructs that are not
 * necessarily in the same scope, you can use a <code>ConcreteDependable</code>. The
 * following creates a single object that represents a dependency on two
 * constructs, <code>constructB</code> and <code>constructC</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Declare the dependable object
 * ConcreteDependable bAndC = new ConcreteDependable();
 * bAndC.add(constructB);
 * bAndC.add(constructC);
 * 
 * // Take the dependency
 * constructA.node.addDependency(bAndC);
 * </pre></blockquote>
 * <p>
 * <h3>Stack Dependencies</h3>
 * <p>
 * Two different stack instances can have a dependency on one another. This
 * happens when an resource from one stack is referenced in another stack. In
 * that case, CDK records the cross-stack referencing of resources,
 * automatically produces the right CloudFormation primitives, and adds a
 * dependency between the two stacks. You can also manually add a dependency
 * between two stacks by using the <code>stackA.addDependency(stackB)</code> method.
 * <p>
 * A stack dependency has the following implications:
 * <p>
 * <ul>
 * <li>Cyclic dependencies are not allowed, so if <code>stackA</code> is using resources from
 * <code>stackB</code>, the reverse is not possible anymore.</li>
 * <li>Stacks with dependencies between them are treated specially by the CDK
 * toolkit:
 * <p>
 * <ul>
 * <li>If <code>stackA</code> depends on <code>stackB</code>, running <code>cdk deploy stackA</code> will also
 * automatically deploy <code>stackB</code>.</li>
 * <li><code>stackB</code>'s deployment will be performed <em>before</em> <code>stackA</code>'s deployment.</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Custom Resources</h2>
 * <p>
 * Custom Resources are CloudFormation resources that are implemented by arbitrary
 * user code. They can do arbitrary lookups or modifications during a
 * CloudFormation deployment.
 * <p>
 * To define a custom resource, use the <code>CustomResource</code> construct:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CustomResource(this, "MyMagicalResource", new CustomResourceProps()
 *         .resourceType("Custom::MyCustomResource")// must start with 'Custom::'
 * 
 *         // the resource properties
 *         .properties(Map.of(
 *                 "Property1", "foo",
 *                 "Property2", "bar"))
 * 
 *         // the ARN of the provider (SNS/Lambda) which handles
 *         // CREATE, UPDATE or DELETE events for this resource type
 *         // see next section for details
 *         .serviceToken("ARN"));
 * </pre></blockquote>
 * <p>
 * <h3>Custom Resource Providers</h3>
 * <p>
 * Custom resources are backed by a <strong>custom resource provider</strong> which can be
 * implemented in one of the following ways. The following table compares the
 * various provider types (ordered from low-level to high-level):
 * <p>
 * | Provider                                                             | Compute Type | Error Handling | Submit to CloudFormation | Max Timeout     | Language | Footprint |
 * |----------------------------------------------------------------------|:------------:|:--------------:|:------------------------:|:---------------:|:--------:|:---------:|
 * | <a href="#amazon-sns-topic">sns.Topic</a>                                       | Self-managed | Manual         | Manual                   | Unlimited       | Any      | Depends   |
 * | <a href="#aws-lambda-function">lambda.Function</a>                              | AWS Lambda   | Manual         | Manual                   | 15min           | Any      | Small     |
 * | <a href="#the-corecustomresourceprovider-class">core.CustomResourceProvider</a> | Lambda       | Auto           | Auto                     | 15min           | Node.js  | Small     |
 * | <a href="#the-custom-resource-provider-framework">custom-resources.Provider</a> | Lambda       | Auto           | Auto                     | Unlimited Async | Any      | Large     |
 * <p>
 * Legend:
 * <p>
 * <ul>
 * <li><strong>Compute type</strong>: which type of compute can is used to execute the handler.</li>
 * <li><strong>Error Handling</strong>: whether errors thrown by handler code are automatically
 * trapped and a FAILED response is submitted to CloudFormation. If this is
 * "Manual", developers must take care of trapping errors. Otherwise, events
 * could cause stacks to hang.</li>
 * <li><strong>Submit to CloudFormation</strong>: whether the framework takes care of submitting
 * SUCCESS/FAILED responses to CloudFormation through the event's response URL.</li>
 * <li><strong>Max Timeout</strong>: maximum allows/possible timeout.</li>
 * <li><strong>Language</strong>: which programming languages can be used to implement handlers.</li>
 * <li><strong>Footprint</strong>: how many resources are used by the provider framework itself.</li>
 * </ul>
 * <p>
 * <strong>A NOTE ABOUT SINGLETONS</strong>
 * <p>
 * When defining resources for a custom resource provider, you will likely want to
 * define them as a <em>stack singleton</em> so that only a single instance of the
 * provider is created in your stack and which is used by all custom resources of
 * that type.
 * <p>
 * Here is a basic pattern for defining stack singletons in the CDK. The following
 * examples ensures that only a single SNS topic is defined:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * public Topic getOrCreate(Construct scope) {
 *     Stack stack = Stack.of(scope);
 *     String uniqueid = "GloballyUniqueIdForSingleton";return (Topic)stack.node.tryFindChild(uniqueid) ?? new Topic(stack, uniqueid)
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Amazon SNS Topic</h4>
 * <p>
 * Every time a resource event occurs (CREATE/UPDATE/DELETE), an SNS notification
 * is sent to the SNS topic. Users must process these notifications (e.g. through a
 * fleet of worker hosts) and submit success/failure responses to the
 * CloudFormation service.
 * <p>
 * Set <code>serviceToken</code> to <code>topic.topicArn</code>  in order to use this provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Topic topic = new Topic(this, "MyProvider");
 * 
 * new CustomResource(this, "MyResource", new CustomResourceProps()
 *         .serviceToken(topic.getTopicArn()));
 * </pre></blockquote>
 * <p>
 * <h4>AWS Lambda Function</h4>
 * <p>
 * An AWS lambda function is called <em>directly</em> by CloudFormation for all resource
 * events. The handler must take care of explicitly submitting a success/failure
 * response to the CloudFormation service and handle various error cases.
 * <p>
 * Set <code>serviceToken</code> to <code>lambda.functionArn</code> to use this provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Function fn = new Function(this, "MyProvider", functionProps);
 * 
 * new CustomResource(this, "MyResource", new CustomResourceProps()
 *         .serviceToken(fn.getFunctionArn()));
 * </pre></blockquote>
 * <p>
 * <h4>The <code>core.CustomResourceProvider</code> class</h4>
 * <p>
 * The class <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_core.CustomResourceProvider.html"><code>&#64;aws-cdk/core.CustomResourceProvider</code></a> offers a basic low-level
 * framework designed to implement simple and slim custom resource providers. It
 * currently only supports Node.js-based user handlers, and it does not have
 * support for asynchronous waiting (handler cannot exceed the 15min lambda
 * timeout).
 * <p>
 * The provider has a built-in singleton method which uses the resource type as a
 * stack-unique identifier and returns the service token:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * String serviceToken = CustomResourceProvider.getOrCreate(this, "Custom::MyCustomResourceType", new CustomResourceProviderProps()
 *         .codeDirectory(String.format("%s/my-handler", __dirname))
 *         .runtime(CustomResourceProviderRuntime.getNODEJS_12()));
 * 
 * new CustomResource(this, "MyResource", new CustomResourceProps()
 *         .resourceType("Custom::MyCustomResourceType")
 *         .serviceToken(serviceToken));
 * </pre></blockquote>
 * <p>
 * The directory (<code>my-handler</code> in the above example) must include an <code>index.js</code> file. It cannot import
 * external dependencies or files outside this directory. It must export an async
 * function named <code>handler</code>. This function accepts the CloudFormation resource
 * event object and returns an object with the following structure:
 * <p>
 * <blockquote><pre>
 * exports.handler = async function(event) {
 *   const id = event.PhysicalResourceId; // only for "Update" and "Delete"
 *   const props = event.ResourceProperties;
 *   const oldProps = event.OldResourceProperties; // only for "Update"s
 * 
 *   switch (event.RequestType) {
 *     case "Create":
 *       // ...
 * 
 *     case "Update":
 *       // ...
 * 
 *       // if an error is thrown, a FAILED response will be submitted to CFN
 *       throw new Error('Failed!');
 * 
 *     case "Delete":
 *       // ...
 *   }
 * 
 *   return {
 *     // (optional) the value resolved from `resource.ref`
 *     // defaults to "event.PhysicalResourceId" or "event.RequestId"
 *     PhysicalResourceId: "REF",
 * 
 *     // (optional) calling `resource.getAtt("Att1")` on the custom resource in the CDK app
 *     // will return the value "BAR".
 *     Data: {
 *       Att1: "BAR",
 *       Att2: "BAZ"
 *     },
 * 
 *     // (optional) user-visible message
 *     Reason: "User-visible message",
 * 
 *     // (optional) hides values from the console
 *     NoEcho: true
 *   };
 * }
 * </pre></blockquote>
 * <p>
 * Here is an complete example of a custom resource that summarizes two numbers:
 * <p>
 * <code>sum-handler/index.js</code>:
 * <p>
 * <blockquote><pre>
 * exports.handler = async (e) =&gt; {
 *   return {
 *     Data: {
 *       Result: e.ResourceProperties.lhs + e.ResourceProperties.rhs,
 *     },
 *   };
 * };
 * </pre></blockquote>
 * <p>
 * <code>sum.ts</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.core.Construct;
 * import software.amazon.awscdk.core.CustomResource;
 * import software.amazon.awscdk.core.CustomResourceProvider;
 * import software.amazon.awscdk.core.CustomResourceProviderRuntime;
 * import software.amazon.awscdk.core.Token;
 * 
 * public class SumProps {
 *     private Number lhs;
 *     public Number getLhs() {
 *         return this.lhs;
 *     }
 *     public SumProps lhs(Number lhs) {
 *         this.lhs = lhs;
 *         return this;
 *     }
 *     private Number rhs;
 *     public Number getRhs() {
 *         return this.rhs;
 *     }
 *     public SumProps rhs(Number rhs) {
 *         this.rhs = rhs;
 *         return this;
 *     }
 * }
 * 
 * public class Sum extends Construct {
 *     public final Number result;
 * 
 *     public Sum(Construct scope, String id, SumProps props) {
 *         super(scope, id);
 * 
 *         String resourceType = "Custom::Sum";
 *         String serviceToken = CustomResourceProvider.getOrCreate(this, resourceType, new CustomResourceProviderProps()
 *                 .codeDirectory(String.format("%s/sum-handler", __dirname))
 *                 .runtime(CustomResourceProviderRuntime.getNODEJS_12()));
 * 
 *         CustomResource resource = new CustomResource(this, "Resource", new CustomResourceProps()
 *                 .resourceType(resourceType)
 *                 .serviceToken(serviceToken)
 *                 .properties(Map.of(
 *                         "lhs", props.getLhs(),
 *                         "rhs", props.getRhs())));
 * 
 *         this.result = Token.asNumber(resource.getAtt("Result"));
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Usage will look like this:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Sum sum = new Sum(this, "MySum", new SumProps().lhs(40).rhs(2));
 * new CfnOutput(this, "Result", new CfnOutputProps().value(Token.asString(sum.getResult())));
 * </pre></blockquote>
 * <p>
 * To access the ARN of the provider's AWS Lambda function role, use the <code>getOrCreateProvider()</code>
 * built-in singleton method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CustomResourceProvider provider = CustomResourceProvider.getOrCreateProvider(this, "Custom::MyCustomResourceType", new CustomResourceProviderProps()
 *         .codeDirectory(String.format("%s/my-handler", __dirname))
 *         .runtime(CustomResourceProviderRuntime.getNODEJS_12()));
 * 
 * String roleArn = provider.getRoleArn();
 * </pre></blockquote>
 * <p>
 * This role ARN can then be used in resource-based IAM policies.
 * <p>
 * <h4>The Custom Resource Provider Framework</h4>
 * <p>
 * The <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html"><code>&#64;aws-cdk/custom-resources</code></a> module includes an advanced framework for
 * implementing custom resource providers.
 * <p>
 * Handlers are implemented as AWS Lambda functions, which means that they can be
 * implemented in any Lambda-supported runtime. Furthermore, this provider has an
 * asynchronous mode, which means that users can provide an <code>isComplete</code> lambda
 * function which is called periodically until the operation is complete. This
 * allows implementing providers that can take up to two hours to stabilize.
 * <p>
 * Set <code>serviceToken</code> to <code>provider.serviceToken</code> to use this type of provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Provider provider = new Provider(this, "MyProvider", new ProviderProps()
 *         .onEventHandler(onEventHandler)
 *         .isCompleteHandler(isCompleteHandler));
 * 
 * new CustomResource(this, "MyResource", new CustomResourceProps()
 *         .serviceToken(provider.getServiceToken()));
 * </pre></blockquote>
 * <p>
 * See the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html">documentation</a> for more details.
 * <p>
 * <h2>AWS CloudFormation features</h2>
 * <p>
 * A CDK stack synthesizes to an AWS CloudFormation Template. This section
 * explains how this module allows users to access low-level CloudFormation
 * features when needed.
 * <p>
 * <h3>Stack Outputs</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">stack outputs</a> and exports are created using
 * the <code>CfnOutput</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CfnOutput(this, "OutputName", new CfnOutputProps()
 *         .value(myBucket.getBucketName())
 *         .description("The name of an S3 bucket")// Optional
 *         .exportName("TheAwesomeBucket"));
 * </pre></blockquote>
 * <p>
 * <h3>Parameters</h3>
 * <p>
 * CloudFormation templates support the use of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a> to
 * customize a template. They enable CloudFormation users to input custom values to
 * a template each time a stack is created or updated. While the CDK design
 * philosophy favors using build-time parameterization, users may need to use
 * CloudFormation in a number of cases (for example, when migrating an existing
 * stack to the AWS CDK).
 * <p>
 * Template parameters can be added to a stack by using the <code>CfnParameter</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CfnParameter(this, "MyParameter", new CfnParameterProps()
 *         .type("Number")
 *         .default(1337));
 * </pre></blockquote>
 * <p>
 * The value of parameters can then be obtained using one of the <code>value</code> methods.
 * As parameters are only resolved at deployment time, the values obtained are
 * placeholder tokens for the real value (<code>Token.isUnresolved()</code> would return <code>true</code>
 * for those):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnParameter param = new CfnParameter(this, "ParameterName", new CfnParameterProps());
 * 
 * // If the parameter is a String
 * param.getValueAsString();
 * 
 * // If the parameter is a Number
 * param.getValueAsNumber();
 * 
 * // If the parameter is a List
 * param.getValueAsList();
 * </pre></blockquote>
 * <p>
 * <h3>Pseudo Parameters</h3>
 * <p>
 * CloudFormation supports a number of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/pseudo-parameter-reference.html">pseudo parameters</a>,
 * which resolve to useful values at deployment time. CloudFormation pseudo
 * parameters can be obtained from static members of the <code>Aws</code> class.
 * <p>
 * It is generally recommended to access pseudo parameters from the scope's <code>stack</code>
 * instead, which guarantees the values produced are qualifying the designated
 * stack, which is essential in cases where resources are shared cross-stack:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // "this" is the current construct
 * Stack stack = Stack.of(this);
 * 
 * stack.getAccount();// Returns the AWS::AccountId for this stack (or the literal value if known)
 * stack.getRegion();// Returns the AWS::Region for this stack (or the literal value if known)
 * stack.getPartition();
 * </pre></blockquote>
 * <p>
 * <h3>Resource Options</h3>
 * <p>
 * CloudFormation resources can also specify <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-product-attribute-reference.html">resource
 * attributes</a>. The <code>CfnResource</code> class allows
 * accessing those through the <code>cfnOptions</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnBucket rawBucket = new CfnBucket(this, "Bucket", new CfnBucketProps());
 * // -or-
 * CfnBucket rawBucketAlt = (CfnBucket)myBucket.node.getDefaultChild();
 * 
 * // then
 * rawBucket.cfnOptions.getCondition() = new CfnCondition(this, "EnableBucket", new CfnConditionProps());
 * rawBucket.cfnOptions.getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * Resource dependencies (the <code>DependsOn</code> attribute) is modified using the
 * <code>cfnResource.addDependsOn</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnResource resourceA = new CfnResource(this, "ResourceA", resourceProps);
 * CfnResource resourceB = new CfnResource(this, "ResourceB", resourceProps);
 * 
 * resourceB.addDependsOn(resourceA);
 * </pre></blockquote>
 * <p>
 * <h3>Intrinsic Functions and Condition Expressions</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html">intrinsic functions</a>. These functions
 * can be accessed from the <code>Fn</code> class, which provides type-safe methods for each
 * intrinsic function as well as condition expressions:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // To use Fn::Base64
 * Fn.base64("SGVsbG8gQ0RLIQo=");
 * 
 * // To compose condition expressions:
 * CfnParameter environmentParameter = new CfnParameter(this, "Environment");
 * Fn.conditionAnd(Fn.conditionEquals("Production", environmentParameter), Fn.conditionNot(Fn.conditionEquals("us-east-1", Aws.getREGION())));
 * </pre></blockquote>
 * <p>
 * When working with deploy-time values (those for which <code>Token.isUnresolved</code>
 * returns <code>true</code>), idiomatic conditionals from the programming language cannot be
 * used (the value will not be known until deployment time). When conditional logic
 * needs to be expressed with un-resolved values, it is necessary to use
 * CloudFormation conditions by means of the <code>CfnCondition</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnParameter environmentParameter = new CfnParameter(this, "Environment");
 * CfnCondition isProd = new CfnCondition(this, "IsProduction", new CfnConditionProps()
 *         .expression(Fn.conditionEquals("Production", environmentParameter)));
 * 
 * // Configuration value that is a different string based on IsProduction
 * String stage = Fn.conditionIf(isProd.logicalId, 'Beta', 'Prod').toString();
 * 
 * // Make Bucket creation condition to IsProduction by accessing
 * // and overriding the CloudFormation resource
 * Bucket bucket = new Bucket(this, "Bucket");
 * CfnBucket cfnBucket = (CfnBucket)myBucket.node.getDefaultChild();
 * cfnBucket.cfnOptions.getCondition() = isProd;
 * </pre></blockquote>
 * <p>
 * <h3>Mappings</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">mappings</a> are created and queried using the
 * <code>CfnMappings</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnMapping regionTable = new CfnMapping(this, "RegionTable", new CfnMappingProps()
 *         .mapping(Map.of(
 *                 "regionName", Map.of(
 *                         "us-east-1", "US East (N. Virginia)",
 *                         "us-east-2", "US East (Ohio)"))));
 * 
 * regionTable.findInMap("regionName", Aws.getREGION());
 * </pre></blockquote>
 * <p>
 * This will yield the following template:
 * <p>
 * <blockquote><pre>
 * Mappings:
 *   RegionTable:
 *     regionName:
 *       us-east-1: US East (N. Virginia)
 *       us-east-2: US East (Ohio)
 * </pre></blockquote>
 * <p>
 * <h3>Dynamic References</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">dynamically resolving</a> values
 * for SSM parameters (including secure strings) and Secrets Manager. Encoding such
 * references is done using the <code>CfnDynamicReference</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CfnDynamicReference(CfnDynamicReferenceService.getSECRETS_MANAGER(), "secret-id:secret-string:json-key:version-stage:version-id");
 * </pre></blockquote>
 * <p>
 * <h3>Template Options &amp; Transform</h3>
 * <p>
 * CloudFormation templates support a number of options, including which Macros or
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html">Transforms</a> to use when deploying the stack. Those can be
 * configured using the <code>stack.templateOptions</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Stack stack = new Stack(app, "StackName");
 * 
 * stack.templateOptions.getDescription() = "This will appear in the AWS console";
 * stack.templateOptions.getTransforms() = asList("AWS::Serverless-2016-10-31");
 * stack.templateOptions.getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * <h3>Emitting Raw Resources</h3>
 * <p>
 * The <code>CfnResource</code> class allows emitting arbitrary entries in the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html">Resources</a> section of the CloudFormation template.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CfnResource(this, "ResourceId", new CfnResourceProps()
 *         .type("AWS::S3::Bucket")
 *         .properties(Map.of(
 *                 "BucketName", "bucket-name")));
 * </pre></blockquote>
 * <p>
 * As for any other resource, the logical ID in the CloudFormation template will be
 * generated by the AWS CDK, but the type and properties will be copied verbatim in
 * the synthesized template.
 * <p>
 * <h3>Including raw CloudFormation template fragments</h3>
 * <p>
 * When migrating a CloudFormation stack to the AWS CDK, it can be useful to
 * include fragments of an existing template verbatim in the synthesized template.
 * This can be achieved using the <code>CfnInclude</code> class.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * new CfnInclude(this, "ID", new CfnIncludeProps()
 *         .template(Map.of(
 *                 "Resources", Map.of(
 *                         "Bucket", Map.of(
 *                                 "Type", "AWS::S3::Bucket",
 *                                 "Properties", Map.of(
 *                                         "BucketName", "my-shiny-bucket"))))));
 * </pre></blockquote>
 * <p>
 * <h3>Termination Protection</h3>
 * <p>
 * You can prevent a stack from being accidentally deleted by enabling termination
 * protection on the stack. If a user attempts to delete a stack with termination
 * protection enabled, the deletion fails and the stack--including its status--remains
 * unchanged. Enabling or disabling termination protection on a stack sets it for any
 * nested stacks belonging to that stack as well. You can enable termination protection
 * on a stack by setting the <code>terminationProtection</code> prop to <code>true</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Stack stack = new Stack(app, "StackName", new StackProps()
 *         .terminationProtection(true));
 * </pre></blockquote>
 * <p>
 * By default, termination protection is disabled.
 * <p>
 * <h3>CfnJson</h3>
 * <p>
 * <code>CfnJson</code> allows you to postpone the resolution of a JSON blob from
 * deployment-time. This is useful in cases where the CloudFormation JSON template
 * cannot express a certain value.
 * <p>
 * A common example is to use <code>CfnJson</code> in order to render a JSON map which needs
 * to use intrinsic functions in keys. Since JSON map keys must be strings, it is
 * impossible to use intrinsics in keys and <code>CfnJson</code> can help.
 * <p>
 * The following example defines an IAM role which can only be assumed by
 * principals that are tagged with a specific tag.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * CfnParameter tagParam = new CfnParameter(this, "TagName");
 * 
 * CfnJson stringEquals = new CfnJson(this, "ConditionJson", new CfnJsonProps()
 *         .value(Map.of(
 *                 String.format("aws:PrincipalTag/%s", tagParam.getValueAsString()), true)));
 * 
 * IPrincipal principal = new iam.AccountRootPrincipal().withConditions(Map.of(
 *         "StringEquals", stringEquals));
 * 
 * new Role(this, "MyRole", new RoleProps().assumedBy(principal));
 * </pre></blockquote>
 * <p>
 * <strong>Explanation</strong>: since in this example we pass the tag name through a parameter, it
 * can only be resolved during deployment. The resolved value can be represented in
 * the template through a <code>{ "Ref": "TagName" }</code>. However, since we want to use
 * this value inside a <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag"><code>aws:PrincipalTag/TAG-NAME</code></a>
 * IAM operator, we need it in the <em>key</em> of a <code>StringEquals</code> condition. JSON keys
 * <em>must be</em> strings, so to circumvent this limitation, we use <code>CfnJson</code>
 * to "delay" the rendition of this template section to deploy-time. This means
 * that the value of <code>StringEquals</code> in the template will be <code>{ "Fn::GetAtt": [ "ConditionJson", "Value" ] }</code>, and will only "expand" to the operator we synthesized during deployment.
 * <p>
 * <h3>Stack Resource Limit</h3>
 * <p>
 * When deploying to AWS CloudFormation, it needs to keep in check the amount of resources being added inside a Stack. Currently it's possible to check the limits in the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html">AWS CloudFormation quotas</a> page.
 * <p>
 * It's possible to synthesize the project with more Resources than the allowed (or even reduce the number of Resources).
 * <p>
 * Set the context key <code>&#64;aws-cdk/core:stackResourceLimit</code> with the proper value, being 0 for disable the limit of resources.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.core;
