/**
 * <h2>AWS Cloud Development Kit Core Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This library includes the basic building blocks of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> (AWS CDK). It defines the core classes that are used in the rest of the
 * AWS Construct Library.
 * <p>
 * See the <a href="https://docs.aws.amazon.com/cdk/latest/guide/home.html">AWS CDK Developer
 * Guide</a> for
 * information of most of the capabilities of this library. The rest of this
 * README will only cover topics not already covered in the Developer Guide.
 * <p>
 * <h2>Durations</h2>
 * <p>
 * To make specifications of time intervals unambiguous, a single class called
 * <code>Duration</code> is used throughout the AWS Construct Library by all constructs
 * that that take a time interval as a parameter (be it for a timeout, a
 * rate, or something else).
 * <p>
 * An instance of Duration is constructed by using one of the static factory
 * methods on it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Duration.seconds(300);// 5 minutes
 * Duration.minutes(5);// 5 minutes
 * Duration.hours(1);// 1 hour
 * Duration.days(7);// 7 days
 * Duration.parse("PT5M");
 * </pre></blockquote>
 * <p>
 * <h2>Size (Digital Information Quantity)</h2>
 * <p>
 * To make specification of digital storage quantities unambiguous, a class called
 * <code>Size</code> is available.
 * <p>
 * An instance of <code>Size</code> is initialized through one of its static factory methods:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Size.kibibytes(200);// 200 KiB
 * Size.mebibytes(5);// 5 MiB
 * Size.gibibytes(40);// 40 GiB
 * Size.tebibytes(200);// 200 TiB
 * Size.pebibytes(3);
 * </pre></blockquote>
 * <p>
 * Instances of <code>Size</code> created with one of the units can be converted into others.
 * By default, conversion to a higher unit will fail if the conversion does not produce
 * a whole number. This can be overridden by unsetting <code>integral</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Size.mebibytes(2).toKibibytes();// yields 2048
 * Size.kibibytes(2050).toMebibyte(Map.of("integral", false));
 * </pre></blockquote>
 * <p>
 * <h2>Secrets</h2>
 * <p>
 * To help avoid accidental storage of secrets as plain text, we use the <code>SecretValue</code> type to
 * represent secrets. Any construct that takes a value that should be a secret (such as
 * a password or an access key) will take a parameter of type <code>SecretValue</code>.
 * <p>
 * The best practice is to store secrets in AWS Secrets Manager and reference them using <code>SecretValue.secretsManager</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var secret = SecretValue.secretsManager("secretId", Map.of(
 *         "jsonField", "password", // optional: key of a JSON field to retrieve (defaults to all content),
 *         "versionId", "id", // optional: id of the version (default AWSCURRENT)
 *         "versionStage", "stage"));
 * </pre></blockquote>
 * <p>
 * Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
 * <code>SecretValue</code> also supports the following secret sources:
 * <p>
 * <ul>
 * <li><code>SecretValue.plainText(secret)</code>: stores the secret as plain text in your app and the resulting template (not recommended).</li>
 * <li><code>SecretValue.ssmSecure(param, version)</code>: refers to a secret stored as a SecureString in the SSM Parameter Store.</li>
 * <li><code>SecretValue.cfnParameter(param)</code>: refers to a secret passed through a CloudFormation parameter (must have <code>NoEcho: true</code>).</li>
 * <li><code>SecretValue.cfnDynamicReference(dynref)</code>: refers to a secret described by a CloudFormation dynamic reference (used by <code>ssmSecure</code> and <code>secretsManager</code>).</li>
 * </ul>
 * <p>
 * <h2>ARN manipulation</h2>
 * <p>
 * Sometimes you will need to put together or pick apart Amazon Resource Names
 * (ARNs). The functions <code>stack.formatArn()</code> and <code>stack.parseArn()</code> exist for
 * this purpose.
 * <p>
 * <code>formatArn()</code> can be used to build an ARN from components. It will automatically
 * use the region and account of the stack you're calling it on:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Builds "arn:&lt;PARTITION&gt;:lambda:&lt;REGION&gt;:&lt;ACCOUNT&gt;:function:MyFunction"
 * stack.formatArn(Map.of(
 *         "service", "lambda",
 *         "resource", "function",
 *         "sep", ":",
 *         "resourceName", "MyFunction"));
 * </pre></blockquote>
 * <p>
 * <code>parseArn()</code> can be used to get a single component from an ARN. <code>parseArn()</code>
 * will correctly deal with both literal ARNs and deploy-time values (tokens),
 * but in case of a deploy-time value be aware that the result will be another
 * deploy-time value which cannot be inspected in the CDK application.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Extracts the function name out of an AWS Lambda Function ARN
 * var arnComponents = stack.parseArn(arn, ":");
 * var functionName = arnComponents.getResourceName();
 * </pre></blockquote>
 * <p>
 * Note that depending on the service, the resource separator can be either
 * <code>:</code> or <code>/</code>, and the resource name can be either the 6th or 7th
 * component in the ARN. When using these functions, you will need to know
 * the format of the ARN you are dealing with.
 * <p>
 * For an exhaustive list of ARN formats used in AWS, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">AWS ARNs and
 * Namespaces</a>
 * in the AWS General Reference.
 * <p>
 * <h2>Dependencies</h2>
 * <p>
 * <h3>Construct Dependencies</h3>
 * <p>
 * Sometimes AWS resources depend on other resources, and the creation of one
 * resource must be completed before the next one can be started.
 * <p>
 * In general, CloudFormation will correctly infer the dependency relationship
 * between resources based on the property values that are used. In the cases where
 * it doesn't, the AWS Construct Library will add the dependency relationship for
 * you.
 * <p>
 * If you need to add an ordering dependency that is not automatically inferred,
 * you do so by adding a dependency relationship using
 * <code>constructA.node.addDependency(constructB)</code>. This will add a dependency
 * relationship between all resources in the scope of <code>constructA</code> and all
 * resources in the scope of <code>constructB</code>.
 * <p>
 * If you want a single object to represent a set of constructs that are not
 * necessarily in the same scope, you can use a <code>ConcreteDependable</code>. The
 * following creates a single object that represents a dependency on two
 * construts, <code>constructB</code> and <code>constructC</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Declare the dependable object
 * var bAndC = new ConcreteDependable();
 * bAndC.add(constructB);
 * bAndC.add(constructC);
 * 
 * // Take the dependency
 * constructA.node.addDependency(bAndC);
 * </pre></blockquote>
 * <p>
 * <h3>Stack Dependencies</h3>
 * <p>
 * Two different stack instances can have a dependency on one another. This
 * happens when an resource from one stack is referenced in another stack. In
 * that case, CDK records the cross-stack referencing of resources,
 * automatically produces the right CloudFormation primitives, and adds a
 * dependency between the two stacks. You can also manually add a dependency
 * between two stacks by using the <code>stackA.addDependency(stackB)</code> method.
 * <p>
 * A stack dependency has the following implications:
 * <p>
 * <ul>
 * <li>Cyclic dependencies are not allowed, so if <code>stackA</code> is using resources from
 * <code>stackB</code>, the reverse is not possible anymore.</li>
 * <li>Stacks with dependencies between them are treated specially by the CDK
 * toolkit:
 * <p>
 * <ul>
 * <li>If <code>stackA</code> depends on <code>stackB</code>, running <code>cdk deploy stackA</code> will also
 * automatically deploy <code>stackB</code>.</li>
 * <li><code>stackB</code>'s deployment will be performed <em>before</em> <code>stackA</code>'s deployment.</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>AWS CloudFormation features</h2>
 * <p>
 * A CDK stack synthesizes to an AWS CloudFormation Template. This section
 * explains how this module allows users to access low-level CloudFormation
 * features when needed.
 * <p>
 * <h3>Stack Outputs</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">stack outputs</a> and exports are created using
 * the <code>CfnOutput</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnOutput.Builder.create(this, "OutputName")
 *         .value(bucket.getBucketName())
 *         .description("The name of an S3 bucket")// Optional
 *         .exportName("TheAwesomeBucket")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Parameters</h3>
 * <p>
 * CloudFormation templates support the use of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a> to
 * customize a template. They enable CloudFormation users to input custom values to
 * a template each time a stack is created or updated. While the CDK design
 * philosophy favors using build-time parameterization, users may need to use
 * CloudFormation in a number of cases (for example, when migrating an existing
 * stack to the AWS CDK).
 * <p>
 * Template parameters can be added to a stack by using the <code>CfnParameter</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnParameter.Builder.create(this, "MyParameter")
 *         .type("Number")
 *         .default(1337)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The value of parameters can then be obtained using one of the <code>value</code> methods.
 * As parameters are only resolved at deployment time, the values obtained are
 * placeholder tokens for the real value (<code>Token.isUnresolved()</code> would return <code>true</code>
 * for those):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var param = CfnParameter.Builder.create(this, "ParameterName").build();
 * 
 * // If the parameter is a String
 * param.getValueAsString();
 * 
 * // If the parameter is a Number
 * param.getValueAsNumber();
 * 
 * // If the parameter is a List
 * param.getValueAsList();
 * </pre></blockquote>
 * <p>
 * <h3>Pseudo Parameters</h3>
 * <p>
 * CloudFormation supports a number of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/pseudo-parameter-reference.html">pseudo parameters</a>,
 * which resolve to useful values at deployment time. CloudFormation pseudo
 * parameters can be obtained from static members of the <code>Aws</code> class.
 * <p>
 * It is generally recommended to access pseudo parameters from the scope's <code>stack</code>
 * instead, which guarantees the values produced are qualifying the designated
 * stack, which is essential in cases where resources are shared cross-stack:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // "this" is the current construct
 * var stack = Stack.of(this);
 * 
 * stack.getAccount();// Returns the AWS::AccountId for this stack (or the literal value if known)
 * stack.getRegion();// Returns the AWS::Region for this stack (or the literal value if known)
 * stack.getPartition();
 * </pre></blockquote>
 * <p>
 * <h3>Resource Options</h3>
 * <p>
 * CloudFormation resources can also specify <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-product-attribute-reference.html">resource
 * attributes</a>. The <code>CfnResource</code> class allows
 * accessing those through the <code>cfnOptions</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var rawBucket = CfnBucket.Builder.create(this, "Bucket").build();
 * // -or-
 * var rawBucket = (Object)bucket.node.getDefaultChild();
 * 
 * // then
 * rawBucket.cfnOptions.getCondition() = CfnCondition.Builder.create(this, "EnableBucket").build();
 * rawBucket.cfnOptions.getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * Resource dependencies (the <code>DependsOn</code> attribute) is modified using the
 * <code>cfnResource.addDependsOn</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var resourceA = CfnResource.Builder.create(this, "ResourceA").build();
 * var resourceB = CfnResource.Builder.create(this, "ResourceB").build();
 * 
 * resourceB.addDependsOn(resourceA);
 * </pre></blockquote>
 * <p>
 * <h3>Intrinsic Functions and Condition Expressions</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html">intrinsic functions</a>. These functions
 * can be accessed from the <code>Fn</code> class, which provides type-safe methods for each
 * intrinsic function as well as condition expressions:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // To use Fn::Base64
 * Fn.base64("SGVsbG8gQ0RLIQo=");
 * 
 * // To compose condition expressions:
 * var environmentParameter = new CfnParameter(this, "Environment");
 * Fn.conditionAnd(Fn.conditionEquals("Production", environmentParameter), Fn.conditionNot(Fn.conditionEquals("us-east-1", Aws.getREGION())));
 * </pre></blockquote>
 * <p>
 * When working with deploy-time values (those for which <code>Token.isUnresolved</code>
 * returns <code>true</code>), idiomatic conditionals from the programming language cannot be
 * used (the value will not be known until deployment time). When conditional logic
 * needs to be expressed with un-resolved values, it is necessary to use
 * CloudFormation conditions by means of the <code>CfnCondition</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var environmentParameter = new CfnParameter(this, "Environment");
 * var isProd = CfnCondition.Builder.create(this, "IsProduction")
 *         .expression(Fn.conditionEquals("Production", environmentParameter))
 *         .build();
 * 
 * // Configuration value that is a different string based on IsProduction
 * var stage = Fn.conditionIf(isProd.logicalId, 'Beta', 'Prod').toString();
 * 
 * // Make Bucket creation condition to IsProduction by accessing
 * // and overriding the CloudFormation resource
 * var bucket = new Bucket(this, "Bucket");
 * var cfnBucket = (Object)bucket.node.getDefaultChild();
 * cfnBucket.cfnOptions.getCondition() = isProd;
 * </pre></blockquote>
 * <p>
 * <h3>Mappings</h3>
 * <p>
 * CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">mappings</a> are created and queried using the
 * <code>CfnMappings</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var mapping = CfnMapping.Builder.create(this, "MappingTable")
 *         .mapping(Map.of(
 *                 "regionName", Map.of(
 *                         "us-east-1", "US East (N. Virginia)",
 *                         "us-east-2", "US East (Ohio)")))
 *         .build();
 * 
 * mapping.findInMap("regionName", Aws.getREGION());
 * </pre></blockquote>
 * <p>
 * <h3>Dynamic References</h3>
 * <p>
 * CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">dynamically resolving</a> values
 * for SSM parameters (including secure strings) and Secrets Manager. Encoding such
 * references is done using the <code>CfnDynamicReference</code> class:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnDynamicReference.Builder.create(this, "SecureStringValue")
 *         .service(CfnDynamicReferenceService.getSECRETS_MANAGER())
 *         .referenceKey("secret-id:secret-string:json-key:version-stage:version-id")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Template Options &amp; Transform</h3>
 * <p>
 * CloudFormation templates support a number of options, including which Macros or
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html">Transforms</a> to use when deploying the stack. Those can be
 * configured using the <code>stack.templateOptions</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var stack = new Stack(app, "StackName");
 * 
 * stack.templateOptions.getDescription() = "This will appear in the AWS console";
 * stack.templateOptions.getTransforms() = asList("AWS::Serverless-2016-10-31");
 * stack.templateOptions.getMetadata() = Map.of(
 *         "metadataKey", "MetadataValue");
 * </pre></blockquote>
 * <p>
 * <h3>Emitting Raw Resources</h3>
 * <p>
 * The <code>CfnResource</code> class allows emitting arbitrary entries in the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html">Resources</a> section of the CloudFormation template.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnResource.Builder.create(this, "ResourceId")
 *         .type("AWS::S3::Bucket")
 *         .properties(Map.of(
 *                 "BucketName", "bucket-name"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * As for any other resource, the logical ID in the CloudFormation template will be
 * generated by the AWS CDK, but the type and properties will be copied verbatim in
 * the synthesized template.
 * <p>
 * <h3>Including raw CloudFormation template fragments</h3>
 * <p>
 * When migrating a CloudFormation stack to the AWS CDK, it can be useful to
 * include fragments of an existing template verbatim in the synthesized template.
 * This can be achieved using the <code>CfnInclude</code> class.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnInclude.Builder.create(this, "ID")
 *         .template(Map.of(
 *                 "Resources", Map.of(
 *                         "Bucket", Map.of(
 *                                 "Type", "AWS::S3::Bucket",
 *                                 "Properties", Map.of(
 *                                         "BucketName", "my-shiny-bucket")))))
 *         .build();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.core;
