/**
 * <h2>AWS Cloud Development Kit Core Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>This library includes the basic building blocks of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> (AWS CDK). It defines the core classes that are used in the rest of the
 * AWS Construct Library.</p>
 * <p>See the <a href="https://docs.aws.amazon.com/cdk/latest/guide/home.html">AWS CDK Developer
 * Guide</a> for
 * information of most of the capabilities of this library. The rest of this
 * README will only cover topics not already covered in the Developer Guide.</p>
 * <h2>Durations</h2>
 * <p>To make specifications of time intervals unambiguous, a single class called
 * <code>Duration</code> is used throughout the AWS Construct Library by all constructs
 * that that take a time interval as a parameter (be it for a timeout, a
 * rate, or something else).</p>
 * <p>An instance of Duration is constructed by using one of the static factory
 * methods on it:</p>
 * <pre><code class="language-ts">// This example is in TypeScript, examples in Java are coming soon.
 * Duration.seconds(300)   // 5 minutes
 * Duration.minutes(5)     // 5 minutes
 * Duration.hours(1)       // 1 hour
 * Duration.days(7)        // 7 days
 * Duration.parse('PT5M')  // 5 minutes
 * </code></pre>
 * <h2>Secrets</h2>
 * <p>To help avoid accidental storage of secrets as plain text, we use the <code>SecretValue</code> type to
 * represent secrets. Any construct that takes a value that should be a secret (such as
 * a password or an access key) will take a parameter of type <code>SecretValue</code>.</p>
 * <p>The best practice is to store secrets in AWS Secrets Manager and reference them using <code>SecretValue.secretsManager</code>:</p>
 * <pre><code class="language-ts">const secret = SecretValue.secretsManager('secretId', {
 *   jsonField: 'password' // optional: key of a JSON field to retrieve (defaults to all content),
 *   versionId: 'id'       // optional: id of the version (default AWSCURRENT)
 *   versionStage: 'stage' // optional: version stage name (default AWSCURRENT)
 * });
 * </code></pre>
 * <p>Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
 * <code>SecretValue</code> also supports the following secret sources:</p>
 * <ul>
 * <li><code>SecretValue.plainText(secret)</code>: stores the secret as plain text in your app and the resulting template (not recommended).</li>
 * <li><code>SecretValue.ssmSecure(param, version)</code>: refers to a secret stored as a SecureString in the SSM Parameter Store.</li>
 * <li><code>SecretValue.cfnParameter(param)</code>: refers to a secret passed through a CloudFormation parameter (must have <code>NoEcho: true</code>).</li>
 * <li><code>SecretValue.cfnDynamicReference(dynref)</code>: refers to a secret described by a CloudFormation dynamic reference (used by <code>ssmSecure</code> and <code>secretsManager</code>).</li>
 * </ul>
 * <h2>ARN manipulation</h2>
 * <p>Sometimes you will need to put together or pick apart Amazon Resource Names
 * (ARNs). The functions <code>stack.formatArn()</code> and <code>stack.parseArn()</code> exist for
 * this purpose.</p>
 * <p><code>formatArn()</code> can be used to build an ARN from components. It will automatically
 * use the region and account of the stack you're calling it on:</p>
 * <pre><code class="language-ts">// Builds &quot;arn:&lt;PARTITION&gt;:lambda:&lt;REGION&gt;:&lt;ACCOUNT&gt;:function:MyFunction&quot;
 * stack.formatArn({
 *   service: 'lambda',
 *   resource: 'function',
 *   sep: ':',
 *   resourceName: 'MyFunction'
 * });
 * </code></pre>
 * <p><code>parseArn()</code> can be used to get a single component from an ARN. <code>parseArn()</code>
 * will correctly deal with both literal ARNs and deploy-time values (tokens),
 * but in case of a deploy-time value be aware that the result will be another
 * deploy-time value which cannot be inspected in the CDK application.</p>
 * <pre><code class="language-ts">// Extracts the function name out of an AWS Lambda Function ARN
 * const arnComponents = stack.parseArn(arn, ':');
 * const functionName = arnComponents.resourceName;
 * </code></pre>
 * <p>Note that depending on the service, the resource separator can be either
 * <code>:</code> or <code>/</code>, and the resource name can be either the 6th or 7th
 * component in the ARN. When using these functions, you will need to know
 * the format of the ARN you are dealing with.</p>
 * <p>For an exhaustive list of ARN formats used in AWS, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">AWS ARNs and
 * Namespaces</a>
 * in the AWS General Reference.</p>
 * <h2>Dependencies</h2>
 * <h3>Construct Dependencies</h3>
 * <p>Sometimes AWS resources depend on other resources, and the creation of one
 * resource must be completed before the next one can be started.</p>
 * <p>In general, CloudFormation will correctly infer the dependency relationship
 * between resources based on the property values that are used. In the cases where
 * it doesn't, the AWS Construct Library will add the dependency relationship for
 * you.</p>
 * <p>If you need to add an ordering dependency that is not automatically inferred,
 * you do so by adding a dependency relationship using
 * <code>constructA.node.addDependency(constructB)</code>. This will add a dependency
 * relationship between all resources in the scope of <code>constructA</code> and all
 * resources in the scope of <code>constructB</code>.</p>
 * <p>If you want a single object to represent a set of constructs that are not
 * necessarily in the same scope, you can use a <code>ConcreteDependable</code>. The
 * following creates a single object that represents a dependency on two
 * construts, <code>constructB</code> and <code>constructC</code>:</p>
 * <pre><code class="language-ts">// Declare the dependable object
 * const bAndC = new ConcreteDependable();
 * bAndC.add(constructB);
 * bAndC.add(constructC);
 * 
 * // Take the dependency
 * constructA.node.addDependency(bAndC);
 * </code></pre>
 * <h3>Stack Dependencies</h3>
 * <p>Two different stack instances can have a dependency on one another. This
 * happens when an resource from one stack is referenced in another stack. In
 * that case, CDK records the cross-stack referencing of resources,
 * automatically produces the right CloudFormation primitives, and adds a
 * dependency between the two stacks. You can also manually add a dependency
 * between two stacks by using the <code>stackA.addDependency(stackB)</code> method.</p>
 * <p>A stack dependency has the following implications:</p>
 * <ul>
 * <li>Cyclic dependencies are not allowed, so if <code>stackA</code> is using resources from
 * <code>stackB</code>, the reverse is not possible anymore.</li>
 * <li>Stacks with dependencies between them are treated specially by the CDK
 * toolkit:
 * <ul>
 * <li>If <code>stackA</code> depends on <code>stackB</code>, running <code>cdk deploy stackA</code> will also
 * automatically deploy <code>stackB</code>.</li>
 * <li><code>stackB</code>'s deployment will be performed <em>before</em> <code>stackA</code>'s deployment.</li>
 * </ul>
 * </li>
 * </ul>
 * <h2>AWS CloudFormation features</h2>
 * <p>A CDK stack synthesizes to an AWS CloudFormation Template. This section
 * explains how this module allows users to access low-level CloudFormation
 * features when needed.</p>
 * <h3>Stack Outputs</h3>
 * <p>CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">stack outputs</a> and exports are created using
 * the <code>CfnOutput</code> class:</p>
 * <pre><code class="language-ts">new CfnOutput(this, 'OutputName', {
 *   value: bucket.bucketName,
 *   description: 'The name of an S3 bucket', // Optional
 *   exportName: 'Global.BucketName', // Registers a CloudFormation export
 * });
 * </code></pre>
 * <h3>Parameters</h3>
 * <p>CloudFormation templates support the use of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a> to
 * customize a template. They enable CloudFormation users to input custom values to
 * a template each time a stack is created or updated. While the CDK design
 * philosophy favors using build-time parameterization, users may need to use
 * CloudFormation in a number of cases (for example, when migrating an existing
 * stack to the AWS CDK).</p>
 * <p>Template parameters can be added to a stack by using the <code>CfnParameter</code> class:</p>
 * <pre><code class="language-ts">new CfnParameter(this, 'MyParameter', {
 *   type: 'Number',
 *   default: 1337,
 *   // See the API reference for more configuration props
 * });
 * </code></pre>
 * <p>The value of parameters can then be obtained using one of the <code>value</code> methods.
 * As parameters are only resolved at deployment time, the values obtained are
 * placeholder tokens for the real value (<code>Token.isUnresolved()</code> would return <code>true</code>
 * for those):</p>
 * <pre><code class="language-ts">const param = new CfnParameter(this, 'ParameterName', { /* config *{@literal /} });
 * 
 * // If the parameter is a String
 * param.valueAsString;
 * 
 * // If the parameter is a Number
 * param.valueAsNumber;
 * 
 * // If the parameter is a List
 * param.valueAsList;
 * </code></pre>
 * <h3>Pseudo Parameters</h3>
 * <p>CloudFormation supports a number of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/pseudo-parameter-reference.html">pseudo parameters</a>,
 * which resolve to useful values at deployment time. CloudFormation pseudo
 * parameters can be obtained from static members of the <code>Aws</code> class.</p>
 * <p>It is generally recommended to access pseudo parameters from the scope's <code>stack</code>
 * instead, which guarantees the values produced are qualifying the designated
 * stack, which is essential in cases where resources are shared cross-stack:</p>
 * <pre><code class="language-ts">// &quot;this&quot; is the current construct
 * const stack = Stack.of(this);
 * 
 * stack.account; // Returns the AWS::AccountId for this stack (or the literal value if known)
 * stack.region;  // Returns the AWS::Region for this stack (or the literal value if known)
 * stack.partition;
 * </code></pre>
 * <h3>Resource Options</h3>
 * <p>CloudFormation resources can also specify <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-product-attribute-reference.html">resource
 * attributes</a>. The <code>CfnResource</code> class allows
 * accessing those though the <code>cfnOptions</code> property:</p>
 * <pre><code class="language-ts">const rawBucket = new s3.CfnBucket(this, 'Bucket', { /* ... *{@literal /} });
 * // -or-
 * const rawBucket = bucket.node.defaultChild as s3.CfnBucket;
 * 
 * // then
 * rawBucket.condition = new CfnCondition(this, 'EnableBucket', { /* ... *{@literal /} });
 * rawBucket.cfnOptions.metadata = {
 *   metadataKey: 'MetadataValue',
 * };
 * </code></pre>
 * <p>Resource dependencies (the <code>DependsOn</code> attribute) is modified using the
 * <code>cfnResource.addDependsOn</code> method:</p>
 * <pre><code class="language-ts">const resourceA = new CfnResource(this, 'ResourceA', { /* ... *{@literal /} });
 * const resourceB = new CfnResource(this, 'ResourceB', { /* ... *{@literal /} });
 * 
 * resourceB.addDependsOn(resourceA);
 * </code></pre>
 * <h3>Intrinsic Functions and Condition Expressions</h3>
 * <p>CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html">intrinsic functions</a>. These functions
 * can be accessed from the <code>Fn</code> class, which provides type-safe methods for each
 * intrinsic function as well as condition expressions:</p>
 * <pre><code class="language-ts">// To use Fn::Base64
 * Fn.base64('SGVsbG8gQ0RLIQo=');
 * 
 * // To compose condition expressions:
 * const environmentParameter = new CfnParameter(this, 'Environment');
 * Fn.conditionAnd(
 *   // The &quot;Environment&quot; CloudFormation template parameter evaluates to &quot;Production&quot;
 *   Fn.conditionEquals('Production', environmentParameter),
 *   // The AWS::Region pseudo-parameter value is NOT equal to &quot;us-east-1&quot;
 *   Fn.conditionNot(Fn.conditionEquals('us-east-1', Aws.REGION)),
 * );
 * </code></pre>
 * <p>When working with deploy-time values (those for which <code>Token.isUnresolved</code>
 * returns <code>true</code>), idiomatic conditionals from the programming language cannot be
 * used (the value will not be known until deployment time). When conditional logic
 * needs to be expressed with un-resolved values, it is necessary to use
 * CloudFormation conditions by means of the <code>CfnCondition</code> class:</p>
 * <pre><code class="language-ts">const environmentParameter = new CfnParameter(this, 'Environment');
 * const isProd = new CfnCondition(this, 'IsProduction', {
 *   expression: Fn.conditionEquals('Production', environmentParameter),
 * });
 * 
 * // Configuration value that is a different string based on IsProduction
 * const stage = Fn.conditionIf(isProd.logicalId, 'Beta', 'Prod').toString();
 * 
 * // Make Bucket creation condition to IsProduction by accessing
 * // and overriding the CloudFormation resource
 * const bucket = new s3.Bucket(this, 'Bucket');
 * const cfnBucket = bucket.node.defaultChild as s3.CfnBucket;
 * cfnBucket.cfnOptions.condition = isProd;
 * </code></pre>
 * <h3>Mappings</h3>
 * <p>CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">mappings</a> are created and queried using the
 * <code>CfnMappings</code> class:</p>
 * <pre><code class="language-ts">const mapping = new CfnMapping(this, 'MappingTable', {
 *   mapping: {
 *     regionName: {
 *       'us-east-1': 'US East (N. Virginia)',
 *       'us-east-2': 'US East (Ohio)',
 *       // ...
 *     },
 *     // ...
 *   }
 * });
 * 
 * mapping.findInMap('regionName', Aws.REGION);
 * </code></pre>
 * <h3>Dynamic References</h3>
 * <p>CloudFormation supports <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">dynamically resolving</a> values
 * for SSM parameters (including secure strings) and Secrets Manager. Encoding such
 * references is done using the <code>CfnDynamicReference</code> class:</p>
 * <pre><code class="language-ts">new CfnDynamicReference(this, 'SecureStringValue', {
 *   service: CfnDynamicReferenceService.SECRETS_MANAGER,
 *   referenceKey: 'secret-id:secret-string:json-key:version-stage:version-id',
 * });
 * </code></pre>
 * <h3>Template Options &amp; Transform</h3>
 * <p>CloudFormation templates support a number of options, including which Macros or
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html">Transforms</a> to use when deploying the stack. Those can be
 * configured using the <code>stack.templateOptions</code> property:</p>
 * <pre><code class="language-ts">const stack = new Stack(app, 'StackName');
 * 
 * stack.templateOptions.description = 'This will appear in the AWS console';
 * stack.templateOptions.transform = 'AWS::Serverless';
 * stack.templateOptions.metadata = {
 *   metadataKey: 'MetadataValue',
 * };
 * </code></pre>
 * <h3>Emitting Raw Resources</h3>
 * <p>The <code>CfnResource</code> class allows emitting arbitrary entries in the
 * [Resources][cfn-resources] section of the CloudFormation template.</p>
 * <pre><code class="language-ts">new CfnResource(this, 'ResourceId', {
 *   type: 'AWS::S3::Bucket',
 *   properties: {
 *     BucketName: 'bucket-name'
 *   },
 * });
 * </code></pre>
 * <p>As for any other resource, the logical ID in the CloudFormation template will be
 * generated by the AWS CDK, but the type and properties will be copied verbatim in
 * the synthesized template.</p>
 * <h3>Including raw CloudFormation template fragments</h3>
 * <p>When migrating a CloudFormation stack to the AWS CDK, it can be useful to
 * include fragments of an existing template verbatim in the synthesized template.
 * This can be achieved using the <code>CfnInclude</code> class.</p>
 * <pre><code class="language-ts">new CfnInclude(this, 'ID', {
 *   template: {
 *     Resources: {
 *       Bucket: {
 *         Type: 'AWS::S3::Bucket',
 *         Properties: {
 *           BucketName: 'my-shiny-bucket'
 *         }
 *       }
 *     }
 *   },
 * });
 * </code></pre>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.core;
