/**
 * <h2>AWS Cloud Development Kit Core Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p><strong>This is a <em>developer preview</em> (public beta) module. Releases might lack important features and might have
 * future breaking changes.</strong></p>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>This library includes the basic building blocks of
 * the <a href="https://github.com/awslabs/aws-cdk">AWS Cloud Development Kit</a> (AWS CDK).</p>
 * <h2>Aspects</h2>
 * <p>Aspects are a mechanism to extend the CDK without having to directly impact the
 * class hierarchy. We have implemented aspects using the <a href="https://en.wikipedia.org/wiki/Visitor_pattern">Visitor
 * Pattern</a>.</p>
 * <p>An aspect in the CDK is defined by this <a href="lib/aspect.ts">interface</a></p>
 * <p>Aspects can be applied to any construct. During the tree
 * &quot;prepare&quot; phase the aspect will visit each construct in the tree once.
 * Aspects are invoked in the order they were added to the construct. They
 * traverse the construct tree in a breadth first order starting at the <code>App</code>
 * ending at the leaf nodes (most commonly the CloudFormation Resource). Aspect
 * authors implement the <code>visit(IConstruct)</code> function and can inspect the
 * <code>Construct</code> for specific characteristics. Such as, is this construct a
 * CloudFormation Resource?</p>
 * <h2>Tagging</h2>
 * <p>Tags are implemented using aspects.</p>
 * <p>Tags can be applied to any construct. Tags are inherited, based on the scope. If
 * you tag construct A, and A contains construct B, construct B inherits the tag.
 * The Tag API supports:</p>
 * <ul>
 * <li><code>Tag</code> add (apply) a tag, either to specific resources or all but specific resources</li>
 * <li><code>RemoveTag</code> remove a tag, again either from specific resources or all but specific resources</li>
 * </ul>
 * <p>A simple example, if you create a stack and want anything in the stack to receive a
 * tag:</p>
 * <pre><code class="language-ts">import cdk = require('@aws-cdk/core');
 * 
 * const app = new cdk.App();
 * const theBestStack = new cdk.Stack(app, 'MarketingSystem');
 * theBestStack.node.apply(new cdk.Tag('StackType', 'TheBest'));
 * 
 * // any resources added that support tags will get them
 * </code></pre>
 * <blockquote>
 * <p>The goal was to enable the ability to define tags in one place and have them
 * applied consistently for all resources that support tagging. In addition
 * the developer should not have to know if the resource supports tags. The
 * developer defines the tagging intents for all resources within a path.
 * If the resources support tags they are added, else no action is taken.</p>
 * </blockquote>
 * <h3>Tag Example with ECS</h3>
 * <p>We are going to use the <a href="https://awslabs.github.io/aws-cdk/ecs_example.html">ECS example</a> as starting point.</p>
 * <p>For the purposes of example, this ECS cluster is for the Marketing Department.
 * Marketing has two core groups Business to Business (B2B) and Business to Consumer
 * (B2C). However, the Marketing team relies on the Platform team to help build the
 * common components across businesses and separates costs to match. The goal here
 * is tag the Platform team resources, the Marketing Department and then Marketing
 * groups to enable proper cost allocations.</p>
 * <p>We have modified the example and the code is located:
 * examples/cdk-examples-typescript/hello-cdk-ecs-tags</p>
 * <p>When the example is run the following tags are created:</p>
 * <blockquote>
 * <p>We are omitting the default tags for VPC components.</p>
 * </blockquote>
 * <p>| Construct Path | Tag Key | Tag Value |
 * | ----------|:---------|:-----|
 * |MarketingSystem/MarketingVpc|CostCenter|Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet1| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet1/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet1/NATGateway| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet2| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet2/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet2/NATGateway| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet3| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet3/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PublicSubnet3/NATGateway| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet1| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet1/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet2| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet2/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet3| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/PrivateSubnet3/RouteTable| CostCenter | Platform|
 * |MarketingSystem/MarketingVpc/IGW|CostCenter|Platform|
 * |MarketingSystem/B2BService/Service/SecurityGroup/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2BService/LB/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2BService/LB/SecurityGroup/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2BService/LB/PublicListener/ECSGroup/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2CService/Service/SecurityGroup/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2CService/LB/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2CService/LB/SecurityGroup/Resource|CostCenter|Marketing|
 * |MarketingSystem/B2CService/LB/PublicListener/ECSGroup/Resource|CostCenter|Marketing|</p>
 * <p>As you can see many tags are generated with only a few intent based directives. The CDK does default some additional tags for suggested <code>Name</code> keys. If you want to remove those tags you can do so by using the <code>RemoveTag</code> aspect, see below:</p>
 * <pre><code class="language-ts">// snip //
 * const vpc = new ec2.Vpc(marketingStack, 'MarketingVpc', {
 *   maxAZs: 3 // Default is all AZs in region
 *   });
 * // override the VPC tags with Platform
 * // this will tag the VPC, Subnets, Route Tables, IGW, and NatGWs
 * vpc.node.apply(new cdk.Tag(COST_CENTER_KEY, 'Platform'));
 * vpc.node.apply(new cdk.RemoveTag('Name'));
 * // snip //
 * </code></pre>
 * <p>This will remove the name tags from the VPC, subnets, route tables and NAT
 * gateways. If you've been following closely, this may lead you to ask how does
 * remove work when the tag is actually applied closer to the resource? The Tag API
 * has a few features that are covered later to explain how this works.</p>
 * <h3>API</h3>
 * <p>In order to enable additional controls a Tag can specifically include or
 * exclude a CloudFormation Resource Type, propagate tags for an autoscaling group,
 * and use priority to override the default precedence. See the <code>TagProps</code>
 * interface for more details.</p>
 * <p>Tags can be configured by using the properties for the AWS CloudFormation layer
 * resources or by using the tag aspects described here. The aspects will always
 * take precedence over the AWS CloudFormation layer in the event of a name
 * collision. The tags will be merged otherwise. For the aspect based tags, the
 * tags applied closest to the resource will take precedence, given an equal
 * priority. A higher priority tag will always take precedence over a lower
 * priority tag.</p>
 * <h4>applyToLaunchedInstances</h4>
 * <p>This property is a boolean that defaults to <code>true</code>. When <code>true</code> and the aspect
 * visits an AutoScalingGroup resource the <code>PropagateAtLaunch</code> property is set to
 * true. If false the property is set accordingly.</p>
 * <pre><code class="language-ts">// ... snip
 * const vpc = new ec2.Vpc(this, 'MyVpc', { ... });
 * vpc.node.apply(new cdk.Tag('MyKey', 'MyValue', { applyToLaunchedInstances: false }));
 * // ... snip
 * </code></pre>
 * <h4>includeResourceTypes</h4>
 * <p>Include is an array property that contains strings of CloudFormation Resource
 * Types. As the aspect visits nodes it only takes action if node is one of the
 * resource types in the array. By default the array is empty and an empty array is
 * interpreted as apply to any resource type.</p>
 * <pre><code class="language-ts">// ... snip
 * const vpc = new ec2.Vpc(this, 'MyVpc', { ... });
 * vpc.node.apply(new cdk.Tag('MyKey', 'MyValue', { includeResourceTypes: ['AWS::EC2::Subnet']}));
 * // ... snip
 * </code></pre>
 * <h4>excludeResourceTypes</h4>
 * <p>Exclude is the inverse of include. Exclude is also an array of CloudFormation
 * Resource Types. As the aspect visit nodes it will not take action if the node is
 * one of the resource types in the array. By default the array is empty and an
 * empty array is interpreted to match no resource type. Exclude takes precedence
 * over include in the event of a collision.</p>
 * <pre><code class="language-ts">// ... snip
 * const vpc = new ec2.Vpc(this, 'MyVpc', { ... });
 * vpc.node.apply(new cdk.Tag('MyKey', 'MyValue', { exludeResourceTypes: ['AWS::EC2::Subnet']}));
 * // ... snip
 * </code></pre>
 * <h4>priority</h4>
 * <p>Priority is used to control precedence when the default pattern does not work.
 * In general users should try to avoid using priority, but in some situations it
 * is required. In the example above, this is how <code>RemoveTag</code> works. The default
 * setting for removing tags uses a higher priority than the standard tag.</p>
 * <pre><code class="language-ts">// ... snip
 * const vpc = new ec2.Vpc(this, 'MyVpc', { ... });
 * vpc.node.apply(new cdk.Tag('MyKey', 'MyValue', { priority: 2 }));
 * // ... snip
 * </code></pre>
 * <h2>Secrets</h2>
 * <p>To help avoid accidental storage of secrets as plain text we use the <code>SecretValue</code> type to
 * represent secrets.</p>
 * <p>The best practice is to store secrets in AWS Secrets Manager and reference them using <code>SecretValue.secretsManager</code>:</p>
 * <pre><code class="language-ts">const secret = SecretValue.secretsManager('secretId', {
 *   jsonField: 'password' // optional: key of a JSON field to retrieve (defaults to all content),
 *   versionId: 'id'       // optional: id of the version (default AWSCURRENT)
 *   versionStage: 'stage' // optional: version stage name (default AWSCURRENT)
 * });
 * </code></pre>
 * <p>Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
 * However, <code>SecretValue</code> supports the following additional options:</p>
 * <ul>
 * <li><code>SecretValue.plainText(secret)</code>: stores the secret as plain text in your app and the resulting template (not recommended).</li>
 * <li><code>SecretValue.ssmSecure(param, version)</code>: refers to a secret stored as a SecureString in the SSM Parameter Store.</li>
 * <li><code>SecretValue.cfnParameter(param)</code>: refers to a secret passed through a CloudFormation parameter (must have <code>NoEcho: true</code>).</li>
 * <li><code>SecretValue.cfnDynamicReference(dynref)</code>: refers to a secret described by a CloudFormation dynamic reference (used by <code>ssmSecure</code> and <code>secretsManager</code>).</li>
 * </ul>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.core;
