/**
 * <h2>Amazon Cognito Construct Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * | Features | Stability |
 * | --- | --- |
 * | CFN Resources | <img alt="Stable" src="https://img.shields.io/badge/stable-success.svg?style=for-the-badge"> |
 * | Higher level constructs for User Pools | <img alt="Developer Preview" src="https://img.shields.io/badge/developer--preview-informational.svg?style=for-the-badge"> |
 * | Higher level constructs for Identity Pools | <img alt="Not Implemented" src="https://img.shields.io/badge/not--implemented-black.svg?style=for-the-badge"> |
 * <p>
 * <blockquote>
 * <p>
 * <strong>CFN Resources:</strong> All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always stable and safe to use.
 * <p>
 * </blockquote><blockquote>
 * <p>
 * <strong>Developer Preview:</strong> Higher level constructs in this module that are marked as developer preview have completed their phase of active development and are looking for adoption and feedback. While the same caveats around non-backward compatible as Experimental constructs apply, they will undergo fewer breaking changes. Just as with Experimental constructs, these are not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be announced in the release notes.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/what-is-amazon-cognito.html">Amazon Cognito</a> provides
 * authentication, authorization, and user management for your web and mobile apps. Your users can sign in directly with a
 * user name and password, or through a third party such as Facebook, Amazon, Google or Apple.
 * <p>
 * The two main components of Amazon Cognito are <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">user
 * pools</a> and <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html">identity
 * pools</a>. User pools are user directories
 * that provide sign-up and sign-in options for your app users. Identity pools enable you to grant your users access to
 * other AWS services.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#user-pools">User Pools</a>
 * <p>
 * <ul>
 * <li><a href="#sign-up">Sign Up</a></li>
 * <li><a href="#sign-in">Sign In</a></li>
 * <li><a href="#attributes">Attributes</a></li>
 * <li><a href="#security">Security</a>
 * <p>
 * <ul>
 * <li><a href="#multi-factor-authentication-mfa">Multi-factor Authentication</a></li>
 * <li><a href="#account-recovery-settings">Account Recovery Settings</a></li>
 * </ul></li>
 * <li><a href="#emails">Emails</a></li>
 * <li><a href="#lambda-triggers">Lambda Triggers</a></li>
 * <li><a href="#importing-user-pools">Import</a></li>
 * <li><a href="#identity-providers">Identity Providers</a></li>
 * <li><a href="#app-clients">App Clients</a></li>
 * <li><a href="#domains">Domains</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>User Pools</h2>
 * <p>
 * User pools allow creating and managing your own directory of users that can sign up and sign in. They enable easy
 * integration with social identity providers such as Facebook, Google, Amazon, Microsoft Active Directory, etc. through
 * SAML.
 * <p>
 * Using the CDK, a new user pool can be created as part of the stack using the construct's constructor. You may specify
 * the <code>userPoolName</code> to give your own identifier to the user pool. If not, CloudFormation will generate a name.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         .userPoolName("myawesomeapp-userpool")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Sign Up</h3>
 * <p>
 * Users can either be signed up by the app's administrators or can sign themselves up. Once a user has signed up, their
 * account needs to be confirmed. Cognito provides several ways to sign users up and confirm their accounts. Learn more
 * about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html">user sign up here</a>.
 * <p>
 * When a user signs up, email and SMS messages are used to verify their account and contact methods. The following code
 * snippet configures a user pool with properties relevant to these verification messages -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .selfSignUpEnabled(true)
 *         .userVerification(Map.of(
 *                 "emailSubject", "Verify your email for our awesome app!",
 *                 "emailBody", "Hello {username}, Thanks for signing up to our awesome app! Your verification code is {####}",
 *                 "emailStyle", VerificationEmailStyle.getCODE(),
 *                 "smsMessage", "Hello {username}, Thanks for signing up to our awesome app! Your verification code is {####}"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, self sign up is disabled. Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html">email and SMS verification messages
 * here</a>.
 * <p>
 * Besides users signing themselves up, an administrator of any user pool can sign users up. The user then receives an
 * invitation to join the user pool. The following code snippet configures a user pool with properties relevant to the
 * invitation messages -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .userInvitation(Map.of(
 *                 "emailSubject", "Invite to join our awesome app!",
 *                 "emailBody", "Hello {username}, you have been invited to join our awesome app! Your temporary password is {####}",
 *                 "smsMessage", "Your temporary password for our awesome app is {####}"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * All email subjects, bodies and SMS messages for both invitation and verification support Cognito's message templating.
 * Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html">message templates
 * here</a>.
 * <p>
 * <h3>Sign In</h3>
 * <p>
 * Users registering or signing in into your application can do so with multiple identifiers. There are 4 options
 * available:
 * <p>
 * <ul>
 * <li><code>username</code>: Allow signing in using the one time immutable user name that the user chose at the time of sign up.</li>
 * <li><code>email</code>: Allow signing in using the email address that is associated with the account.</li>
 * <li><code>phone</code>: Allow signing in using the phone number that is associated with the account.</li>
 * <li><code>preferredUsername</code>: Allow signing in with an alternate user name that the user can change at any time. However, this
 * is not available if the <code>username</code> option is not chosen.</li>
 * </ul>
 * <p>
 * The following code sets up a user pool so that the user can sign in with either their username or their email address -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         // ...
 *         .signInAliases(Map.of(
 *                 "username", true,
 *                 "email", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * User pools can either be configured so that user name is primary sign in form, but also allows for the other three to be
 * used additionally; or it can be configured so that email and/or phone numbers are the only ways a user can register and
 * sign in. Read more about this
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases-settings">here</a>.
 * <p>
 * To match with 'Option 1' in the above link, with a verified email, <code>signInAliases</code> should be set to
 * <code>{ username: true, email: true }</code>. To match with 'Option 2' in the above link with both a verified
 * email and phone number, this property should be set to <code>{ email: true, phone: true }</code>.
 * <p>
 * Cognito recommends that email and phone number be automatically verified, if they are one of the sign in methods for
 * the user pool. Read more about that
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases">here</a>.
 * The CDK does this by default, when email and/or phone number are specified as part of <code>signInAliases</code>. This can be
 * overridden by specifying the <code>autoVerify</code> property.
 * <p>
 * The following code snippet sets up only email as a sign in alias, but both email and phone number to be auto-verified.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         // ...
 *         .signInAliases(Map.of("username", true, "email", true))
 *         .autoVerify(Map.of("email", true, "phone", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * A user pool can optionally ignore case when evaluating sign-ins. When <code>signInCaseSensitive</code> is false, Cognito will not
 * check the capitalization of the alias when signing in. Default is true.
 * <p>
 * <h3>Attributes</h3>
 * <p>
 * Attributes represent the various properties of each user that's collected and stored in the user pool. Cognito
 * provides a set of standard attributes that are available for all user pools. Users are allowed to select any of these
 * standard attributes to be required. Users will not be able to sign up to the user pool without providing the required
 * attributes. Besides these, additional attributes can be further defined, and are known as custom attributes.
 * <p>
 * Learn more on <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html">attributes in Cognito's
 * documentation</a>.
 * <p>
 * The following code configures a user pool with two standard attributes (name and address) as required and mutable, and adds
 * four custom attributes.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .standardAttributes(Map.of(
 *                 "fullname", Map.of(
 *                         "required", true,
 *                         "mutable", false),
 *                 "address", Map.of(
 *                         "required", false,
 *                         "mutable", true)))
 *         .customAttributes(Map.of(
 *                 "myappid", StringAttribute.Builder.create().minLen(5).maxLen(15).mutable(false).build(),
 *                 "callingcode", NumberAttribute.Builder.create().min(1).max(3).mutable(true).build(),
 *                 "isEmployee", BooleanAttribute.Builder.create().mutable(true).build(),
 *                 "joinedOn", new DateTimeAttribute()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * As shown in the code snippet, there are data types that are available for custom attributes. The 'String' and 'Number'
 * data types allow for further constraints on their length and values, respectively.
 * <p>
 * Custom attributes cannot be marked as required.
 * <p>
 * All custom attributes share the property <code>mutable</code> that specifies whether the value of the attribute can be changed.
 * The default value is <code>false</code>.
 * <p>
 * <h3>Security</h3>
 * <p>
 * Cognito sends various messages to its users via SMS, for different actions, ranging from account verification to
 * marketing. In order to send SMS messages, Cognito needs an IAM role that it can assume, with permissions that allow it
 * to send SMS messages.
 * <p>
 * By default, the CDK looks at all of the specified properties (and their defaults when not explicitly specified) and
 * automatically creates an SMS role, when needed. For example, if MFA second factor by SMS is enabled, the CDK will
 * create a new role. The <code>smsRole</code> property can be used to specify the user supplied role that should be used instead.
 * Additionally, the property <code>enableSmsRole</code> can be used to override the CDK's default behaviour to either enable or
 * suppress automatic role creation.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.iam.Role;
 * 
 * Role poolSmsRole = new Role(this, "userpoolsmsrole", new RoleProps());
 * 
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .smsRole(poolSmsRole)
 *         .smsRoleExternalId("c87467be-4f34-11ea-b77f-2e728ce88125")
 *         .build();
 * </pre></blockquote>
 * <p>
 * When the <code>smsRole</code> property is specified, the <code>smsRoleExternalId</code> may also be specified. The value of
 * <code>smsRoleExternalId</code> will be used as the <code>sts:ExternalId</code> when the Cognito service assumes the role. In turn, the role's
 * assume role policy should be configured to accept this value as the ExternalId. Learn more about <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html">ExternalId
 * here</a>.
 * <p>
 * <h4>Multi-factor Authentication (MFA)</h4>
 * <p>
 * User pools can be configured to enable multi-factor authentication (MFA). It can either be turned off, set to optional
 * or made required. Setting MFA to optional means that individual users can choose to enable it.
 * Additionally, the MFA code can be sent either via SMS text message or via a time-based software token.
 * See the <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html">documentation on MFA</a> to
 * learn more.
 * <p>
 * The following code snippet marks MFA for the user pool as required. This means that all users are required to
 * configure an MFA token and use it for sign in. It also allows for the users to use both SMS based MFA, as well,
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-totp.html">time-based one time password
 * (TOTP)</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .mfa(Mfa.getREQUIRED())
 *         .mfaSecondFactor(Map.of(
 *                 "sms", true,
 *                 "otp", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * User pools can be configured with policies around a user's password. This includes the password length and the
 * character sets that they must contain.
 * <p>
 * Further to this, it can also be configured with the validity of the auto-generated temporary password. A temporary
 * password is generated by the user pool either when an admin signs up a user or when a password reset is requested.
 * The validity of this password dictates how long to give the user to use this password before expiring it.
 * <p>
 * The following code snippet configures these properties -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .passwordPolicy(Map.of(
 *                 "minLength", 12,
 *                 "requireLowercase", true,
 *                 "requireUppercase", true,
 *                 "requireDigits", true,
 *                 "requireSymbols", true,
 *                 "tempPasswordValidity", Duration.days(3)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that, <code>tempPasswordValidity</code> can be specified only in whole days. Specifying fractional days would throw an error.
 * <p>
 * <h4>Account Recovery Settings</h4>
 * <p>
 * User pools can be configured on which method a user should use when recovering the password for their account. This
 * can either be email and/or SMS. Read more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html">Recovering User Accounts</a>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "UserPool")...
 *         .accountRecovery(AccountRecovery.getEMAIL_ONLY())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Emails</h3>
 * <p>
 * Cognito sends emails to users in the user pool, when particular actions take place, such as welcome emails, invitation
 * emails, password resets, etc. The address from which these emails are sent can be configured on the user pool.
 * Read more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html">email settings here</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .emailSettings(Map.of(
 *                 "from", "noreply&#64;myawesomeapp.com",
 *                 "replyTo", "support&#64;myawesomeapp.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, user pools are configured to use Cognito's built-in email capability, but it can also be configured to use
 * Amazon SES, however, support for Amazon SES is not available in the CDK yet. If you would like this to be implemented,
 * give <a href="https://github.com/aws/aws-cdk/issues/6768">this issue</a> a +1. Until then, you can use the <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html">cfn
 * layer</a> to configure this.
 * <p>
 * <h3>Lambda Triggers</h3>
 * <p>
 * User pools can be configured such that AWS Lambda functions can be triggered when certain user operations or actions
 * occur, such as, sign up, user confirmation, sign in, etc. They can also be used to add custom authentication
 * challenges, user migrations and custom verification messages. Learn more about triggers at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">User Pool Workflows with
 * Triggers</a>.
 * <p>
 * Lambda triggers can either be specified as part of the <code>UserPool</code> initialization, or it can be added later, via methods
 * on the construct, as so -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var authChallengeFn = Function.Builder.create(this, "authChallengeFn").build();
 * 
 * var userpool = UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .triggers(Map.of(
 *                 "createAuthChallenge", authChallengeFn))
 *         .build();
 * 
 * userpool.addTrigger(UserPoolOperation.getUSER_MIGRATION(), Function.Builder.create(this, "userMigrationFn").build());
 * </pre></blockquote>
 * <p>
 * The following table lists the set of triggers available, and their corresponding method to add it to the user pool.
 * For more information on the function of these triggers and how to configure them, read <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">User Pool Workflows with
 * Triggers</a>.
 * <p>
 * <h3>Importing User Pools</h3>
 * <p>
 * Any user pool that has been created outside of this stack, can be imported into the CDK app. Importing a user pool
 * allows for it to be used in other parts of the CDK app that reference an <code>IUserPool</code>. However, imported user pools have
 * limited configurability. As a rule of thumb, none of the properties that is are part of the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html"><code>AWS::Cognito::UserPool</code></a>
 * CloudFormation resource can be configured.
 * <p>
 * User pools can be imported either using their id via the <code>UserPool.fromUserPoolId()</code>, or by using their ARN, via the
 * <code>UserPool.fromUserPoolArn()</code> API.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var stack = new Stack(app, "my-stack");
 * 
 * var awesomePool = UserPool.fromUserPoolId(stack, "awesome-user-pool", "us-east-1_oiuR12Abd");
 * 
 * var otherAwesomePool = UserPool.fromUserPoolArn(stack, "other-awesome-user-pool", "arn:aws:cognito-idp:eu-west-1:123456789012:userpool/us-east-1_mtRyYQ14D");
 * </pre></blockquote>
 * <p>
 * <h3>Identity Providers</h3>
 * <p>
 * Users that are part of a user pool can sign in either directly through a user pool, or federate through a third-party
 * identity provider. Once configured, the Cognito backend will take care of integrating with the third-party provider.
 * Read more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-identity-federation.html">Adding User Pool Sign-in Through a Third
 * Party</a>.
 * <p>
 * The following third-party identity providers are currentlhy supported in the CDK -
 * <p>
 * <ul>
 * <li><a href="https://developer.amazon.com/apps-and-games/login-with-amazon">Login With Amazon</a></li>
 * <li><a href="https://developers.facebook.com/docs/facebook-login/">Facebook Login</a></li>
 * </ul>
 * <p>
 * The following code configures a user pool to federate with the third party provider, 'Login with Amazon'. The identity
 * provider needs to be configured with a set of credentials that the Cognito backend can use to federate with the
 * third-party identity provider.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var userpool = new UserPool(stack, "Pool");
 * 
 * var provider = UserPoolIdentityProviderAmazon.Builder.create(stack, "Amazon")
 *         .clientId("amzn-client-id")
 *         .clientSecret("amzn-client-secret")
 *         .userPool(userpool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Attribute mapping allows mapping attributes provided by the third-party identity providers to <a href="#Attributes">standard and custom
 * attributes</a> of the user pool. Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html">Specifying Identity Provider Attribute Mappings for Your
 * User Pool</a>.
 * <p>
 * The following code shows how different attributes provided by 'Login With Amazon' can be mapped to standard and custom
 * user pool attributes.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * UserPoolIdentityProviderAmazon.Builder.create(stack, "Amazon")
 *         // ...
 *         .attributeMapping(Map.of(
 *                 "email", ProviderAttribute.getAMAZON_EMAIL(),
 *                 "website", ProviderAttribute.other("url"), // use other() when an attribute is not pre-defined in the CDK
 *                 "custom", Map.of(
 *                         // custom user pool attributes go here
 *                         "uniqueId", ProviderAttribute.getAMAZON_USER_ID())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>App Clients</h3>
 * <p>
 * An app is an entity within a user pool that has permission to call unauthenticated APIs (APIs that do not have an
 * authenticated user), such as APIs to register, sign in, and handle forgotten passwords. To call these APIs, you need an
 * app client ID and an optional client secret. Read <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html">Configuring a User Pool App
 * Client</a> to learn more.
 * <p>
 * The following code creates an app client and retrieves the client id -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "pool");
 * var client = pool.addClient("customer-app-client");
 * var clientId = client.getUserPoolClientId();
 * </pre></blockquote>
 * <p>
 * Existing app clients can be imported into the CDK app using the <code>UserPoolClient.fromUserPoolClientId()</code> API. For new
 * and imported user pools, clients can also be created via the <code>UserPoolClient</code> constructor, as so -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var importedPool = UserPool.fromUserPoolId(this, "imported-pool", "us-east-1_oiuR12Abd");
 * UserPoolClient.Builder.create(this, "customer-app-client")
 *         .userPool(importedPool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Clients can be configured with authentication flows. Authentication flows allow users on a client to be authenticated
 * with a user pool. Cognito user pools provide several several different types of authentication, such as, SRP (Secure
 * Remote Password) authentication, username-and-password authentication, etc. Learn more about this at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html">UserPool Authentication
 * Flow</a>.
 * <p>
 * The following code configures a client to use both SRP and username-and-password authentication -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "pool");
 * pool.addClient("app-client", Map.of(
 *         "authFlows", Map.of(
 *                 "userPassword", true,
 *                 "userSrp", true)));
 * </pre></blockquote>
 * <p>
 * Custom authentication protocols can be configured by setting the <code>custom</code> property under <code>authFlow</code> and defining lambda
 * functions for the corresponding user pool <a href="#lambda-triggers">triggers</a>. Learn more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html#amazon-cognito-user-pools-custom-authentication-flow">Custom Authentication
 * Flow</a>.
 * <p>
 * In addition to these authentication mechanisms, Cognito user pools also support using OAuth 2.0 framework for
 * authenticating users. User pool clients can be configured with OAuth 2.0 authorization flows and scopes. Learn more
 * about the <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0 authorization framework</a> and <a href="https://aws.amazon.com/blogs/mobile/understanding-amazon-cognito-user-pool-oauth-2-0-grants/">Cognito user pool's
 * implementation of
 * OAuth2.0</a>.
 * <p>
 * The following code configures an app client with the authorization code grant flow and registers the the app's welcome
 * page as a callback (or redirect) URL. It also configures the access token scope to 'openid'. All of these concepts can
 * be found in the <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0 RFC</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", Map.of(
 *         "oAuth", Map.of(
 *                 "flows", Map.of(
 *                         "authorizationCodeGrant", true),
 *                 "scopes", asList(OAuthScope.getOPENID()),
 *                 "callbackUrls", asList("https://my-app-domain.com/welcome"))));
 * </pre></blockquote>
 * <p>
 * An app client can be configured to prevent user existence errors. This
 * instructs the Cognito authentication API to return generic authentication
 * failure responses instead of an UserNotFoundException. By default, the flag
 * is not set, which means different things for existing and new stacks. See the
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-managing-errors.html">documentation</a>
 * for the full details on the behavior of this flag.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", Map.of(
 *         "preventUserExistenceErrors", true));
 * </pre></blockquote>
 * <p>
 * All identity providers created in the CDK app are automatically registered into the corresponding user pool. All app
 * clients created in the CDK have all of the identity providers enabled by default. The 'Cognito' identity provider,
 * that allows users to register and sign in directly with the Cognito user pool, is also enabled by default.
 * Alternatively, the list of supported identity providers for a client can be explicitly specified -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", Map.of(
 *         // ...
 *         "supportedIdentityProviders", asList(UserPoolClientIdentityProvider.getAMAZON(), UserPoolClientIdentityProvider.getCOGNITO())));
 * </pre></blockquote>
 * <p>
 * <h3>Domains</h3>
 * <p>
 * After setting up an <a href="#app-clients">app client</a>, the address for the user pool's sign-up and sign-in webpages can be
 * configured using domains. There are two ways to set up a domain - either the Amazon Cognito hosted domain can be chosen
 * with an available domain prefix, or a custom domain name can be chosen. The custom domain must be one that is already
 * owned, and whose certificate is registered in AWS Certificate Manager.
 * <p>
 * The following code sets up a user pool domain in Amazon Cognito hosted domain with the prefix 'my-awesome-app', and another domain with the custom domain 'user.myapp.com' -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var pool = new UserPool(this, "Pool");
 * 
 * pool.addDomain("CognitoDomain", Map.of(
 *         "cognitoDomain", Map.of(
 *                 "domainPrefix", "my-awesome-app")));
 * 
 * var domainCert = new fromCertificateArn(this, "domainCert", certificateArn);
 * pool.addDomain("CustomDomain", Map.of(
 *         "customDomain", Map.of(
 *                 "domainName", "user.myapp.com",
 *                 "certificate", domainCert)));
 * </pre></blockquote>
 * <p>
 * Read more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain-prefix.html">Using the Amazon Cognito
 * Domain</a> and <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-add-custom-domain.html">Using Your Own
 * Domain</a>.
 * <p>
 * The <code>signInUrl()</code> methods returns the fully qualified URL to the login page for the user pool. This page comes from the
 * hosted UI configured with Cognito. Learn more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-integration.html#cognito-user-pools-create-an-app-integration">Hosted UI with the Amazon Cognito
 * Console</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var userpool = UserPool.Builder.create(this, "UserPool").build();
 * var client = userpool.addClient("Client", Map.of(
 *         // ...
 *         "oAuth", Map.of(
 *                 "flows", Map.of(
 *                         "implicitCodeGrant", true),
 *                 "callbackUrls", asList("https://myapp.com/home", "https://myapp.com/users"))));
 * var domain = userpool.addDomain("Domain", Map.of());
 * var signInUrl = domain.signInUrl(client, Map.of(
 *         "redirectUrl", "https://myapp.com/home"));
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.cognito;
