/**
 * <h2>AWS Batch Construct Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always stable and safe to use.
 * <p>
 * </blockquote>
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development. They are subject to non-backward compatible changes or removal in any future version. These are not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be announced in the release notes. This means that while you may use them, you may need to update your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * AWS Batch is a batch processing tool for efficiently running hundreds of thousands computing jobs in AWS. Batch can dynamically provision different types of compute resources based on the resource requirements of submitted jobs.
 * <p>
 * AWS Batch simplifies the planning, scheduling, and executions of your batch workloads across a full range of compute services like <a href="https://aws.amazon.com/ec2/">Amazon EC2</a> and <a href="https://aws.amazon.com/ec2/spot/">Spot Resources</a>.
 * <p>
 * Batch achieves this by utilizing queue processing of batch job requests. To successfully submit a job for execution, you need the following resources:
 * <p>
 * <ol>
 * <li><a href="#job-definition">Job Definition</a> - <em>Group various job properties (container image, resource requirements, env variables...) into a single definition. These definitions are used at job submission time.</em></li>
 * <li><a href="#compute-environment">Compute Environment</a> - <em>the execution runtime of submitted batch jobs</em></li>
 * <li><a href="#job-queue">Job Queue</a> - <em>the queue where batch jobs can be submitted to via AWS SDK/CLI</em></li>
 * </ol>
 * <p>
 * For more information on <strong>AWS Batch</strong> visit the <a href="https://docs.aws.amazon.com/batch/index.html">AWS Docs for Batch</a>.
 * <p>
 * <h2>Compute Environment</h2>
 * <p>
 * At the core of AWS Batch is the compute environment. All batch jobs are processed within a compute environment, which uses resource like OnDemand or Spot EC2 instances.
 * <p>
 * In <strong>MANAGED</strong> mode, AWS will handle the provisioning of compute resources to accommodate the demand. Otherwise, in <strong>UNMANAGED</strong> mode, you will need to manage the provisioning of those resources.
 * <p>
 * Below is an example of each available type of compute environment:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var defaultVpc = new Vpc(this, "VPC");
 * 
 * // default is managed
 * var awsManagedEnvironment = ComputeEnvironment.Builder.create(stack, "AWS-Managed-Compute-Env")
 *         .computeResources(Map.of(
 *                 "vpc", vpc))
 *         .build();
 * 
 * var customerManagedEnvironment = ComputeEnvironment.Builder.create(stack, "Customer-Managed-Compute-Env")
 *         .managed(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Spot-Based Compute Environment</h3>
 * <p>
 * It is possible to have AWS Batch submit spotfleet requests for obtaining compute resources. Below is an example of how this can be done:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var vpc = new Vpc(this, "VPC");
 * 
 * var spotEnvironment = ComputeEnvironment.Builder.create(stack, "MySpotEnvironment")
 *         .computeResources(Map.of(
 *                 "type", batch.ComputeResourceType.getSPOT(),
 *                 "bidPercentage", 75, // Bids for resources at 75% of the on-demand price
 *                 "vpc", vpc))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Understanding Progressive Allocation Strategies</h3>
 * <p>
 * AWS Batch uses an <a href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">allocation strategy</a> to determine what compute resource will efficiently handle incoming job requests. By default, <strong>BEST_FIT</strong> will pick an available compute instance based on vCPU requirements. If none exist, the job will wait until resources become available. However, with this strategy, you may have jobs waiting in the queue unnecessarily despite having more powerful instances available. Below is an example of how that situation might look like:
 * <p>
 * <blockquote><pre>
 * Compute Environment:
 * 
 * 1. m5.xlarge =&gt; 4 vCPU
 * 2. m5.2xlarge =&gt; 8 vCPU
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * Job Queue:
 * ---------
 * | A | B |
 * ---------
 * 
 * Job Requirements:
 * A =&gt; 4 vCPU - ALLOCATED TO m5.xlarge
 * B =&gt; 2 vCPU - WAITING
 * </pre></blockquote>
 * <p>
 * In this situation, Batch will allocate <strong>Job A</strong> to compute resource #1 because it is the most cost efficient resource that matches the vCPU requirement. However, with this <code>BEST_FIT</code> strategy, <strong>Job B</strong> will not be allocated to our other available compute resource even though it is strong enough to handle it. Instead, it will wait until the first job is finished processing or wait a similar <code>m5.xlarge</code> resource to be provisioned.
 * <p>
 * The alternative would be to use the <code>BEST_FIT_PROGRESSIVE</code> strategy in order for the remaining job to be handled in larger containers regardless of vCPU requirement and costs.
 * <p>
 * <h3>Launch template support</h3>
 * <p>
 * Simply define your Launch Template:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var myLaunchTemplate = CfnLaunchTemplate.Builder.create(this, "LaunchTemplate")
 *         .launchTemplateName("extra-storage-template")
 *         .launchTemplateData(Map.of(
 *                 "blockDeviceMappings", asList(Map.of(
 *                         "deviceName", "/dev/xvdcz",
 *                         "ebs", Map.of(
 *                                 "encrypted", true,
 *                                 "volumeSize", 100,
 *                                 "volumeType", "gp2")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * and use it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var myComputeEnv = ComputeEnvironment.Builder.create(this, "ComputeEnv")
 *         .computeResources(Map.of(
 *                 "launchTemplate", Map.of(
 *                         "launchTemplateName", (String)myLaunchTemplate.getLaunchTemplateName()),
 *                 "vpc", vpc))
 *         .computeEnvironmentName("MyStorageCapableComputeEnvironment")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Importing an existing Compute Environment</h3>
 * <p>
 * To import an existing batch compute environment, call <code>ComputeEnvironment.fromComputeEnvironmentArn()</code>.
 * <p>
 * Below is an example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var computeEnv = batch.ComputeEnvironment.fromComputeEnvironmentArn(this, "imported-compute-env", "arn:aws:batch:us-east-1:555555555555:compute-environment/My-Compute-Env");
 * </pre></blockquote>
 * <p>
 * <h3>Change the baseline AMI of the compute resources</h3>
 * <p>
 * Ocassionally, you will need to deviate from the default processing AMI.
 * <p>
 * ECS Optimized Amazon Linux 2 example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var myComputeEnv = ComputeEnvironment.Builder.create(this, "ComputeEnv")
 *         .computeResources(Map.of(
 *                 "image", EcsOptimizedAmi.Builder.create()
 *                         .generation(ec2.AmazonLinuxGeneration.getAMAZON_LINUX_2())
 *                         .build(),
 *                 "vpc", vpc))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Custom based AMI example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var myComputeEnv = ComputeEnvironment.Builder.create(this, "ComputeEnv")
 *         .computeResources(Map.of(
 *                 "image", ec2.MachineImage.genericLinux(Map.of(
 *                         "[aws-region]", "[ami-ID]")),
 *                 "vpc", vpc))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Job Queue</h2>
 * <p>
 * Jobs are always submitted to a specific queue. This means that you have to create a queue before you can start submitting jobs. Each queue is mapped to at least one (and no more than three) compute environment. When the job is scheduled for execution, AWS Batch will select the compute environment based on ordinal priority and available capacity in each environment.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var jobQueue = JobQueue.Builder.create(stack, "JobQueue")
 *         .computeEnvironments(asList(Map.of(
 *                 // Defines a collection of compute resources to handle assigned batch jobs
 *                 "computeEnvironment", computeEnvironment,
 *                 // Order determines the allocation order for jobs (i.e. Lower means higher preferance for job assignment)
 *                 "order", 1)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Priorty-Based Queue Example</h3>
 * <p>
 * Sometimes you might have jobs that are more important than others, and when submitted, should take precedence over the existing jobs. To achieve this, you can create a priority based execution strategy, by assigning each queue its own priority:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var highPrioQueue = JobQueue.Builder.create(stack, "JobQueue")
 *         .computeEnvironments(sharedComputeEnvs)
 *         .priority(2)
 *         .build();
 * 
 * var lowPrioQueue = JobQueue.Builder.create(stack, "JobQueue")
 *         .computeEnvironments(sharedComputeEnvs)
 *         .priority(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By making sure to use the same compute environments between both job queues, we will give precedence to the <code>highPrioQueue</code> for the assigning of jobs to available compute environments.
 * <p>
 * <h3>Importing an existing Job Queue</h3>
 * <p>
 * To import an existing batch job queue, call <code>JobQueue.fromJobQueueArn()</code>.
 * <p>
 * Below is an example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var jobQueue = batch.JobQueue.fromJobQueueArn(this, "imported-job-queue", "arn:aws:batch:us-east-1:555555555555:job-queue/High-Prio-Queue");
 * </pre></blockquote>
 * <p>
 * <h2>Job Definition</h2>
 * <p>
 * A Batch Job definition helps AWS Batch understand important details about how to run your application in the scope of a Batch Job. This involves key information like resource requirements, what containers to run, how the compute environment should be prepared, and more. Below is a simple example of how to create a job definition:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var repo = ecr.Repository.fromRepositoryName(stack, "batch-job-repo", "todo-list");
 * 
 * JobDefinition.Builder.create(stack, "batch-job-def-from-ecr")
 *         .container(Map.of(
 *                 "image", new EcrImage(repo, "latest")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Using a local Docker project</h3>
 * <p>
 * Below is an example of how you can create a Batch Job Definition from a local Docker application.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * JobDefinition.Builder.create(stack, "batch-job-def-from-local")
 *         .container(Map.of(
 *                 // todo-list is a directory containing a Dockerfile to build the application
 *                 "image", ecs.ContainerImage.fromAsset("../todo-list")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Importing an existing Job Definition</h3>
 * <p>
 * To import an existing batch job definition, call <code>JobDefinition.fromJobDefinitionArn()</code>.
 * <p>
 * Below is an example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var job = batch.JobDefinition.fromJobDefinitionArn(this, "imported-job-definition", "arn:aws:batch:us-east-1:555555555555:job-definition/my-job-definition");
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.batch;
