/**
 * <h1>AWS Secrets Manager Construct Library</h1>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.*;
 * </pre></blockquote>
 * <p>
 * <h2>Create a new Secret in a Stack</h2>
 * <p>
 * To have SecretsManager generate a new secret value automatically,
 * follow this example:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * // Simple secret
 * Secret secret = new Secret(this, "Secret");
 * // Using the secret
 * DatabaseInstance instance1 = DatabaseInstance.Builder.create(this, "PostgresInstance1")
 *         .engine(DatabaseInstanceEngine.POSTGRES)
 *         .credentials(Credentials.fromSecret(secret))
 *         .vpc(vpc)
 *         .build();
 * // Templated secret with username and password fields
 * Secret templatedSecret = Secret.Builder.create(this, "TemplatedSecret")
 *         .generateSecretString(SecretStringGenerator.builder()
 *                 .secretStringTemplate(JSON.stringify(Map.of("username", "postgres")))
 *                 .generateStringKey("password")
 *                 .build())
 *         .build();
 * // Using the templated secret as credentials
 * DatabaseInstance instance2 = DatabaseInstance.Builder.create(this, "PostgresInstance2")
 *         .engine(DatabaseInstanceEngine.POSTGRES)
 *         .credentials(Map.of(
 *                 "username", templatedSecret.secretValueFromJson("username").toString(),
 *                 "password", templatedSecret.secretValueFromJson("password")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you need to use a pre-existing secret, the recommended way is to manually
 * provision the secret in <em>AWS SecretsManager</em> and use the <code>Secret.fromSecretArn</code>
 * or <code>Secret.fromSecretAttributes</code> method to make it available in your CDK Application:
 * <p>
 * <blockquote><pre>
 * Key encryptionKey;
 * 
 * ISecret secret = Secret.fromSecretAttributes(this, "ImportedSecret", SecretAttributes.builder()
 *         .secretArn("arn:aws:secretsmanager:&lt;region&gt;:&lt;account-id-number&gt;:secret:&lt;secret-name&gt;-&lt;random-6-characters&gt;")
 *         // If the secret is encrypted using a KMS-hosted CMK, either import or reference that key:
 *         .encryptionKey(encryptionKey)
 *         .build());
 * </pre></blockquote>
 * <p>
 * SecretsManager secret values can only be used in select set of properties. For the
 * list of properties, see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">the CloudFormation Dynamic References documentation</a>.
 * <p>
 * A secret can set <code>RemovalPolicy</code>. If it set to <code>RETAIN</code>, removing that secret will fail.
 * <p>
 * <h2>Grant permission to use the secret to a role</h2>
 * <p>
 * You must grant permission to a resource for that resource to be allowed to
 * use a secret. This can be achieved with the <code>Secret.grantRead</code> and/or <code>Secret.grantWrite</code>
 * method, depending on your need:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "SomeRole").assumedBy(new AccountRootPrincipal()).build();
 * Secret secret = new Secret(this, "Secret");
 * secret.grantRead(role);
 * secret.grantWrite(role);
 * </pre></blockquote>
 * <p>
 * If, as in the following example, your secret was created with a KMS key:
 * <p>
 * <blockquote><pre>
 * Role role;
 * 
 * Key key = new Key(this, "KMS");
 * Secret secret = Secret.Builder.create(this, "Secret").encryptionKey(key).build();
 * secret.grantRead(role);
 * secret.grantWrite(role);
 * </pre></blockquote>
 * <p>
 * then <code>Secret.grantRead</code> and <code>Secret.grantWrite</code> will also grant the role the
 * relevant encrypt and decrypt permissions to the KMS key through the
 * SecretsManager service principal.
 * <p>
 * The principal is automatically added to Secret resource policy and KMS Key policy for cross account access:
 * <p>
 * <blockquote><pre>
 * AccountPrincipal otherAccount = new AccountPrincipal("1234");
 * Key key = new Key(this, "KMS");
 * Secret secret = Secret.Builder.create(this, "Secret").encryptionKey(key).build();
 * secret.grantRead(otherAccount);
 * </pre></blockquote>
 * <p>
 * <h2>Rotating a Secret</h2>
 * <p>
 * <h3>Using a Custom Lambda Function</h3>
 * <p>
 * A rotation schedule can be added to a Secret using a custom Lambda function:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function fn;
 * 
 * Secret secret = new Secret(this, "Secret");
 * 
 * secret.addRotationSchedule("RotationSchedule", RotationScheduleOptions.builder()
 *         .rotationLambda(fn)
 *         .automaticallyAfter(Duration.days(15))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Note: The required permissions for Lambda to call SecretsManager and the other way round are automatically granted based on <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets-required-permissions.html">AWS Documentation</a> as long as the Lambda is not imported.
 * <p>
 * See <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets-lambda-function-overview.html">Overview of the Lambda Rotation Function</a> on how to implement a Lambda Rotation Function.
 * <p>
 * <h3>Using a Hosted Lambda Function</h3>
 * <p>
 * Use the <code>hostedRotation</code> prop to rotate a secret with a hosted Lambda function:
 * <p>
 * <blockquote><pre>
 * Secret secret = new Secret(this, "Secret");
 * 
 * secret.addRotationSchedule("RotationSchedule", RotationScheduleOptions.builder()
 *         .hostedRotation(HostedRotation.mysqlSingleUser())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Hosted rotation is available for secrets representing credentials for MySQL, PostgreSQL, Oracle,
 * MariaDB, SQLServer, Redshift and MongoDB (both for the single and multi user schemes).
 * <p>
 * When deployed in a VPC, the hosted rotation implements <code>ec2.IConnectable</code>:
 * <p>
 * <blockquote><pre>
 * Vpc myVpc;
 * Connections dbConnections;
 * Secret secret;
 * 
 * 
 * HostedRotation myHostedRotation = HostedRotation.mysqlSingleUser(SingleUserHostedRotationOptions.builder().vpc(myVpc).build());
 * secret.addRotationSchedule("RotationSchedule", RotationScheduleOptions.builder().hostedRotation(myHostedRotation).build());
 * dbConnections.allowDefaultPortFrom(myHostedRotation);
 * </pre></blockquote>
 * <p>
 * Use the <code>excludeCharacters</code> option to customize the characters excluded from
 * the generated password when it is rotated. By default, the rotation excludes
 * the same characters as the ones excluded for the secret. If none are defined
 * then the following set is used: <code>% +~</code>#$&amp;*()|[]{}:;&lt;&gt;?!'/&#64;"`.
 * <p>
 * See also <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html">Automating secret creation in AWS CloudFormation</a>.
 * <p>
 * <h2>Rotating database credentials</h2>
 * <p>
 * Define a <code>SecretRotation</code> to rotate database credentials:
 * <p>
 * <blockquote><pre>
 * Secret mySecret;
 * IConnectable myDatabase;
 * Vpc myVpc;
 * 
 * 
 * SecretRotation.Builder.create(this, "SecretRotation")
 *         .application(SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER) // MySQL single user scheme
 *         .secret(mySecret)
 *         .target(myDatabase) // a Connectable
 *         .vpc(myVpc) // The VPC where the secret rotation application will be deployed
 *         .excludeCharacters(" %+:;{}")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The secret must be a JSON string with the following format:
 * <p>
 * <blockquote><pre>
 * {
 *   "engine": "&lt;required: database engine&gt;",
 *   "host": "&lt;required: instance host name&gt;",
 *   "username": "&lt;required: username&gt;",
 *   "password": "&lt;required: password&gt;",
 *   "dbname": "&lt;optional: database name&gt;",
 *   "port": "&lt;optional: if not specified, default port will be used&gt;",
 *   "masterarn": "&lt;required for multi user rotation: the arn of the master secret which will be used to create users/change passwords&gt;"
 * }
 * </pre></blockquote>
 * <p>
 * For the multi user scheme, a <code>masterSecret</code> must be specified:
 * <p>
 * <blockquote><pre>
 * Secret myUserSecret;
 * Secret myMasterSecret;
 * IConnectable myDatabase;
 * Vpc myVpc;
 * 
 * 
 * SecretRotation.Builder.create(this, "SecretRotation")
 *         .application(SecretRotationApplication.MYSQL_ROTATION_MULTI_USER)
 *         .secret(myUserSecret) // The secret that will be rotated
 *         .masterSecret(myMasterSecret) // The secret used for the rotation
 *         .target(myDatabase)
 *         .vpc(myVpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/aws-rds/README.md">aws-rds</a> where
 * credentials generation and rotation is integrated.
 * <p>
 * <h2>Importing Secrets</h2>
 * <p>
 * Existing secrets can be imported by ARN, name, and other attributes (including the KMS key used to encrypt the secret).
 * Secrets imported by name should use the short-form of the name (without the SecretsManager-provided suffx);
 * the secret name must exist in the same account and region as the stack.
 * Importing by name makes it easier to reference secrets created in different regions, each with their own suffix and ARN.
 * <p>
 * <blockquote><pre>
 * String secretCompleteArn = "arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret-f3gDy9";
 * String secretPartialArn = "arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret"; // No Secrets Manager suffix
 * IKey encryptionKey = Key.fromKeyArn(this, "MyEncKey", "arn:aws:kms:eu-west-1:111111111111:key/21c4b39b-fde2-4273-9ac0-d9bb5c0d0030");
 * ISecret mySecretFromCompleteArn = Secret.fromSecretCompleteArn(this, "SecretFromCompleteArn", secretCompleteArn);
 * ISecret mySecretFromPartialArn = Secret.fromSecretPartialArn(this, "SecretFromPartialArn", secretPartialArn);
 * ISecret mySecretFromName = Secret.fromSecretNameV2(this, "SecretFromName", "MySecret");
 * ISecret mySecretFromAttrs = Secret.fromSecretAttributes(this, "SecretFromAttributes", SecretAttributes.builder()
 *         .secretCompleteArn(secretCompleteArn)
 *         .encryptionKey(encryptionKey)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Replicating secrets</h2>
 * <p>
 * Secrets can be replicated to multiple regions by specifying <code>replicaRegions</code>:
 * <p>
 * <blockquote><pre>
 * Key myKey;
 * 
 * Secret.Builder.create(this, "Secret")
 *         .replicaRegions(List.of(ReplicaRegion.builder()
 *                 .region("eu-west-1")
 *                 .build(), ReplicaRegion.builder()
 *                 .region("eu-central-1")
 *                 .encryptionKey(myKey)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, use <code>addReplicaRegion()</code>:
 * <p>
 * <blockquote><pre>
 * Secret secret = new Secret(this, "Secret");
 * secret.addReplicaRegion("eu-west-1");
 * </pre></blockquote>
 * <p>
 * <h2>Creating JSON Secrets</h2>
 * <p>
 * Sometimes it is necessary to create a secret in SecretsManager that contains a JSON object.
 * For example:
 * <p>
 * <blockquote><pre>
 * {
 *   "username": "myUsername",
 *   "database": "foo",
 *   "password": "mypassword"
 * }
 * </pre></blockquote>
 * <p>
 * In order to create this type of secret, use the <code>secretObjectValue</code> input prop.
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * User user = new User(stack, "User");
 * AccessKey accessKey = AccessKey.Builder.create(stack, "AccessKey").user(user).build();
 * 
 * Secret.Builder.create(stack, "Secret")
 *         .secretObjectValue(Map.of(
 *                 "username", SecretValue.unsafePlainText(user.getUserName()),
 *                 "database", SecretValue.unsafePlainText("foo"),
 *                 "password", accessKey.getSecretAccessKey()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * In this case both the <code>username</code> and <code>database</code> are not a <code>Secret</code> so <code>SecretValue.unsafePlainText</code> needs to be used.
 * This means that they will be rendered as plain text in the template, but in this case neither of those
 * are actual "secrets".
 */
package software.amazon.awscdk.services.secretsmanager;
