/**
 * <h1>Amazon ECS Service Discovery Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * This package contains constructs for working with <strong>AWS Cloud Map</strong>
 * <p>
 * AWS Cloud Map is a fully managed service that you can use to create and
 * maintain a map of the backend services and resources that your applications
 * depend on.
 * <p>
 * For further information on AWS Cloud Map,
 * see the <a href="https://docs.aws.amazon.com/cloud-map">AWS Cloud Map documentation</a>
 * <p>
 * <h2>HTTP Namespace Example</h2>
 * <p>
 * The following example creates an AWS Cloud Map namespace that
 * supports API calls, creates a service in that namespace, and
 * registers an instance to it:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "aws-servicediscovery-integ");
 * 
 * HttpNamespace namespace = HttpNamespace.Builder.create(stack, "MyNamespace")
 *         .name("MyHTTPNamespace")
 *         .build();
 * 
 * Service service1 = namespace.createService("NonIpService", BaseServiceProps.builder()
 *         .description("service registering non-ip instances")
 *         .build());
 * 
 * service1.registerNonIpInstance("NonIpInstance", NonIpInstanceBaseProps.builder()
 *         .customAttributes(Map.of("arn", "arn:aws:s3:::mybucket"))
 *         .build());
 * 
 * Service service2 = namespace.createService("IpService", BaseServiceProps.builder()
 *         .description("service registering ip instances")
 *         .healthCheck(HealthCheckConfig.builder()
 *                 .type(HealthCheckType.HTTP)
 *                 .resourcePath("/check")
 *                 .build())
 *         .build());
 * 
 * service2.registerIpInstance("IpInstance", IpInstanceBaseProps.builder()
 *         .ipv4("54.239.25.192")
 *         .build());
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * <h2>Private DNS Namespace Example</h2>
 * <p>
 * The following example creates an AWS Cloud Map namespace that
 * supports both API calls and DNS queries within a vpc, creates a
 * service in that namespace, and registers a loadbalancer as an
 * instance:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "aws-servicediscovery-integ");
 * 
 * Vpc vpc = Vpc.Builder.create(stack, "Vpc").maxAzs(2).build();
 * 
 * PrivateDnsNamespace namespace = PrivateDnsNamespace.Builder.create(stack, "Namespace")
 *         .name("boobar.com")
 *         .vpc(vpc)
 *         .build();
 * 
 * Service service = namespace.createService("Service", DnsServiceProps.builder()
 *         .dnsRecordType(DnsRecordType.A_AAAA)
 *         .dnsTtl(Duration.seconds(30))
 *         .loadBalancer(true)
 *         .build());
 * 
 * ApplicationLoadBalancer loadbalancer = ApplicationLoadBalancer.Builder.create(stack, "LB").vpc(vpc).internetFacing(true).build();
 * 
 * service.registerLoadBalancer("Loadbalancer", loadbalancer);
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * <h2>Public DNS Namespace Example</h2>
 * <p>
 * The following example creates an AWS Cloud Map namespace that
 * supports both API calls and public DNS queries, creates a service in
 * that namespace, and registers an IP instance:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "aws-servicediscovery-integ");
 * 
 * PublicDnsNamespace namespace = PublicDnsNamespace.Builder.create(stack, "Namespace")
 *         .name("foobar.com")
 *         .build();
 * 
 * Service service = namespace.createService("Service", DnsServiceProps.builder()
 *         .name("foo")
 *         .dnsRecordType(DnsRecordType.A)
 *         .dnsTtl(Duration.seconds(30))
 *         .healthCheck(HealthCheckConfig.builder()
 *                 .type(HealthCheckType.HTTPS)
 *                 .resourcePath("/healthcheck")
 *                 .failureThreshold(2)
 *                 .build())
 *         .build());
 * 
 * service.registerIpInstance("IpInstance", IpInstanceBaseProps.builder()
 *         .ipv4("54.239.25.192")
 *         .port(443)
 *         .build());
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * For DNS namespaces, you can also register instances to services with CNAME records:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.*;
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "aws-servicediscovery-integ");
 * 
 * PublicDnsNamespace namespace = PublicDnsNamespace.Builder.create(stack, "Namespace")
 *         .name("foobar.com")
 *         .build();
 * 
 * Service service = namespace.createService("Service", DnsServiceProps.builder()
 *         .name("foo")
 *         .dnsRecordType(DnsRecordType.CNAME)
 *         .dnsTtl(Duration.seconds(30))
 *         .build());
 * 
 * service.registerCnameInstance("CnameInstance", CnameInstanceBaseProps.builder()
 *         .instanceCname("service.pizza")
 *         .build());
 * 
 * app.synth();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.servicediscovery;
