/**
 * <h1>Amazon DocumentDB Construct Library</h1>
 * <p>
 * <h2>Starting a Clustered Database</h2>
 * <p>
 * To set up a clustered DocumentDB database, define a <code>DatabaseCluster</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser") // NOTE: 'admin' is reserved by DocumentDB
 *                 .excludeCharacters("\"&#64;/:") // optional, defaults to the set "\"&#64;/" and is also used for eventually created rotations
 *                 .secretName("/myapp/mydocdb/masteruser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.R5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.
 * <p>
 * Your cluster will be empty by default.
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster, use the <code>.connections</code> attribute. DocumentDB databases have a default port, so
 * you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.connections.allowDefaultPortFromAnyIpv4("Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoints to access your database cluster will be available as the <code>.clusterEndpoint</code> and <code>.clusterReadEndpoint</code>
 * attributes:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * String writeAddress = cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * If you have existing security groups you would like to add to the cluster, use the <code>addSecurityGroups</code> method. Security
 * groups added in this way will not be managed by the <code>Connections</code> object of the cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * DatabaseCluster cluster;
 * 
 * 
 * SecurityGroup securityGroup = SecurityGroup.Builder.create(this, "SecurityGroup")
 *         .vpc(vpc)
 *         .build();
 * cluster.addSecurityGroups(securityGroup);
 * </pre></blockquote>
 * <p>
 * <h2>Deletion protection</h2>
 * <p>
 * Deletion protection can be enabled on an Amazon DocumentDB cluster to prevent accidental deletion of the cluster:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.R5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .deletionProtection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.addRotationSingleUser();
 * </pre></blockquote>
 * <p>
 * <a href="test/integ.cluster-rotation.lit.ts">example of setting up master password rotation for a cluster</a>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.*;
 * 
 * Secret myImportedSecret;
 * DatabaseCluster cluster;
 * 
 * 
 * cluster.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder()
 *         .secret(myImportedSecret)
 *         .build());
 * </pre></blockquote>
 * <p>
 * It's also possible to create user credentials together with the cluster and add rotation:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * DatabaseSecret myUserSecret = DatabaseSecret.Builder.create(this, "MyUserSecret")
 *         .username("myuser")
 *         .masterSecret(cluster.getSecret())
 *         .build();
 * ISecret myUserSecretAttached = myUserSecret.attach(cluster); // Adds DB connections information in the secret
 * 
 * cluster.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder() // Add rotation using the multi user scheme
 *         .secret(myUserSecretAttached).build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: This user must be created manually in the database using the master credentials.
 * The rotation will start as soon as this user exists.
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-secretsmanager/README.md">&#64;aws-cdk/aws-secretsmanager</a> for credentials rotation of existing clusters.
 * <p>
 * <h2>Audit and profiler Logs</h2>
 * <p>
 * Sending audit or profiler needs to be configured in two places:
 * <p>
 * <ol>
 * <li>Check / create the needed options in your ParameterGroup for <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html#event-auditing-enabling-auditing">audit</a> and
 * <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/profiling.html#profiling.enable-profiling">profiler</a> logs.</li>
 * <li>Enable the corresponding option(s) when creating the <code>DatabaseCluster</code>:</li>
 * </ol>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.logs.*;
 * 
 * Role myLogsPublishingRole;
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.R5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .exportProfilerLogsToCloudWatch(true) // Enable sending profiler logs
 *         .exportAuditLogsToCloudWatch(true) // Enable sending audit logs
 *         .cloudWatchLogsRetention(RetentionDays.THREE_MONTHS) // Optional - default is to never expire logs
 *         .cloudWatchLogsRetentionRole(myLogsPublishingRole)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.docdb;
