/**
 * <h1>Include CloudFormation templates in the CDK</h1>
 * <p>
 * This module contains a set of classes whose goal is to facilitate working
 * with existing CloudFormation templates in the CDK.
 * It can be thought of as an extension of the capabilities of the
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_core.CfnInclude.html"><code>CfnInclude</code> class</a>.
 * <p>
 * <h2>Basic usage</h2>
 * <p>
 * Assume we have a file with an existing template.
 * It could be in JSON format, in a file <code>my-template.json</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "Bucket": {
 *       "Type": "AWS::S3::Bucket",
 *       "Properties": {
 *         "BucketName": "some-bucket-name"
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Or it could by in YAML format, in a file <code>my-template.yaml</code>:
 * <p>
 * <blockquote><pre>
 * Resources:
 *   Bucket:
 *     Type: AWS::S3::Bucket
 *     Properties:
 *       BucketName: some-bucket-name
 * </pre></blockquote>
 * <p>
 * It can be included in a CDK application with the following code:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate = CfnInclude.Builder.create(this, "Template")
 *         .templateFile("my-template.json")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or, if your template uses YAML:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate = CfnInclude.Builder.create(this, "Template")
 *         .templateFile("my-template.yaml")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: different YAML parsers sometimes don't agree on what exactly constitutes valid YAML.
 * If you get a YAML exception when including your template,
 * try converting it to JSON, and including that file instead.
 * If you're downloading your template from the CloudFormation AWS Console,
 * you can easily get it in JSON format by clicking the 'View in Designer'
 * button on the 'Template' tab -
 * once in Designer, select JSON in the "Choose template language"
 * radio buttons on the bottom pane.
 * <p>
 * This will add all resources from <code>my-template.json</code> / <code>my-template.yaml</code> into the CDK application,
 * preserving their original logical IDs from the template file.
 * <p>
 * Note that this including process will <em>not</em> execute any
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-macros.html">CloudFormation transforms</a> -
 * including the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-aws-serverless.html">Serverless transform</a>.
 * <p>
 * Any resource from the included template can be retrieved by referring to it by its logical ID from the template.
 * If you know the class of the CDK object that corresponds to that resource,
 * you can cast the returned object to the correct type:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnBucket cfnBucket = (CfnBucket)cfnTemplate.getResource("Bucket");
 * </pre></blockquote>
 * <p>
 * Note that any resources not present in the latest version of the CloudFormation schema
 * at the time of publishing the version of this module that you depend on,
 * including <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">Custom Resources</a>,
 * will be returned as instances of the class <code>CfnResource</code>,
 * and so cannot be cast to a different resource type.
 * <p>
 * Any modifications made to that resource will be reflected in the resulting CDK template;
 * for example, the name of the bucket can be changed:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnBucket cfnBucket = (CfnBucket)cfnTemplate.getResource("Bucket");
 * cfnBucket.getBucketName() = "my-bucket-name";
 * </pre></blockquote>
 * <p>
 * You can also refer to the resource when defining other constructs,
 * including the higher-level ones
 * (those whose name does not start with <code>Cfn</code>),
 * for example:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnBucket cfnBucket = (CfnBucket)cfnTemplate.getResource("Bucket");
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new AnyPrincipal())
 *         .build();
 * role.addToPolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("s3:*"))
 *         .resources(List.of(cfnBucket.getAttrArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Converting L1 resources to L2</h3>
 * <p>
 * The resources the <code>getResource</code> method returns are what the CDK calls
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_cfn">Layer 1 resources</a>
 * (like <code>CfnBucket</code>).
 * However, in many places in the Construct Library,
 * the CDK requires so-called Layer 2 resources, like <code>IBucket</code>.
 * There are two ways of going from an L1 to an L2 resource.
 * <p>
 * <h4>Using<code>fromCfn*()</code> methods</h4>
 * <p>
 * This is the preferred method of converting an L1 resource to an L2.
 * It works by invoking a static method of the class of the L2 resource
 * whose name starts with <code>fromCfn</code> -
 * for example, for KMS Keys, that would be the <code>Kms.fromCfnKey()</code> method -
 * and passing the L1 instance as an argument:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnKey cfnKey = (CfnKey)cfnTemplate.getResource("Key");
 * IKey key = Key.fromCfnKey(cfnKey);
 * </pre></blockquote>
 * <p>
 * This returns an instance of the <code>kms.IKey</code> type that can be passed anywhere in the CDK an <code>IKey</code> is expected.
 * What is more, that <code>IKey</code> instance will be mutable -
 * which means calling any mutating methods on it,
 * like <code>addToResourcePolicy()</code>,
 * will be reflected in the resulting template.
 * <p>
 * Note that, in some cases, the <code>fromCfn*()</code> method might not be able to create an L2 from the underlying L1.
 * This can happen when the underlying L1 heavily uses CloudFormation functions.
 * For example, if you tried to create an L2 <code>IKey</code>
 * from an L1 represented as this CloudFormation template:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "Key": {
 *       "Type": "AWS::KMS::Key",
 *       "Properties": {
 *         "KeyPolicy": {
 *           "Statement": [
 *             {
 *               "Fn::If": [
 *                 "Condition",
 *                 {
 *                   "Action": "kms:if-action",
 *                   "Resource": "*",
 *                   "Principal": "*",
 *                   "Effect": "Allow"
 *                 },
 *                 {
 *                   "Action": "kms:else-action",
 *                   "Resource": "*",
 *                   "Principal": "*",
 *                   "Effect": "Allow"
 *                 }
 *               ]
 *             }
 *           ],
 *           "Version": "2012-10-17"
 *         }
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * The <code>Key.fromCfnKey()</code> method does not know how to translate that into CDK L2 concepts,
 * and would throw an exception.
 * <p>
 * In those cases, you need the use the second method of converting an L1 to an L2.
 * <p>
 * <h4>Using <code>from*Name/Arn/Attributes()</code> methods</h4>
 * <p>
 * If the resource you need does not have a <code>fromCfn*()</code> method,
 * or if it does, but it throws an exception for your particular L1,
 * you need to use the second method of converting an L1 resource to L2.
 * <p>
 * Each L2 class has static factory methods with names like <code>from*Name()</code>,
 * <code>from*Arn()</code>, and/or <code>from*Attributes()</code>.
 * You can obtain an L2 resource from an L1 by passing the correct properties of the L1 as the arguments to those methods:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * // using from*Attributes()
 * CfnSubnet privateCfnSubnet1;
 * CfnSubnet privateCfnSubnet2;
 * 
 * 
 * // using from*Name()
 * CfnBucket cfnBucket = (CfnBucket)cfnTemplate.getResource("Bucket");
 * IBucket bucket = Bucket.fromBucketName(this, "L2Bucket", cfnBucket.getRef());
 * 
 * // using from*Arn()
 * CfnKey cfnKey = (CfnKey)cfnTemplate.getResource("Key");
 * IKey key = Key.fromKeyArn(this, "L2Key", cfnKey.getAttrArn());
 * CfnVPC cfnVpc = (CfnVPC)cfnTemplate.getResource("Vpc");
 * IVpc vpc = Vpc.fromVpcAttributes(this, "L2Vpc", VpcAttributes.builder()
 *         .vpcId(cfnVpc.getRef())
 *         .availabilityZones(Fn.getAzs())
 *         .privateSubnetIds(List.of(privateCfnSubnet1.getRef(), privateCfnSubnet2.getRef()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * As long as they just need to be referenced,
 * and not changed in any way, everything should work;
 * however, note that resources returned from those methods,
 * unlike those returned by <code>fromCfn*()</code> methods,
 * are immutable, which means calling any mutating methods on them will have no effect.
 * You will have to mutate the underlying L1 in order to change them.
 * <p>
 * <h2>Non-resource template elements</h2>
 * <p>
 * In addition to resources,
 * you can also retrieve and mutate all other template elements:
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnParameter param = cfnTemplate.getParameter("MyParameter");
 * 
 * // mutating the parameter
 * param.getDefault() = "MyDefault";
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html">Conditions</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnCondition condition = cfnTemplate.getCondition("MyCondition");
 * 
 * // mutating the condition
 * condition.getExpression() = Fn.conditionEquals(1, 2);
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">Mappings</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * CfnMapping mapping = cfnTemplate.getMapping("MyMapping");
 * 
 * // mutating the mapping
 * mapping.setValue("my-region", "AMI", "ami-04681a1dbd79675a5");
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html">Service Catalog template Rules</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * // mutating the rule
 * CfnParameter myParameter;
 * 
 * CfnRule rule = cfnTemplate.getRule("MyRule");
 * rule.addAssertion(Fn.conditionContains(List.of("m1.small"), myParameter.getValueAsString()), "MyParameter has to be m1.small");
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">Outputs</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * // mutating the output
 * CfnBucket cfnBucket;
 * 
 * CfnOutput output = cfnTemplate.getOutput("MyOutput");
 * output.getValue() = cfnBucket.getAttrArn();
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html">Hooks for blue-green deployments</a>:
 * <p>
 * <blockquote><pre>
 * CfnInclude cfnTemplate;
 * 
 * // mutating the hook
 * Role myRole;
 * 
 * CfnHook hook = cfnTemplate.getHook("MyOutput");
 * CfnCodeDeployBlueGreenHook codeDeployHook = (CfnCodeDeployBlueGreenHook)hook;
 * codeDeployHook.getServiceRole() = myRole.getRoleArn();
 * </pre></blockquote></li>
 * </ul>
 * <p>
 * <h2>Parameter replacement</h2>
 * <p>
 * If your existing template uses CloudFormation Parameters,
 * you may want to remove them in favor of build-time values.
 * You can do that using the <code>parameters</code> property:
 * <p>
 * <blockquote><pre>
 * CfnInclude.Builder.create(this, "includeTemplate")
 *         .templateFile("path/to/my/template")
 *         .parameters(Map.of(
 *                 "MyParam", "my-value"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will replace all references to <code>MyParam</code> with the string <code>'my-value'</code>,
 * and <code>MyParam</code> will be removed from the 'Parameters' section of the resulting template.
 * <p>
 * <h2>Nested Stacks</h2>
 * <p>
 * This module also supports templates that use <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested stacks</a>.
 * <p>
 * For example, if you have the following parent template:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "ChildStack": {
 *       "Type": "AWS::CloudFormation::Stack",
 *       "Properties": {
 *         "TemplateURL": "https://my-s3-template-source.s3.amazonaws.com/child-stack.json"
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * where the child template pointed to by <code>https://my-s3-template-source.s3.amazonaws.com/child-stack.json</code> is:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "MyBucket": {
 *       "Type": "AWS::S3::Bucket"
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * You can include both the parent stack,
 * and the nested stack in your CDK application as follows:
 * <p>
 * <blockquote><pre>
 * CfnInclude parentTemplate = CfnInclude.Builder.create(this, "ParentStack")
 *         .templateFile("path/to/my-parent-template.json")
 *         .loadNestedStacks(Map.of(
 *                 "ChildStack", CfnIncludeProps.builder()
 *                         .templateFile("path/to/my-nested-template.json")
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Here, <code>path/to/my-nested-template.json</code>
 * represents the path on disk to the downloaded template file from the original template URL of the nested stack
 * (<code>https://my-s3-template-source.s3.amazonaws.com/child-stack.json</code>).
 * In the CDK application,
 * this file will be turned into an <a href="https://docs.aws.amazon.com/cdk/latest/guide/assets.html">Asset</a>,
 * and the <code>TemplateURL</code> property of the nested stack resource
 * will be modified to point to that asset.
 * <p>
 * The included nested stack can be accessed with the <code>getNestedStack</code> method:
 * <p>
 * <blockquote><pre>
 * CfnInclude parentTemplate;
 * 
 * 
 * IncludedNestedStack includedChildStack = parentTemplate.getNestedStack("ChildStack");
 * NestedStack childStack = includedChildStack.getStack();
 * CfnInclude childTemplate = includedChildStack.getIncludedTemplate();
 * </pre></blockquote>
 * <p>
 * Now you can reference resources from <code>ChildStack</code>,
 * and modify them like any other included template:
 * <p>
 * <blockquote><pre>
 * CfnInclude childTemplate;
 * 
 * 
 * CfnBucket cfnBucket = (CfnBucket)childTemplate.getResource("MyBucket");
 * cfnBucket.getBucketName() = "my-new-bucket-name";
 * 
 * Role role = Role.Builder.create(this, "MyRole")
 *         .assumedBy(new AccountRootPrincipal())
 *         .build();
 * 
 * role.addToPolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("s3:GetObject*", "s3:GetBucket*", "s3:List*"))
 *         .resources(List.of(cfnBucket.getAttrArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also include the nested stack after the <code>CfnInclude</code> object was created,
 * instead of doing it on construction:
 * <p>
 * <blockquote><pre>
 * CfnInclude parentTemplate;
 * 
 * IncludedNestedStack includedChildStack = parentTemplate.loadNestedStack("ChildTemplate", CfnIncludeProps.builder()
 *         .templateFile("path/to/my-nested-template.json")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Vending CloudFormation templates as Constructs</h2>
 * <p>
 * In many cases, there are existing CloudFormation templates that are not entire applications,
 * but more like specialized fragments, implementing a particular pattern or best practice.
 * If you have templates like that,
 * you can use the <code>CfnInclude</code> class to vend them as CDK Constructs:
 * <p>
 * <blockquote><pre>
 * import constructs.Construct;
 * import software.amazon.awscdk.cloudformation.include.*;
 * import path.*;
 * 
 * public class MyConstruct extends Construct {
 *     public MyConstruct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         // include a template inside the Construct
 *         // include a template inside the Construct
 *         CfnInclude.Builder.create(this, "MyConstruct")
 *                 .templateFile(join(__dirname, "my-template.json"))
 *                 .preserveLogicalIds(false)
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Notice the <code>preserveLogicalIds</code> parameter -
 * it makes sure the logical IDs of all the included template elements are re-named using CDK's algorithm,
 * guaranteeing they are unique within your application.
 * Without that parameter passed,
 * instantiating <code>MyConstruct</code> twice in the same Stack would result in duplicated logical IDs.
 */
package software.amazon.awscdk.cloudformation.include;
