/*
 * Decompiled with CFR 0.152.
 */
package ro.pippo.core.route;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Path;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ro.pippo.core.Application;
import ro.pippo.core.PippoRuntimeException;
import ro.pippo.core.route.RouteContext;
import ro.pippo.core.route.RouteHandler;
import ro.pippo.core.util.StringUtils;

public class DirectoryHandler
implements RouteHandler {
    private static final Logger log = LoggerFactory.getLogger(DirectoryHandler.class);
    public static final String PATH_PARAMETER = "path";
    private final String urlPath;
    private final String uriPattern;
    private final File directory;
    private String timestampPattern = "yyyy-MM-dd HH:mm Z";
    private String fileSizePattern = "#,000";
    private String directoryTemplate;
    private boolean chunked;
    private Comparator<DirEntry> comparator;

    public DirectoryHandler(String urlPath, File directory) {
        this.urlPath = urlPath;
        String normalizedPath = this.getNormalizedPath(urlPath);
        this.uriPattern = normalizedPath.length() > 0 ? String.format("/%s/?{%s: .*}", this.getNormalizedPath(urlPath), PATH_PARAMETER) : String.format("/{%s: .*}", PATH_PARAMETER);
        this.directory = directory.getAbsoluteFile();
        this.comparator = new DirectoryBeforeComparator();
    }

    public DirectoryHandler(String urlPath, String directory) {
        this(urlPath, new File(directory));
    }

    public String getUrlPath() {
        return this.urlPath;
    }

    public String getUriPattern() {
        return this.uriPattern;
    }

    public String getTimestampPattern() {
        return this.timestampPattern;
    }

    public DirectoryHandler setTimestampPattern(String pattern) {
        this.timestampPattern = pattern;
        return this;
    }

    public String getFileSizePattern() {
        return this.fileSizePattern;
    }

    public DirectoryHandler setFileSizePattern(String pattern) {
        this.fileSizePattern = pattern;
        return this;
    }

    public String getDirectoryTemplate() {
        return this.directoryTemplate;
    }

    public DirectoryHandler setDirectoryTemplate(String template) {
        this.directoryTemplate = template;
        return this;
    }

    public boolean getChunkedTransfer() {
        return this.chunked;
    }

    public DirectoryHandler setChunkedTransfer(boolean chunked) {
        this.chunked = chunked;
        return this;
    }

    public Comparator<DirEntry> getComparator() {
        return this.comparator;
    }

    public DirectoryHandler setComparator(Comparator<DirEntry> comparator) {
        this.comparator = comparator;
        return this;
    }

    public final void handle(RouteContext routeContext) {
        String resourcePath = this.getResourcePath(routeContext);
        log.trace("Request resource '{}'", (Object)resourcePath);
        this.handle(routeContext, resourcePath);
        routeContext.next();
    }

    protected void handle(RouteContext routeContext, String resourcePath) {
        try {
            Path requestedPath = new File(this.directory, resourcePath).toPath().normalize().toAbsolutePath();
            if (!requestedPath.startsWith(this.directory.getAbsolutePath())) {
                log.warn("Request for '{}' which is not located in '{}'", (Object)requestedPath, (Object)this.directory);
            } else if (StringUtils.isNullOrEmpty(resourcePath) || "/".equals(resourcePath)) {
                this.handleDirectoryRequest(routeContext, this.directory);
            } else {
                File file = requestedPath.toFile();
                if (file.exists()) {
                    if (file.isFile()) {
                        routeContext.getResponse().contentLength(file.length());
                        URL url = requestedPath.toUri().toURL();
                        switch (routeContext.getRequestMethod()) {
                            case "HEAD": {
                                this.setResponseHeaders(url, routeContext);
                                routeContext.getResponse().commit();
                                break;
                            }
                            case "GET": {
                                this.streamResource(url, routeContext);
                                break;
                            }
                            default: {
                                log.warn("Unsupported request method {} for {}", (Object)routeContext.getRequestMethod(), (Object)routeContext.getRequestUri());
                                break;
                            }
                        }
                    } else {
                        this.handleDirectoryRequest(routeContext, file);
                    }
                } else {
                    log.warn("{} not found for request path {}", (Object)requestedPath, (Object)routeContext.getRequestUri());
                }
            }
        }
        catch (MalformedURLException e) {
            log.error(e.getMessage(), (Throwable)e);
        }
    }

    protected File getIndexFile(File dir) {
        String[] welcomeFiles;
        for (String welcomeFile : welcomeFiles = new String[]{"index.html", "index.htm"}) {
            File file = new File(dir, welcomeFile);
            if (!file.exists()) continue;
            return file;
        }
        return null;
    }

    protected void handleDirectoryRequest(RouteContext routeContext, File dir) throws MalformedURLException {
        File index = this.getIndexFile(dir);
        if (index != null) {
            routeContext.getResponse().contentLength(index.length());
            URL url = index.toURI().toURL();
            this.streamResource(url, routeContext);
            return;
        }
        this.sendDirectoryListing(routeContext, dir);
    }

    protected File[] getFiles(File directory) {
        return Optional.ofNullable(directory.listFiles()).orElse(new File[0]);
    }

    protected List<DirEntry> getDirEntries(RouteContext routeContext, File dir, String absoluteDirUri) {
        ArrayList<DirEntry> list = new ArrayList<DirEntry>();
        for (File file : this.getFiles(dir)) {
            String fileUrl = routeContext.getRequest().getApplicationPath() + StringUtils.removeEnd(StringUtils.addStart(absoluteDirUri, "/"), "/") + StringUtils.addStart(file.getName(), "/");
            list.add(new DirEntry(fileUrl, file));
        }
        if (this.comparator != null) {
            list.sort(this.comparator);
        }
        if (!this.directory.equals(dir)) {
            File upDir = new File(dir, "../");
            list.add(0, new DirEntry(routeContext.getRequest().getApplicationPath() + StringUtils.removeEnd(StringUtils.addStart(absoluteDirUri, "/"), "/") + StringUtils.addStart(upDir.getName(), "/"), upDir));
        }
        return list;
    }

    protected String getResourcePath(RouteContext routeContext) {
        return this.getNormalizedPath(routeContext.getParameter(PATH_PARAMETER).toString());
    }

    protected String getNormalizedPath(String path) {
        if (path.length() > 0 && '/' == path.charAt(0)) {
            path = path.substring(1);
        }
        if (path.length() > 0 && '/' == path.charAt(path.length() - 1)) {
            path = path.substring(0, path.length() - 1);
        }
        return path;
    }

    protected void setResponseHeaders(URL resourceUrl, RouteContext routeContext) {
        try {
            long lastModified = resourceUrl.openConnection().getLastModified();
            ((Application)routeContext.getApplication()).getHttpCacheToolkit().addEtag(routeContext, lastModified);
            String filename = resourceUrl.getFile();
            String mimeType = ((Application)routeContext.getApplication()).getMimeTypes().getContentType(filename);
            if (!StringUtils.isNullOrEmpty(mimeType)) {
                routeContext.getResponse().contentType(mimeType);
            }
        }
        catch (Exception e) {
            throw new PippoRuntimeException(e, "Failed to stream resource {}", resourceUrl);
        }
    }

    protected void streamResource(URL resourceUrl, RouteContext routeContext) {
        try {
            this.setResponseHeaders(resourceUrl, routeContext);
            if (routeContext.getResponse().getStatus() == 304) {
                routeContext.getResponse().commit();
            } else {
                this.sendResource(resourceUrl, routeContext);
            }
        }
        catch (IOException e) {
            String message = e.getMessage();
            if (!StringUtils.isNullOrEmpty(message)) {
                log.warn("Error sending resource {} to {}: {}", new Object[]{resourceUrl, routeContext.getRequest().getClientIp(), message});
            }
            throw new PippoRuntimeException(e, "Failed to stream resource {}", resourceUrl);
        }
        catch (Exception e) {
            throw new PippoRuntimeException(e, "Failed to stream resource {}", resourceUrl);
        }
    }

    protected void sendResource(URL resourceUrl, RouteContext routeContext) throws IOException {
        String filename = resourceUrl.getFile();
        String mimeType = ((Application)routeContext.getApplication()).getMimeTypes().getContentType(filename);
        if (!StringUtils.isNullOrEmpty(mimeType)) {
            log.debug("Streaming as resource '{}'", (Object)resourceUrl);
            routeContext.getResponse().ok().chunked(this.chunked).resource(resourceUrl.openStream());
        } else {
            log.debug("Streaming as file '{}'", (Object)resourceUrl);
            routeContext.getResponse().ok().chunked(this.chunked).file(filename, resourceUrl.openStream());
        }
    }

    private void sendDirectoryListing(RouteContext routeContext, File dir) {
        String absoluteDirUri = this.getUrlPath() + StringUtils.addStart(this.directory.toPath().relativize(dir.toPath()).toString(), "/");
        if (StringUtils.isNullOrEmpty(this.directoryTemplate)) {
            String page = this.generateDefaultDirectoryListing(routeContext, dir, absoluteDirUri);
            routeContext.html().send(page);
        } else {
            int numFiles = 0;
            int numDirs = 0;
            long diskUsage = 0L;
            List<DirEntry> dirEntries = this.getDirEntries(routeContext, dir, absoluteDirUri);
            for (DirEntry dirEntry : dirEntries) {
                if (dirEntry.isFile()) {
                    ++numFiles;
                    diskUsage += dirEntry.getSize();
                    continue;
                }
                if (!dirEntry.isDirectory() || dirEntry.getName().contains("..")) continue;
                ++numDirs;
            }
            routeContext.setLocal("dirUrl", absoluteDirUri);
            routeContext.setLocal("dirPath", absoluteDirUri.substring(this.getUrlPath().length()));
            routeContext.setLocal("dirEntries", dirEntries);
            routeContext.setLocal("numDirs", numDirs);
            routeContext.setLocal("numFiles", numFiles);
            routeContext.setLocal("diskUsage", diskUsage);
            routeContext.render(this.directoryTemplate);
        }
    }

    protected String generateDefaultDirectoryListing(RouteContext routeContext, File dir, String absoluteDirUri) {
        StringBuilder sb = new StringBuilder();
        sb.append("<html><body><table>");
        SimpleDateFormat df = new SimpleDateFormat(this.timestampPattern);
        DecimalFormat nf = new DecimalFormat(this.fileSizePattern);
        for (DirEntry dirEntry : this.getDirEntries(routeContext, dir, absoluteDirUri)) {
            sb.append(StringUtils.format("<tr><td><a href=\"{}\">{}</a></td><td>{}</td><td>{}</td></tr>\n", dirEntry.getUrl(), dirEntry.getName(), dirEntry.isFile() ? nf.format(dirEntry.getSize()) : "", df.format(dirEntry.getLastModified())));
        }
        sb.append("</table></body></html>");
        return sb.toString();
    }

    public static class DirectoryBeforeComparator
    implements Comparator<DirEntry> {
        @Override
        public int compare(DirEntry o1, DirEntry o2) {
            if (o1.isDirectory() && !o2.isDirectory()) {
                return -1;
            }
            if (!o1.isDirectory() && o2.isDirectory()) {
                return 1;
            }
            return o1.file.compareTo(o2.file);
        }
    }

    public static class DirEntry {
        private final String url;
        private final File file;

        public DirEntry(String url, File file) {
            this.url = url;
            this.file = file;
        }

        public String getUrl() {
            return this.url;
        }

        public String getName() {
            return this.file.getName();
        }

        public long getLength() {
            return this.file.length();
        }

        public long getSize() {
            return this.file.length();
        }

        public Date getLastModified() {
            return new Date(this.file.lastModified());
        }

        public boolean isFile() {
            return this.file.isFile();
        }

        public boolean isDirectory() {
            return this.file.isDirectory();
        }
    }
}

