/*
 * Decompiled with CFR 0.152.
 */
package redis.clients.authentication.core;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.authentication.core.AuthXException;
import redis.clients.authentication.core.IdentityProvider;
import redis.clients.authentication.core.Token;
import redis.clients.authentication.core.TokenListener;
import redis.clients.authentication.core.TokenManagerConfig;
import redis.clients.authentication.core.TokenRequestException;

public class TokenManager {
    private TokenManagerConfig tokenManagerConfig;
    private IdentityProvider identityProvider;
    private TokenListener listener;
    private ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
    private ExecutorService executor = Executors.newSingleThreadExecutor();
    private boolean stopped = false;
    private ScheduledFuture<?> scheduledTask;
    private int numberOfRetries = 0;
    private Exception lastException;
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private Token currentToken = null;
    private AtomicBoolean started = new AtomicBoolean(false);

    public TokenManager(IdentityProvider identityProvider, TokenManagerConfig tokenManagerConfig) {
        this.identityProvider = identityProvider;
        this.tokenManagerConfig = tokenManagerConfig;
    }

    public void start(TokenListener listener, boolean blockForInitialToken) {
        if (!this.started.compareAndSet(false, true)) {
            throw new AuthXException("Token manager already started!");
        }
        this.listener = listener;
        ScheduledFuture<?> currentTask = this.scheduleNext(0L);
        this.scheduledTask = currentTask;
        if (blockForInitialToken) {
            try {
                while (currentTask.get() == null) {
                    currentTask = this.scheduledTask;
                }
            }
            catch (RuntimeException e) {
                throw e;
            }
            catch (Exception e) {
                throw new TokenRequestException(this.unwrap(e), this.lastException);
            }
        }
    }

    public void stop() {
        this.stopped = true;
        this.scheduledTask.cancel(true);
        this.scheduler.shutdown();
        this.executor.shutdown();
    }

    public TokenManagerConfig getConfig() {
        return this.tokenManagerConfig;
    }

    private ScheduledFuture<?> scheduleNext(long delay) {
        return this.scheduler.schedule(() -> this.renewToken(), delay, TimeUnit.MILLISECONDS);
    }

    protected Token renewToken() {
        if (this.stopped) {
            return null;
        }
        Token newToken = null;
        try {
            Future<Token> requestResult = this.executor.submit(() -> this.requestToken());
            this.currentToken = newToken = requestResult.get(this.tokenManagerConfig.getTokenRequestExecTimeoutInMs(), TimeUnit.MILLISECONDS);
            long delay = this.calculateRenewalDelay(newToken.getExpiresAt(), newToken.getReceivedAt());
            this.scheduledTask = this.scheduleNext(delay);
            this.listener.onTokenRenewed(newToken);
            return newToken;
        }
        catch (Exception e) {
            if (this.numberOfRetries < this.tokenManagerConfig.getRetryPolicy().getMaxAttempts()) {
                ++this.numberOfRetries;
            } else {
                TokenRequestException tre = new TokenRequestException(this.unwrap(e), this.lastException);
                this.listener.onError(tre);
                throw tre;
            }
            this.scheduledTask = this.scheduleNext(this.tokenManagerConfig.getRetryPolicy().getdelayInMs());
            return null;
        }
    }

    protected Token requestToken() {
        this.lastException = null;
        try {
            return this.identityProvider.requestToken();
        }
        catch (Exception e) {
            this.lastException = e;
            this.logger.error("Request to identity provider failed with message: " + e.getMessage(), (Throwable)e);
            throw e;
        }
    }

    private Throwable unwrap(Exception e) {
        return e instanceof ExecutionException ? e.getCause() : e;
    }

    public Token getCurrentToken() {
        return this.currentToken;
    }

    public long calculateRenewalDelay(long expireDate, long issueDate) {
        long ttlRatioRefresh;
        long ttlLowerRefresh = this.ttlForLowerRefresh(expireDate);
        long delay = Math.min(ttlLowerRefresh, ttlRatioRefresh = this.ttlForRatioRefresh(expireDate, issueDate));
        return delay < 0L ? 0L : delay;
    }

    public long ttlForLowerRefresh(long expireDate) {
        return expireDate - (long)this.tokenManagerConfig.getLowerRefreshBoundMillis() - System.currentTimeMillis();
    }

    protected long ttlForRatioRefresh(long expireDate, long issueDate) {
        long validDuration = expireDate - issueDate;
        long refreshBefore = validDuration - (long)((float)validDuration * this.tokenManagerConfig.getExpirationRefreshRatio());
        return expireDate - refreshBefore - System.currentTimeMillis();
    }
}

