package pub.dsb.framework.boot.cloud.configuration.gateway;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.gateway.filter.ratelimit.KeyResolver;
import org.springframework.cloud.gateway.filter.ratelimit.RedisRateLimiter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.ReactiveRedisTemplate;
import org.springframework.data.redis.core.script.RedisScript;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.validation.Validator;
import pub.dsb.framework.boot.cloud.properties.GatewayLimitProperties;
import pub.dsb.framework.boot.redis.properties.RedisClusterProperties;
import reactor.core.publisher.Mono;

import java.util.List;

/**
 * <p>
 *
 * </p>
 *
 * @author Yiyuery
 * @date 2020/8/28
 * @since 0.0.1
 */
@Configuration
@ConditionalOnClass(KeyResolver.class)
@Import({RedisClusterProperties.class, GatewayLimitProperties.class})
public class GatewayLimiterConfiguration {
    /**
     * IP 限流 key 处理器
     *
     * @return
     */
    @Bean
    @ConditionalOnProperty(name = "dsb.cloud.gateway.limit.type", havingValue = "ipKeyResolver", matchIfMissing = true)
    public KeyResolver ipKeyResolver() {
        return (serverWebExchange) -> Mono.just(serverWebExchange.getRequest().getRemoteAddress().getHostString());
    }

    /**
     * Url 限流 key 处理器
     *
     * @return
     */
    @Bean
    @ConditionalOnProperty(name = "dsb.cloud.gateway.limit.type", havingValue = "pathKeyResolver")
    public KeyResolver pathKeyResolver() {
        return (serverWebExchange) -> Mono.just(serverWebExchange.getRequest().getPath().value());
    }

    /**
     * <p>
     * 复写默认限流使用的 RedisTemplate
     * </p>
     */
    @Bean
    @ConditionalOnMissingBean(name = "reactiveRedisTemplate")
    public ReactiveRedisTemplate<Object, Object> reactiveRedisTemplate(LettuceConnectionFactory reactiveRedisConnectionFactory) {
        JdkSerializationRedisSerializer jdkSerializer = new JdkSerializationRedisSerializer(
                GatewayLimiterConfiguration.class.getClassLoader());
        RedisSerializationContext<Object, Object> serializationContext = RedisSerializationContext.newSerializationContext().key(jdkSerializer).value(jdkSerializer).hashKey(jdkSerializer)
                .hashValue(jdkSerializer).build();
        return new ReactiveRedisTemplate<>(reactiveRedisConnectionFactory, serializationContext);
    }

    @Bean
    public RedisRateLimiter redisRateLimiter(ReactiveRedisTemplate<String, String> redisTemplate,
                                             @Qualifier(RedisRateLimiter.REDIS_SCRIPT_NAME) RedisScript<List<Long>> redisScript,
                                             Validator validator) {
        return new RedisRateLimiter(redisTemplate, redisScript, validator);
    }

    /**
     * <p>
     * 复写默认限流使用的 RedisTemplate 的默认连接池
     * </p>
     */
    @Bean
    public LettuceConnectionFactory reactiveRedisConnectionFactory(RedisClusterProperties redisClusterProperties) {
//        ErrorCodeEnum.YAML_CONFIG_ERROR.assertTrue(CollectionUtils.isNotEmpty(redisClusterProperties.getSegments()), "Redis 网关限流拦截必须配置 segment 节点。");
//        return new RedisTemplateCreateSupport().lettuceConnectionFactory(redisClusterProperties, redisProxyProperties.getSegments().get(0));
        return null;
    }
}
