package pub.dsb.framework.boot.cloud.beans.gateway;

import org.springframework.cloud.gateway.filter.GatewayFilter;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.factory.AbstractGatewayFilterFactory;
import org.springframework.cloud.gateway.filter.factory.rewrite.CachedBodyOutputMessage;
import org.springframework.cloud.gateway.support.BodyInserterContext;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.codec.HttpMessageReader;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpRequestDecorator;
import org.springframework.web.reactive.function.BodyInserter;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.server.HandlerStrategies;
import org.springframework.web.reactive.function.server.ServerRequest;
import org.springframework.web.server.ServerWebExchange;
import pub.dsb.framework.boot.common.functions.ApplyValue;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.List;
import java.util.Optional;

/**
 * 解密网关配置
 */
public class DecryptGatewayFilterFactory extends
        AbstractGatewayFilterFactory<DecryptGatewayFilterFactory.Config> {

    /**
     * rewriteFunction key.
     */
    public static final String KEY_REWRITE_FUNCTION = "rewriteFunction";

    private final ApplyValue<Mono<String>, String> defaultRewriteFunction;

    private final List<HttpMessageReader<?>> messageReaders;

    public DecryptGatewayFilterFactory(ApplyValue<Mono<String>, String> defaultRewriteFunction) {
        super(Config.class);
        this.messageReaders = HandlerStrategies.withDefaults().messageReaders();
        this.defaultRewriteFunction = defaultRewriteFunction;
    }

    @Override
    public GatewayFilter apply(Config config) {
        return ((exchange, chain) -> {
            //获取请求头、请求方式、请求报文
            HttpHeaders headers = exchange.getRequest().getHeaders();
            boolean isJsonPost = MediaType.APPLICATION_JSON.isCompatibleWith(headers.getContentType()) && HttpMethod.POST.equals(exchange.getRequest().getMethod());
            if (!isJsonPost) {
                return chain.filter(exchange);
            }
            return processOnEncryptBody(exchange, chain, config);
        });
    }

    /**
     * 处理加密的 Body 请求数据
     *
     * @param exchange
     * @param chain
     * @param config
     * @return
     */
    private Mono<Void> processOnEncryptBody(ServerWebExchange exchange, GatewayFilterChain chain, Config config) {
        ServerRequest serverRequest = ServerRequest.create(exchange,
                messageReaders);
        Mono<String> modifiedBody = serverRequest.bodyToMono(String.class).flatMap(o -> getOrDefault(config.rewriteFunction).apply(o));
        BodyInserter bodyInserter = BodyInserters.fromPublisher(modifiedBody,
                String.class);
        HttpHeaders headers = new HttpHeaders();
        headers.addAll(exchange.getRequest().getHeaders());
        headers.remove(HttpHeaders.CONTENT_LENGTH);
        CachedBodyOutputMessage outputMessage = new CachedBodyOutputMessage(exchange, headers);
        return bodyInserter.insert(outputMessage, new BodyInserterContext()).then(Mono.defer(() -> {
            ServerHttpRequest decorator = decorate(exchange, headers,
                    outputMessage);
            return chain
                    .filter(exchange.mutate().request(decorator).build());
        }));

    }

    /**
     * 获取默认处理器
     *
     * @param rewriteFunction
     * @return
     */
    private ApplyValue<Mono<String>, String> getOrDefault(ApplyValue<Mono<String>, String> rewriteFunction) {
        return Optional.ofNullable(rewriteFunction).orElse(defaultRewriteFunction);
    }

    /**
     * 数据加工
     *
     * @param exchange
     * @param headers
     * @param outputMessage
     * @return
     */
    ServerHttpRequestDecorator decorate(ServerWebExchange exchange, HttpHeaders headers,
                                        CachedBodyOutputMessage outputMessage) {
        return new ServerHttpRequestDecorator(exchange.getRequest()) {
            @Override
            public HttpHeaders getHeaders() {
                long contentLength = headers.getContentLength();
                HttpHeaders httpHeaders = new HttpHeaders();
                httpHeaders.addAll(super.getHeaders());
                if (contentLength > 0) {
                    httpHeaders.setContentLength(contentLength);
                } else {
                    httpHeaders.set(HttpHeaders.TRANSFER_ENCODING, "chunked");
                }
                return httpHeaders;
            }

            @Override
            public Flux<DataBuffer> getBody() {
                return outputMessage.getBody();
            }
        };
    }

    /**
     * 动态配置类
     */
    public static class Config {

        private ApplyValue<Mono<String>, String> rewriteFunction;

        public Config() {
        }

        public ApplyValue<Mono<String>, String> getRewriteFunction() {
            return rewriteFunction;
        }

        public void setRewriteFunction(ApplyValue<Mono<String>, String> rewriteFunction) {
            this.rewriteFunction = rewriteFunction;
        }
    }

}