package pro.siper.adept.core

import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.RecyclerView
import pro.siper.adept.core.diff.DefaultDiffUtilCallback
import pro.siper.adept.core.diff.DiffUtilCallback
import pro.siper.adept.core.diff.DiffUtilCallbackCreator
import pro.siper.adept.core.endless.EndlessScrollingCallback
import pro.siper.adept.core.endless.EndlessScrollingListener
import java.lang.IllegalArgumentException
import kotlin.reflect.KClass

class Adept : RecyclerView.Adapter<AdeptViewHolder>() {
    val renderers: MutableMap<KClass<*>, ItemViewRenderer<*>> = mutableMapOf()
    val layouts: MutableMap<KClass<*>, Int> = mutableMapOf()

    val dataset: MutableList<Any?> = mutableListOf()

    private val LOADING_VIEW = -11

    private val viewTypes: MutableList<KClass<out Any>> = mutableListOf()

    private var diffUtilCallback: DiffUtilCallback? = null
    private val useDiffUtil
        get() = diffUtilCallback != null

    private var endlessScrollingListener: EndlessScrollingListener? = null
    private val useEndlessScrolling
        get() = endlessScrollingListener != null

    private var loadingLayoutResId = -1
    private val useLoadingView
        get() = loadingLayoutResId != -1

    override fun getItemViewType(position: Int): Int {
        if (position == itemCount - 1 && useLoadingView && endlessScrollingListener!!.hasNextPage) {
            return LOADING_VIEW
        }
        val type = dataset[position]!!::class
        if (viewTypes.indexOf(type) == -1) {
            viewTypes.add(type)
        }
        return viewTypes.indexOf(type)
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): AdeptViewHolder {
        if (viewType == LOADING_VIEW) {
            val view = LayoutInflater
                    .from(parent.context)
                    .inflate(loadingLayoutResId, parent, false)
            return AdeptViewHolder(view, viewType)
        }

        val layout = layouts[viewTypes[viewType]]
                ?: throw IllegalArgumentException("Layout for ${viewTypes[viewType]} is not registered")
        val view = LayoutInflater
                .from(parent.context)
                .inflate(layout, parent, false)
        return AdeptViewHolder(view, viewType)
    }

    override fun getItemCount(): Int{
        return if (useLoadingView && endlessScrollingListener!!.hasNextPage) {
            dataset.size + 1
        } else {
            dataset.size
        }
    }

    override fun onBindViewHolder(holder: AdeptViewHolder, position: Int) {
        if (useLoadingView && holder.viewType == LOADING_VIEW) return

        val renderer = renderers[viewTypes[holder.viewType]]
                ?: throw IllegalArgumentException("There are no renderer for ${viewTypes[holder.viewType]}")
        renderer.anyTypeRender(dataset[position], ViewBinder(holder))
    }

    override fun onAttachedToRecyclerView(recyclerView: RecyclerView) {
        super.onAttachedToRecyclerView(recyclerView)
        if (useEndlessScrolling) {
            recyclerView.addOnScrollListener(endlessScrollingListener!!)
        }
    }

    override fun onDetachedFromRecyclerView(recyclerView: RecyclerView) {
        super.onDetachedFromRecyclerView(recyclerView)
        if (useEndlessScrolling) {
            recyclerView.removeOnScrollListener(endlessScrollingListener!!)
        }
    }

    fun updateDataset(dataset: List<Any?>) {
        if (useDiffUtil) {
            val diff = DiffUtilCallbackCreator(
                    this.dataset,
                    dataset,
                    diffUtilCallback!!
            )
            val diffResult = DiffUtil.calculateDiff(diff)
            this.dataset.clear()
            this.dataset.addAll(dataset)
            diffResult.dispatchUpdatesTo(this)
        } else {
            this.dataset.clear()
            this.dataset.addAll(dataset)
        }
    }

    inline fun <reified T> addRenderer(layoutId: Int, renderer: ItemViewRenderer<T>): Adept {
        if (layouts.containsKey(T::class)) {
            throw IllegalArgumentException("Renderer for ${T::class} already registered")
        }
        layouts[T::class] = layoutId
        renderers[T::class] = renderer
        return this
    }

    inline fun <reified T> addRenderer(
            layoutId: Int,
            crossinline renderer: (data: T, viewBinder: ViewBinder) -> Unit): Adept {
        addRenderer(layoutId, object : ItemViewRenderer<T>() {
            override fun render(data: T, viewBinder: ViewBinder) {
                renderer.invoke(data, viewBinder)
            }
        })
        return this
    }

    fun attachTo(recyclerView: RecyclerView): Adept {
        recyclerView.adapter = this
        return this
    }

    fun useDiffUtil(diffUtilCallback: DiffUtilCallback = DefaultDiffUtilCallback): Adept {
        this.diffUtilCallback = diffUtilCallback
        return this
    }

    fun useEndlessScrolling(
            loadingLayoutResId: Int = -1,
            endlessScrollingCallback: EndlessScrollingCallback): Adept {
        this.loadingLayoutResId = loadingLayoutResId
        this.endlessScrollingListener = EndlessScrollingListener(endlessScrollingCallback)
        return this
    }

    fun useEndlessScrolling(
            loadingLayoutResId: Int = -1,
            endlessScrollingCallback: (currentPage: Int, totalItemsCount: Int) -> Boolean): Adept {
        useEndlessScrolling(loadingLayoutResId, object : EndlessScrollingCallback {
            override fun onLoadMore(currentPage: Int, totalItemsCount: Int): Boolean {
                return endlessScrollingCallback.invoke(currentPage, totalItemsCount)
            }
        })
        return this
    }
}